/**
 ********************************************************************************
 * Copyright (c) 2019 Robert Bosch GmbH and others.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Robert Bosch GmbH - initial API and implementation
 ********************************************************************************
 */
package org.eclipse.app4mc.amalthea.converters.log4j.configuration;

import java.io.File;
import java.text.SimpleDateFormat;
import java.util.Date;

import org.apache.log4j.FileAppender;

/**
 * Custom Log4j appender that is writing log statements to a dedicated log file
 * if the system property <i>OUTPUT_DIRECTORY</i> is set. With this mechanism it
 * is possible to write a log file per migration session to a dedicated log file
 * only.
 */
public class MigrationSessionFileAppender extends FileAppender {

	private String dateFormat;
	private String currentOutputDirectory;

	@Override
	protected boolean checkEntryConditions() {
		// we only append if an output directory is set
		String outputDirectory = System.getProperty("APP4MC_MIGRATION_OUTPUT_DIRECTORY");
		if (outputDirectory != null) {
			if (this.currentOutputDirectory == null || !this.currentOutputDirectory.equals(outputDirectory)) {
				// close an open file as the output directory changed
				closeFile();
				this.currentOutputDirectory = outputDirectory;
				// create and set a new session log file
				createNewFile();
			}

			return super.checkEntryConditions();
		} else if (currentOutputDirectory != null) {
			// system property is null but local output directory is set
			// so we assume the session logging was disabled
			closeFile();
			this.currentOutputDirectory = null;
		}
		return false;
	}

	private void createNewFile() {
		String dateToStr = new SimpleDateFormat(this.dateFormat).format(new Date());

		super.setFile(new File(this.currentOutputDirectory, "ModelMigration__" + dateToStr + ".log").getAbsolutePath());
		super.activateOptions();
	}

	@Override
	public void activateOptions() {
		// simply avoid the log warning for missing fileName configuration
		if (fileName != null) {
			super.activateOptions();
		}
	}

	public void setDateFormat(String dateFormat) {
		this.dateFormat = dateFormat;
	}
}
