/**
 ********************************************************************************
 * Copyright (c) 2020 Eclipse APP4MC contributors.
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 ********************************************************************************
 */

package org.eclipse.app4mc.amalthea._import.atdb;

import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.app4mc.amalthea.model.ActivityGraph;
import org.eclipse.app4mc.amalthea.model.Amalthea;
import org.eclipse.app4mc.amalthea.model.AmaltheaFactory;
import org.eclipse.app4mc.amalthea.model.AmaltheaPackage;
import org.eclipse.app4mc.amalthea.model.AmaltheaServices;
import org.eclipse.app4mc.amalthea.model.Frequency;
import org.eclipse.app4mc.amalthea.model.FrequencyUnit;
import org.eclipse.app4mc.amalthea.model.IDiscreteValueDeviation;
import org.eclipse.app4mc.amalthea.model.Label;
import org.eclipse.app4mc.amalthea.model.LabelAccess;
import org.eclipse.app4mc.amalthea.model.LabelAccessEnum;
import org.eclipse.app4mc.amalthea.model.Runnable;
import org.eclipse.app4mc.amalthea.model.SWModel;
import org.eclipse.app4mc.amalthea.model.Ticks;
import org.eclipse.app4mc.amalthea.model.TimeUnit;
import org.eclipse.app4mc.amalthea.model.util.FactoryUtil;
import org.eclipse.app4mc.amalthea.model.util.ModelUtil;
import org.eclipse.app4mc.amalthea.model.util.RuntimeUtil;
import org.eclipse.app4mc.atdb.ATDBConnection;
import org.eclipse.app4mc.atdb.MetricAggregation;
import org.eclipse.emf.ecore.util.EcoreUtil;

public class RunnableConverter extends AConverter {
	
	private static final String rt = "runningTime_";
	
	private final boolean extractRuntimes;
	private final boolean extractLabelAccesses;
	private final Frequency frequency;

	public RunnableConverter(final Amalthea model, final ATDBConnection con,
			final boolean extractRuntimes, final boolean extractLabelAccesses, final double frequencyInHz) {
		super(model, con, "runnables");
		this.extractRuntimes = extractRuntimes;
		this.extractLabelAccesses = extractLabelAccesses;
		this.frequency = FactoryUtil.createFrequency(frequencyInHz, FrequencyUnit.HZ);
	}

	@Override
	protected void execute() throws SQLException {
		final SWModel swModel = ModelUtil.getOrCreateSwModel(this.model);
		final TimeUnit timeBase = TimeUnit.getByName(this.con.getTimeBase().toLowerCase());
				
		final List<Runnable> runnables = new ArrayList<>();
		this.con.getAllRunnables().forEach(runnableName -> {
			final Runnable runnable = AmaltheaModelUtil.getOrAddNew(swModel,
					AmaltheaPackage.eINSTANCE.getSWModel_Runnables(), runnableName, Runnable.class);
			runnables.add(runnable);
		});
		

		for(final Runnable runnable:runnables) {
			final ActivityGraph ag = ModelUtil.getOrCreateActivityGraph(runnable);
			// label reads
			if (this.extractLabelAccesses) {
				this.con.getLabelsReadByRunnable(runnable.getName()).forEach(labelName -> {
					final LabelAccess la = AmaltheaModelUtil.getOrAddNewWithContainer(ag,
							AmaltheaPackage.eINSTANCE.getIActivityGraphItemContainer_Items(), LabelAccess.class,
							AmaltheaPackage.eINSTANCE.getLabelAccess(),
							AmaltheaPackage.eINSTANCE.getLabelAccess_Data(), Label.class,
							AmaltheaPackage.eINSTANCE.getLabel(), labelName).getKey();
					la.setAccess(LabelAccessEnum.READ);
				});
			}
			// runtimes
			if (this.extractRuntimes && timeBase != null && this.frequency.getValue() >= 1) {
				double runningTimeAvg = -1;
				double runningTimeStDev = -1;
				try {
					runningTimeAvg = Double.parseDouble(this.con.getValueForMetricAndEntity(runnable.getName(), rt + MetricAggregation.Avg));
					runningTimeStDev = Double.parseDouble(this.con.getValueForMetricAndEntity(runnable.getName(), rt + MetricAggregation.StDev));
				} catch (NumberFormatException e) {
					// fail silently
				}
				try {
					final long runningTimeMin = Long.parseLong(this.con.getValueForMetricAndEntity(runnable.getName(), rt + MetricAggregation.Min));
					final long runningTimeMax = Long.parseLong(this.con.getValueForMetricAndEntity(runnable.getName(), rt + MetricAggregation.Max));
					final Ticks ticks = AmaltheaFactory.eINSTANCE.createTicks();
					IDiscreteValueDeviation dvd;
					if (runningTimeMin == runningTimeMax) {
						final long constTicks = (long)RuntimeUtil.getTicksForExecutionTimeInSeconds(
								AmaltheaServices.convertToSeconds(runningTimeMin, timeBase), this.frequency);
						dvd = FactoryUtil.createDiscreteValueConstant(constTicks);
					} else {
						final long ticksMin = (long)RuntimeUtil.getTicksForExecutionTimeInSeconds(
								AmaltheaServices.convertToSeconds(runningTimeMin, timeBase), this.frequency);
						final long ticksMax = (long)RuntimeUtil.getTicksForExecutionTimeInSeconds(
								AmaltheaServices.convertToSeconds(runningTimeMax, timeBase), this.frequency);
						if (runningTimeAvg >= 0 && runningTimeStDev >= 0) {
							final double ticksAvg = RuntimeUtil.getTicksForExecutionTimeInSeconds(
									AmaltheaServices.convertToSeconds(runningTimeAvg, timeBase), this.frequency);
							final double ticksStDev = RuntimeUtil.getTicksForExecutionTimeInSeconds(
									AmaltheaServices.convertToSeconds(runningTimeStDev, timeBase), this.frequency);
							dvd = FactoryUtil.createDiscreteValueGaussDistribution(ticksAvg, ticksStDev, ticksMin, ticksMax);
						} else {
							dvd = FactoryUtil.createDiscreteValueBoundaries(ticksMin, ticksMax);
						}
					}
					ticks.setDefault(dvd);
					RuntimeUtil.clearRuntimeOfRunnable(runnable, null);
					ag.getItems().add(ticks);
				} catch (NumberFormatException e) {
					// fail silently
				}
			}
			// label writes
			if (this.extractLabelAccesses) {
				this.con.getLabelsWrittenByRunnable(runnable.getName()).forEach(labelName -> {
					final LabelAccess la = AmaltheaModelUtil.getOrAddNewWithContainer(ag,
							AmaltheaPackage.eINSTANCE.getIActivityGraphItemContainer_Items(), LabelAccess.class,
							AmaltheaPackage.eINSTANCE.getLabelAccess(),
							AmaltheaPackage.eINSTANCE.getLabelAccess_Data(), Label.class,
							AmaltheaPackage.eINSTANCE.getLabel(), labelName).getKey();
					la.setAccess(LabelAccessEnum.WRITE);
				});
			}
			if (ag.getItems().isEmpty()) {
				EcoreUtil.delete(ag);
			}
		}
	}

}
