/*******************************************************************************
 * Copyright (c) 2004, 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.datatools.sqltools.parsers.sql.postparse;

import java.util.List;
import java.util.Map;

import org.eclipse.datatools.modelbase.sql.query.SQLQueryObject;
import org.eclipse.datatools.sqltools.parsers.sql.SQLParserException;

/**
 * @author ckadner
 *
 * A <code>PostParseProcessor</code> can be invoked by the
 * <code>SQLParserManager</code> as part of the semantical resolving and
 * validation phase after the syntactical phase, the actual parse itself.
 * 
 * After the successful parse the <code>SQLParserManager</code> will
 * bottom-up process the created a <code>SQLQueryObject</code> model
 * instance and for each contained <code>SQLQueryObject</code> will check for a
 * given <code>PostParseProcessor</code>, to be invoked by
 * {@link #process(SQLQueryObject)}, if the <code>SQLQueryObject</code> is a
 * subtype of one of the <code>PostParseProcessor</code>'s candidate types
 * ({@link #getProcessCandidateTypes()}).
 * 
 * Stateful <code>PostParseProcessor</code>s can collect information that are
 * valid within the context of one <code>QueryStatement</code>.
 * If a <code>PostParseProcessor</code> is stateful, its state can be reset
 * after each processed <code>QueryStatement</code> by implementing
 * {@link #resetState()} which will be called after the
 * <code>QueryStatement</code> processing is completed.
 * 
 * A <code>PostParseProcessor</code> can also do modifications on the
 * <code>SQLObject</code> stack of the parser, for example to invalidate
 * <code>SQLObject</code>s for further post parse processing or replacing a
 * <code>SQLObject</code> that was generated by the parser with a
 * <code>SQLObject</code> generated in exchange by the
 * <code>PostParseProcessor</code>. The <code>SQLObject</code> replacement
 * mapping will be retrieved with {@link #getParsedObjectsReplacementMap()}
 * before the {@link #resetState()} is called.
 */
public interface PostParseProcessor
{
    
    /**
     * The candidate types for this <code>PostParseProcessor</code>.
     * 
     * @return the <code>SQLQueryObject</code> types that this
     * 		 {@link PostParseProcessor#process(SQLQueryObject)} method gets
     * 		 invoked for 
     */
    public Class[] getProcessCandidateTypes();
    
    /**
     * Configures this <code>PostParseProcessor</code> with the arguments
     * provided in the given <code>PostParseProcessorConfiguration</code>.
     * 
     * @param config the configuration arguments that can be consumed by this
     *        <code>PostParseProcessor</code>
     */
    public void config(PostParseProcessorConfiguration config);
    
    /**
     * The call back method invoked for each of the
     * <code>SQLQueryObject</code>s in the <code>SQLQueryObject</code> model
     * instance returned by the parser, if the <code>SQLQueryObject</code>
     * complies with one of the types in the <code>Class[]</code> returned by
     * this {@link #getProcessCandidateTypes()}
     * 
     * @param sqlQuery instance of one of the types in the <code>Class[]</code>
     * 		 returned by {@link #getProcessCandidateTypes()}
     * @return List of {@link org.eclipse.datatools.sqltools.parsers.sql.SQLParseErrorInfo}
     * 		 objects, with information about the error encountered,
     * 		 must not be <code>null</code>
     * @throws SQLParserException if the error encountered is severe enough to
     * 		 discard the instance of the <code>SQLQueryObject</code> model
     * 		 returned by the parser
     */
    public List process(SQLQueryObject sqlQuery) throws SQLParserException;
    
    /**
     * If this <code>PostParseProcessor</code> is stateful within the context of
     * one <code>QueryStatement</code>, this method should reset the state, as
     * it might be used for post parse processing of more than one
     * <code>QueryStatement</code>.
     * This method is invoked after the <code>SQLParserManager</code>
     * bottom-up processed the <code>SQLQueryObject</code> elements of a
     * <code>QueryStatement</code> up to the top/root-element - the
     * <code>QueryStatement</code> object itself.
     */
    public void resetState();
    
    /**
     * Returns a Map containing parsed <code>SQLQueryObject</code>s mapped to
     * either the <code>SQLQueryObject</code> that replaces it, or is mapped to
     * <code>null</code>, if that parsed object simply is to be removed.
     * This method will be invoked after this <code>PostParseProcessor</code>
     * is done processing one <code>QueryStatement</code> before the
     * {@link #resetState()} method is called.
     * This method should return a mapping, if during the post parse processing
     * <code>SQLQueryObject</code>s are found that are determined to have been
     * created by the parser mistakenly, because the parser did not have the
     * knowledge about context or semantics. This then
     * <code>PostParseProcessor</code> might create a substitution for it or
     * simply delete it by mapping the original <code>SQLQueryObject</code> to
     * <code>null</code>.
     * By providing this mapping, invalid <code>SQLQueryObject</code>s can be
     * removed from the parser's parsed-element-list and therefore will not be
     * subject to further post parse processing of other
     * <code>PostParseProcessor</code>s.
     * 
     * <ul>
     * 	<li>key: <code>SQLQueryObject</code> originally created by parser</li>
     * 	<li>value: <code>SQLQueryObject</code> created by this
     *   	<code>PostParseProcessor</code> to substitute the originally
     * 		created by the parser</li>
     * </ul>
     * 
     * @return Map containing parsed <code>SQLQueryObject</code>s mapped to
     * 		either the <code>SQLQueryObject</code> that replaces it to
     * 		<code>null</code>, if that parsed object simply is to be removed
     */
    public Map getParsedObjectsReplacementMap();
    
}
