/*
 * Copyright (c) 2022-2022 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0, which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the
 * Eclipse Public License v. 2.0 are satisfied: GNU General Public License,
 * version 2 with the GNU Classpath Exception, which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 */
package ee.jakarta.tck.authentication.test.common;

import static java.lang.Boolean.getBoolean;
import static java.util.logging.Level.SEVERE;
import static org.jboss.shrinkwrap.api.ShrinkWrap.create;
import static org.jsoup.Jsoup.parse;
import static org.jsoup.parser.Parser.xmlParser;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.Map;
import java.util.logging.Logger;

import org.jboss.arquillian.test.api.ArquillianResource;
import org.jboss.shrinkwrap.api.Archive;
import org.jboss.shrinkwrap.api.ArchivePath;
import org.jboss.shrinkwrap.api.Node;
import org.jboss.shrinkwrap.api.spec.EnterpriseArchive;
import org.jboss.shrinkwrap.api.spec.WebArchive;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.rules.TestWatcher;
import org.junit.runner.Description;

import com.gargoylesoftware.htmlunit.FailingHttpStatusCodeException;
import com.gargoylesoftware.htmlunit.WebClient;

/**
 *
 * @author Arjan Tijms
 *
 */
public class ArquillianBase {

    private static final String WEBAPP_SRC = "src/main/webapp";
    private static final Logger logger = Logger.getLogger(ArquillianBase.class.getName());

    private WebClient webClient;
    private String response;

    @Rule
    public TestWatcher ruleExample = new TestWatcher() {
        @Override
        protected void failed(Throwable e, Description description) {
            super.failed(e, description);

            logger.log(SEVERE,
                "\n\nTest failed: " +
                description.getClassName() + "." + description.getMethodName() +

                "\nMessage: " + e.getMessage() +

                "\nLast response: " +

                "\n\n"  + formatHTML(response) + "\n\n");

        }
    };

    public static String formatHTML(String html) {
        try {
            return parse(html, "", xmlParser()).toString();
        } catch (Exception e) {
            return html;
        }
    }

    public static Archive<?> defaultArchive() {
        return tryWrapEAR(defaultWebArchive());
    }

    public static WebArchive defaultWebArchive() {
        return
            removeTestClasses(
                create(WebArchive.class, "test.war")
                    .addPackages(true, "ee.jakarta.tck.authentication.test")
                    .addAsWebInfResource(resource("web.xml"))
                    );
    }

    private static WebArchive removeTestClasses(WebArchive archive) {
        for (Map.Entry<ArchivePath, Node> content : archive.getContent().entrySet()) {
            if (content.getKey().get().endsWith("Test.class")) {
                archive.delete(content.getKey().get());
            }
        }
        archive.deleteClass(ArquillianBase.class);

        return archive;
    }

    public static Archive<?> tryWrapEAR(WebArchive webArchive) {
        if (getBoolean("useEarForJaspic")) {
            return
                // EAR archive
                create(EnterpriseArchive.class, "test.ear")

                    // Web module
                    // This is needed to prevent Arquillian generating an illegal application.xml
                    .addAsModule(
                        webArchive
                    );
        } else {
            return webArchive;
        }
    }

    public static File resource(String name) {
        return new File(WEBAPP_SRC + "/WEB-INF", name);
    }

    public static File web(String name) {
        return new File(WEBAPP_SRC, name);
    }

    @ArquillianResource
    private URL base;

    @Before
    public void setUp() {
        webClient = new WebClient();
        webClient.getOptions().setThrowExceptionOnFailingStatusCode(false);
        if (System.getProperty("glassfish.suspend") != null) {
            webClient.getOptions().setTimeout(0);
        }
    }

    @After
    public void tearDown() {
        webClient.getCookieManager().clearCookies();
        webClient.close();
    }



    protected WebClient getWebClient() {
        return webClient;
    }

    protected URL getBase() {
        return base;
    }

    /**
     * Gets content from the path that's relative to the base URL on which the Arquillian test
     * archive is deployed.
     *
     * @param path the path relative to the URL on which the Arquillian test is deployed
     * @return the raw content as a string as returned by the server
     */
    protected String getFromServerPath(final String path) {
        response = null;
        for (int i=0; i<=3; i++) {
            try {
                response = webClient.getPage(base + path).getWebResponse().getContentAsString();
                if (!response.contains("The response wrapper must wrap the response obtained from getResponse()")) {
                    return response;
                }
            } catch (FailingHttpStatusCodeException | IOException e) {
                throw new IllegalStateException(e);
            }
        }

        return response;
    }

}
