/*
 * KIELER - Kiel Integrated Environment for Layout Eclipse RichClient
 *
 * http://www.informatik.uni-kiel.de/rtsys/kieler/
 * 
 * Copyright 2010 by
 * + Christian-Albrechts-University of Kiel
 *   + Department of Computer Science
 *     + Real-Time and Embedded Systems Group
 * 
 * This code is provided under the terms of the Eclipse Public License (EPL).
 * See the file epl-v10.html for the license text.
 */
package de.cau.cs.kieler.klay.layered.p5edges;

import java.util.EnumSet;
import java.util.List;
import java.util.ListIterator;
import java.util.Set;
import de.cau.cs.kieler.core.alg.AbstractAlgorithm;
import de.cau.cs.kieler.kiml.options.LayoutOptions;
import de.cau.cs.kieler.klay.layered.ILayoutPhase;
import de.cau.cs.kieler.klay.layered.IntermediateProcessingConfiguration;
import de.cau.cs.kieler.klay.layered.graph.LNode;
import de.cau.cs.kieler.klay.layered.graph.Layer;
import de.cau.cs.kieler.klay.layered.graph.LGraph;
import de.cau.cs.kieler.klay.layered.intermediate.LayoutProcessorStrategy;
import de.cau.cs.kieler.klay.layered.properties.GraphProperties;
import de.cau.cs.kieler.klay.layered.properties.Properties;

/**
 * Edge routing implementation that creates orthogonal bend points. Inspired by
 * <ul>
 *   <li>Georg Sander, Layout of directed hypergraphs with orthogonal hyperedges. In
 *     <i>Proceedings of the 11th International Symposium on Graph Drawing (GD '03)</i>,
 *     LNCS vol. 2912, pp. 381-386, Springer, 2004.</li>
 *   <li>Giuseppe di Battista, Peter Eades, Roberto Tamassia, Ioannis G. Tollis,
 *     <i>Graph Drawing: Algorithms for the Visualization of Graphs</i>,
 *     Prentice Hall, New Jersey, 1999 (Section 9.4, for cycle breaking in the
 *     hyperedge segment graph)
 * </ul>
 * 
 * <dl>
 *   <dt>Precondition:</dt><dd>the graph has a proper layering with
 *     assigned node and port positions; the size of each layer is
 *     correctly set; edges connected to ports on strange sides were
 *     processed</dd>
 *   <dt>Postcondition:</dt><dd>each node is assigned a horizontal coordinate;
 *     the bend points of each edge are set; the width of the whole graph is set</dd>
 * </dl>
 *
 * @author msp
 * @author cds
 * @author jjc
 * @kieler.design proposed by msp
 * @kieler.rating proposed yellow by msp
 */
public class OrthogonalEdgeRouter extends AbstractAlgorithm implements ILayoutPhase {
    
    /* The basic processing strategy for this phase is empty. Depending on
     * the graph features, dependencies on intermediate processors are added
     * dynamically as follows:
     * 
     * Before phase 1:
     *   - None.
     * 
     * Before phase 2:
     *   - For center edge labels:
     *      - LABEL_DUMMY_INSERTER
     * 
     * Before phase 3:
     *   - For non-free ports:
     *     - NORTH_SOUTH_PORT_PREPROCESSOR
     *     - INVERTED_PORT_PROCESSOR
     *   
     *   - For self-loops:
     *     - SELF_LOOP_PROCESSOR
     *   
     *   - For hierarchical ports:
     *     - HIERARCHICAL_PORT_CONSTRAINT_PROCESSOR
     *   
     *   - For center edge labels:
     *      - LABEL_DUMMY_SWITCHER
     * 
     * Before phase 4:
     *   - For hyperedges:
     *     - HYPEREDGE_DUMMY_MERGER
     * 
     * Before phase 5:
     *   - For hierarchical ports:
     *     - HIERARCHICAL_PORT_DUMMY_SIZE_PROCESSOR
     * 
     * After phase 5:
     *   - For non-free ports:
     *     - NORTH_SOUTH_PORT_POSTPROCESSOR
     *   
     *   - For hierarchical ports:
     *     - HIERARCHICAL_PORT_ORTHOGONAL_EDGE_ROUTER
     *     
     *   - For center edge labels:
     *      - LABEL_DUMMY_REMOVER
     *     
     *   - For end edge labels:
     *     - END_LABEL_PROCESSOR
     */
    
    /** additional processor dependencies for graphs with hyperedges. */
    private static final IntermediateProcessingConfiguration HYPEREDGE_PROCESSING_ADDITIONS =
        new IntermediateProcessingConfiguration(IntermediateProcessingConfiguration.BEFORE_PHASE_4,
                LayoutProcessorStrategy.HYPEREDGE_DUMMY_MERGER);
    
    /** additional processor dependencies for graphs with possible inverted ports. */
    private static final IntermediateProcessingConfiguration INVERTED_PORT_PROCESSING_ADDITIONS =
        new IntermediateProcessingConfiguration(IntermediateProcessingConfiguration.BEFORE_PHASE_3,
                LayoutProcessorStrategy.INVERTED_PORT_PROCESSOR);
    
    /** additional processor dependencies for graphs with northern / southern non-free ports. */
    private static final IntermediateProcessingConfiguration NORTH_SOUTH_PORT_PROCESSING_ADDITIONS =
        new IntermediateProcessingConfiguration(
                // Before Phase 1
                null,
                
                // Before Phase 2
                null,
                
                // Before Phase 3
                EnumSet.of(LayoutProcessorStrategy.NORTH_SOUTH_PORT_PREPROCESSOR),
                
                // Before Phase 4
                null,
                
                // Before Phase 5
                null,
                
                // After Phase 5
                EnumSet.of(LayoutProcessorStrategy.NORTH_SOUTH_PORT_POSTPROCESSOR));
    
    /** additional processor dependencies for graphs with hierarchical ports. */
    private static final IntermediateProcessingConfiguration HIERARCHICAL_PORT_PROCESSING_ADDITIONS =
        new IntermediateProcessingConfiguration(
                // Before Phase 1
                null,
                
                // Before Phase 2
                null,
                
                // Before Phase 3
                EnumSet.of(LayoutProcessorStrategy.HIERARCHICAL_PORT_CONSTRAINT_PROCESSOR),
                
                // Before Phase 4
                null,
                
                // Before Phase 5
                EnumSet.of(LayoutProcessorStrategy.HIERARCHICAL_PORT_DUMMY_SIZE_PROCESSOR),
                
                // After Phase 5
                EnumSet.of(LayoutProcessorStrategy.HIERARCHICAL_PORT_ORTHOGONAL_EDGE_ROUTER));
    
    /** additional processor dependencies for graphs with self-loops. */
    private static final IntermediateProcessingConfiguration SELF_LOOP_PROCESSING_ADDITIONS =
        new IntermediateProcessingConfiguration(IntermediateProcessingConfiguration.BEFORE_PHASE_3,
                LayoutProcessorStrategy.SELF_LOOP_PROCESSOR);
    
    /** additional processor dependencies for graphs with hypernodes. */
    private static final IntermediateProcessingConfiguration HYPERNODE_PROCESSING_ADDITIONS =
        new IntermediateProcessingConfiguration(IntermediateProcessingConfiguration.AFTER_PHASE_5,
                LayoutProcessorStrategy.HYPERNODE_PROCESSOR);
    
//    /** additional processor dependencies for graphs with center edge labels. */
//    private static final IntermediateProcessingConfiguration CENTER_EDGE_LABEL_PROCESSING_ADDITIONS =
//        new IntermediateProcessingConfiguration(
//                // Before Phase 1
//                null,
//                
//                // Before Phase 2
//                EnumSet.of(LayoutProcessorStrategy.LABEL_DUMMY_INSERTER),
//                
//                // Before Phase 3
//                EnumSet.of(LayoutProcessorStrategy.LABEL_DUMMY_SWITCHER),
//                
//                // Before Phase 4
//                null,
//                
//                // Before Phase 5
//                null,
//                
//                // After Phase 5
//                EnumSet.of(LayoutProcessorStrategy.LABEL_DUMMY_REMOVER));
//    
//    /** additional processor dependencies for graphs with head or tail edge labels. */
//    private static final IntermediateProcessingConfiguration END_EDGE_LABEL_PROCESSING_ADDITIONS =
//        new IntermediateProcessingConfiguration(
//                // Before Phase 1
//                null,
//                
//                // Before Phase 2
//                null,
//                
//                // Before Phase 3
//                null,
//                
//                // Before Phase 4
//                EnumSet.of(LayoutProcessorStrategy.NODE_MARGIN_CALCULATOR),
//                
//                // Before Phase 5
//                null,
//                
//                // After Phase 5
//                EnumSet.of(LayoutProcessorStrategy.END_LABEL_PROCESSOR));
    
    /**
     * {@inheritDoc}
     */
    public IntermediateProcessingConfiguration getIntermediateProcessingConfiguration(
            final LGraph graph) {
        
        Set<GraphProperties> graphProperties = graph.getProperty(Properties.GRAPH_PROPERTIES);
        
        // Basic configuration
        IntermediateProcessingConfiguration configuration = new IntermediateProcessingConfiguration();
        
        // Additional dependencies
        if (graphProperties.contains(GraphProperties.HYPEREDGES)) {
            configuration.addAll(HYPEREDGE_PROCESSING_ADDITIONS);
            configuration.addAll(INVERTED_PORT_PROCESSING_ADDITIONS);
        }
        
        if (graphProperties.contains(GraphProperties.NON_FREE_PORTS)) {
            configuration.addAll(INVERTED_PORT_PROCESSING_ADDITIONS);

            if (graphProperties.contains(GraphProperties.NORTH_SOUTH_PORTS)) {
                configuration.addAll(NORTH_SOUTH_PORT_PROCESSING_ADDITIONS);
            }
        }

        if (graphProperties.contains(GraphProperties.EXTERNAL_PORTS)) {
            configuration.addAll(HIERARCHICAL_PORT_PROCESSING_ADDITIONS);
        }

        if (graphProperties.contains(GraphProperties.SELF_LOOPS)) {
            configuration.addAll(SELF_LOOP_PROCESSING_ADDITIONS);
        }
        
        if (graphProperties.contains(GraphProperties.HYPERNODES)) {
            configuration.addAll(HYPERNODE_PROCESSING_ADDITIONS);
        }
        
        // For the release, we deactivate label handling for the moment as it's still in development
//        if (graphProperties.contains(GraphProperties.CENTER_LABELS)) {
//            configuration.addAll(CENTER_EDGE_LABEL_PROCESSING_ADDITIONS);
//        }
//        
//        if (graphProperties.contains(GraphProperties.END_LABELS)) {
//            configuration.addAll(END_EDGE_LABEL_PROCESSING_ADDITIONS);
//        }
        
        return configuration;
    }
    
    /**
     * {@inheritDoc}
     */
    public void process(final LGraph layeredGraph) {
        getMonitor().begin("Orthogonal edge routing", 1);
        
        // Retrieve some generic values
        double nodeSpacing = layeredGraph.getProperty(Properties.OBJ_SPACING);
        double edgeSpacing = nodeSpacing * layeredGraph.getProperty(Properties.EDGE_SPACING_FACTOR);
        boolean debug = layeredGraph.getProperty(LayoutOptions.DEBUG_MODE);
        
        // Prepare for iteration!
        OrthogonalRoutingGenerator routingGenerator = new OrthogonalRoutingGenerator(
                new OrthogonalRoutingGenerator.WestToEastRoutingStrategy(), edgeSpacing,
                debug ? "phase5" : null);
        float xpos = 0.0f;
        ListIterator<Layer> layerIter = layeredGraph.getLayers().listIterator();
        Layer leftLayer = null;
        Layer rightLayer = null;
        List<LNode> leftLayerNodes = null;
        List<LNode> rightLayerNodes = null;
        int leftLayerIndex = -1;
        int rightLayerIndex = -1;
        
        // Iterate!
        do {
            int slotsCount;
            
            // Fetch the next layer, if any
            rightLayer = layerIter.hasNext() ? layerIter.next() : null;
            rightLayerNodes = rightLayer == null ? null : rightLayer.getNodes();
            rightLayerIndex = layerIter.previousIndex();
            
            // Place the left layer's nodes, if any
            if (leftLayer != null) {
                leftLayer.placeNodes(xpos);
                xpos += leftLayer.getSize().x + nodeSpacing;
            }
            
            // Route edges between the two layers
            slotsCount = routingGenerator.routeEdges(layeredGraph, leftLayerNodes, leftLayerIndex,
                    rightLayerNodes, xpos);
            if (slotsCount > 0) {
                xpos += slotsCount * edgeSpacing - edgeSpacing + nodeSpacing;
            }
            
            leftLayer = rightLayer;
            leftLayerNodes = rightLayerNodes;
            leftLayerIndex = rightLayerIndex;
        } while (rightLayer != null);
        
        layeredGraph.getSize().x = xpos;
        
        getMonitor().done();
    }
    
}
