/**
 * Copyright (c) 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package fr.inria.diverse.melange.ast;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import fr.inria.diverse.melange.ast.LanguageExtensions;
import fr.inria.diverse.melange.lib.EcoreExtensions;
import fr.inria.diverse.melange.lib.ModelUtils;
import fr.inria.diverse.melange.metamodel.melange.Import;
import fr.inria.diverse.melange.metamodel.melange.Metamodel;
import fr.inria.diverse.melange.metamodel.melange.ModelingElement;
import fr.inria.diverse.melange.utils.EPackageProvider;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import org.apache.log4j.Logger;
import org.eclipse.emf.codegen.ecore.genmodel.GenClass;
import org.eclipse.emf.codegen.ecore.genmodel.GenClassifier;
import org.eclipse.emf.codegen.ecore.genmodel.GenDataType;
import org.eclipse.emf.codegen.ecore.genmodel.GenEnum;
import org.eclipse.emf.codegen.ecore.genmodel.GenModel;
import org.eclipse.emf.codegen.ecore.genmodel.GenPackage;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.EMap;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EAnnotation;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EDataType;
import org.eclipse.emf.ecore.EEnum;
import org.eclipse.emf.ecore.EModelElement;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EOperation;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.StringExtensions;

/**
 * A collection of utilities around {@link ModelingElement}s
 */
@SuppressWarnings("all")
public class ModelingElementExtensions {
  @Inject
  @Extension
  private EcoreExtensions _ecoreExtensions;
  
  @Inject
  @Extension
  private LanguageExtensions _languageExtensions;
  
  @Inject
  private EPackageProvider registry;
  
  @Inject
  private ModelUtils modelUtils;
  
  private static final Logger log = Logger.getLogger(ModelingElementExtensions.class);
  
  /**
   * Returns all the {@link EPackage}s defined in {@code m}.
   * 
   * @see EPackageProvider#getPackages
   */
  public Set<EPackage> getPkgs(final ModelingElement m) {
    return this.registry.getPackages(m);
  }
  
  /**
   * Returns all the {@link GenModel}s used by {@code m}.
   * 
   * @see EPackageProvider#getGenModels
   */
  public Set<GenModel> getGenmodels(final ModelingElement m) {
    return this.registry.getGenModels(m);
  }
  
  /**
   * Returns all the {@link GenModel}s used by {@code withOperator}.
   */
  public Set<GenModel> getGenmodels(final Import withOperator) {
    final HashSet<String> genmodelUris = CollectionLiterals.<String>newHashSet();
    EList<String> _genmodelUris = withOperator.getGenmodelUris();
    Iterables.<String>addAll(genmodelUris, _genmodelUris);
    if (((withOperator.getGenmodelUris().size() == 0) && (withOperator.getEcoreUri() != null))) {
      String _substring = withOperator.getEcoreUri().substring(0, withOperator.getEcoreUri().lastIndexOf("."));
      String _plus = (_substring + ".genmodel");
      genmodelUris.add(_plus);
    }
    final Function1<String, GenModel> _function = new Function1<String, GenModel>() {
      @Override
      public GenModel apply(final String it) {
        return ModelingElementExtensions.this.modelUtils.loadGenmodel(it);
      }
    };
    return IterableExtensions.<GenModel>toSet(IterableExtensions.<String, GenModel>map(genmodelUris, _function));
  }
  
  /**
   * Checks whether the Ecore pointed by the {@code ecoreUri} of {@code m}
   * is an Xcore file.
   */
  public boolean isXcore(final ModelingElement m) {
    return ((!StringExtensions.isNullOrEmpty(m.getEcoreUri())) && m.getEcoreUri().endsWith(".xcore"));
  }
  
  /**
   * Returns the set of all {@link GenPackage}s defined by the {@link GenModel}s
   * of the {@link ModelingElement} {@code m}.
   */
  public Set<GenPackage> getAllGenPkgs(final ModelingElement m) {
    final Function1<GenModel, List<GenPackage>> _function = new Function1<GenModel, List<GenPackage>>() {
      @Override
      public List<GenPackage> apply(final GenModel it) {
        return ModelingElementExtensions.this._ecoreExtensions.getAllGenPkgs(it);
      }
    };
    return IterableExtensions.<GenPackage>toSet(Iterables.<GenPackage>concat(IterableExtensions.<GenModel, List<GenPackage>>map(this.getGenmodels(m), _function)));
  }
  
  /**
   * Returns the set of all {@link GenPackage}s defined by the {@link GenModel}s
   * of the {@link Import} {@code withOperator}.
   */
  public Set<GenPackage> getAllGenPkgs(final Import withOperator) {
    final Function1<GenModel, List<GenPackage>> _function = new Function1<GenModel, List<GenPackage>>() {
      @Override
      public List<GenPackage> apply(final GenModel it) {
        return ModelingElementExtensions.this._ecoreExtensions.getAllGenPkgs(it);
      }
    };
    return IterableExtensions.<GenPackage>toSet(Iterables.<GenPackage>concat(IterableExtensions.<GenModel, List<GenPackage>>map(this.getGenmodels(withOperator), _function)));
  }
  
  /**
   * Returns the {@link GenPackage} corresponding to the {@link EPackage}
   * {@code pkg} in {@code m}.
   */
  public GenPackage getGenPkgFor(final ModelingElement m, final EPackage pkg) {
    final Function1<GenPackage, Boolean> _function = new Function1<GenPackage, Boolean>() {
      @Override
      public Boolean apply(final GenPackage it) {
        String _uniqueId = ModelingElementExtensions.this._ecoreExtensions.getUniqueId(it.getEcorePackage());
        String _uniqueId_1 = ModelingElementExtensions.this._ecoreExtensions.getUniqueId(pkg);
        return Boolean.valueOf(Objects.equal(_uniqueId, _uniqueId_1));
      }
    };
    return IterableExtensions.<GenPackage>findFirst(this.getAllGenPkgs(m), _function);
  }
  
  /**
   * Returns the {@link GenClassifier} corresponding to the {@link EClassifier}
   * {@code cls} in {@code m}.
   */
  public GenClassifier getGenClassifierFor(final ModelingElement m, final EClassifier cls) {
    final Function1<GenClassifier, Boolean> _function = new Function1<GenClassifier, Boolean>() {
      @Override
      public Boolean apply(final GenClassifier it) {
        String _name = it.getName();
        String _name_1 = cls.getName();
        return Boolean.valueOf(Objects.equal(_name, _name_1));
      }
    };
    return IterableExtensions.<GenClassifier>findFirst(this.getGenPkgFor(m, cls.getEPackage()).getGenClassifiers(), _function);
  }
  
  /**
   * Returns the {@link GenClass} corresponding to the {@link EClass}
   * {@code cls} in {@code m}.
   */
  public GenClass getGenClsFor(final ModelingElement m, final EClass cls) {
    final Function1<GenClass, Boolean> _function = new Function1<GenClass, Boolean>() {
      @Override
      public Boolean apply(final GenClass it) {
        String _name = it.getName();
        String _name_1 = cls.getName();
        return Boolean.valueOf(Objects.equal(_name, _name_1));
      }
    };
    return IterableExtensions.<GenClass>findFirst(this.getGenPkgFor(m, cls.getEPackage()).getGenClasses(), _function);
  }
  
  /**
   * Returns the {@link GenDataType} corresponding to the {@link EDataType}
   * {@code dt} in {@code m}.
   */
  public GenDataType getGenDataTypeFor(final ModelingElement m, final EDataType dt) {
    GenDataType _xifexpression = null;
    if ((dt instanceof EEnum)) {
      final Function1<GenEnum, Boolean> _function = new Function1<GenEnum, Boolean>() {
        @Override
        public Boolean apply(final GenEnum it) {
          String _name = it.getName();
          String _name_1 = ((EEnum)dt).getName();
          return Boolean.valueOf(Objects.equal(_name, _name_1));
        }
      };
      _xifexpression = IterableExtensions.<GenEnum>findFirst(this.getGenPkgFor(m, ((EEnum)dt).getEPackage()).getGenEnums(), _function);
    } else {
      final Function1<GenDataType, Boolean> _function_1 = new Function1<GenDataType, Boolean>() {
        @Override
        public Boolean apply(final GenDataType it) {
          String _name = it.getName();
          String _name_1 = dt.getName();
          return Boolean.valueOf(Objects.equal(_name, _name_1));
        }
      };
      _xifexpression = IterableExtensions.<GenDataType>findFirst(this.getGenPkgFor(m, dt.getEPackage()).getGenDataTypes(), _function_1);
    }
    return _xifexpression;
  }
  
  /**
   * Returns the URI of the first {@link EPackage} defined by {@code m}.
   */
  public String getRootPackageUri(final ModelingElement m) {
    return IterableExtensions.<GenPackage>head(this.getAllGenPkgs(m)).getEcorePackage().getNsURI();
  }
  
  /**
   * Returns all the {@link EClassifier}s defined by {@code m}.
   */
  public Iterable<EClassifier> getAllClassifiers(final ModelingElement m) {
    final Function1<EPackage, List<EClassifier>> _function = new Function1<EPackage, List<EClassifier>>() {
      @Override
      public List<EClassifier> apply(final EPackage it) {
        return ModelingElementExtensions.this._ecoreExtensions.getAllClassifiers(it);
      }
    };
    return IterableExtensions.<EClassifier>toSet(Iterables.<EClassifier>concat(IterableExtensions.<EPackage, List<EClassifier>>map(this.getPkgs(m), _function)));
  }
  
  /**
   * Returns all the {@link EClass}es defined in {@code m}.
   */
  public Iterable<EClass> getAllClasses(final ModelingElement m) {
    final Function1<EPackage, List<EClass>> _function = new Function1<EPackage, List<EClass>>() {
      @Override
      public List<EClass> apply(final EPackage it) {
        return ModelingElementExtensions.this._ecoreExtensions.getAllClasses(it);
      }
    };
    return IterableExtensions.<EClass>toSet(Iterables.<EClass>concat(IterableExtensions.<EPackage, List<EClass>>map(this.getPkgs(m), _function)));
  }
  
  /**
   * Returns the {@link EClass} in {@code m} with the name {@code clsName}.
   * {@code clsName} can be simple or qualified.
   * 
   * Return null if not found
   * 
   * @see EcoreExtensions.findClass(EPackage pkg, String clsName)
   */
  public EClass findClass(final ModelingElement m, final String clsName) {
    final Function1<EPackage, Boolean> _function = new Function1<EPackage, Boolean>() {
      @Override
      public Boolean apply(final EPackage it) {
        EPackage _eSuperPackage = it.getESuperPackage();
        return Boolean.valueOf((_eSuperPackage == null));
      }
    };
    final Function1<EPackage, EClass> _function_1 = new Function1<EPackage, EClass>() {
      @Override
      public EClass apply(final EPackage it) {
        return ModelingElementExtensions.this._ecoreExtensions.findClass(it, clsName);
      }
    };
    return IterableExtensions.<EClass>head(IterableExtensions.<EClass>filterNull(IterableExtensions.<EPackage, EClass>map(IterableExtensions.<EPackage>filter(this.getPkgs(m), _function), _function_1)));
  }
  
  /**
   * Returns the {@link EClassifier} in {@code m} with the name {@code clsName}.
   * {@code clsName} can be simple or qualified.
   * 
   * Return null if not found
   * 
   * @see EcoreExtensions.findClassifier(EPackage pkg, String clsName)
   */
  public EClassifier findClassifier(final ModelingElement m, final String clsName) {
    final Function1<EPackage, Boolean> _function = new Function1<EPackage, Boolean>() {
      @Override
      public Boolean apply(final EPackage it) {
        EPackage _eSuperPackage = it.getESuperPackage();
        return Boolean.valueOf((_eSuperPackage == null));
      }
    };
    final Function1<EPackage, EClassifier> _function_1 = new Function1<EPackage, EClassifier>() {
      @Override
      public EClassifier apply(final EPackage it) {
        return ModelingElementExtensions.this._ecoreExtensions.findClassifier(it, clsName);
      }
    };
    return IterableExtensions.<EClassifier>head(IterableExtensions.<EClassifier>filterNull(IterableExtensions.<EPackage, EClassifier>map(IterableExtensions.<EPackage>filter(this.getPkgs(m), _function), _function_1)));
  }
  
  /**
   * Creates the Ecore file corresponding to {@code m} at the location
   * {@code uri} using {@code baseUri} as beginning for its EPackages nsURI
   * 
   * @param hideAspectElements whether the elements coming from aspects
   * (ie. annotated with 'aspect') should be hidden in the serialized Ecore
   * @return the serialized root {@link EPackage}
   */
  public EPackage createEcore(final ModelingElement m, final String uri, final String baseUri, final boolean hideAspectElements) {
    final ResourceSetImpl resSet = new ResourceSetImpl();
    final Resource res = resSet.createResource(URI.createURI(uri));
    final Function1<EPackage, Boolean> _function = new Function1<EPackage, Boolean>() {
      @Override
      public Boolean apply(final EPackage it) {
        EPackage _eSuperPackage = it.getESuperPackage();
        return Boolean.valueOf((_eSuperPackage == null));
      }
    };
    final Iterable<EPackage> rootPkgs = IterableExtensions.<EPackage>filter(this.getPkgs(m), _function);
    final Function1<EPackage, Boolean> _function_1 = new Function1<EPackage, Boolean>() {
      @Override
      public Boolean apply(final EPackage it) {
        EPackage _eSuperPackage = it.getESuperPackage();
        return Boolean.valueOf((_eSuperPackage == null));
      }
    };
    final Collection<EPackage> copy = EcoreUtil.<EPackage>copyAll(IterableExtensions.<EPackage>toList(IterableExtensions.<EPackage>filter(this.getPkgs(m), _function_1)));
    if ((baseUri != null)) {
      final Consumer<EPackage> _function_2 = new Consumer<EPackage>() {
        @Override
        public void accept(final EPackage it) {
          ModelingElementExtensions.this._ecoreExtensions.initializeNsUriWith(it, baseUri);
        }
      };
      copy.forEach(_function_2);
    }
    if ((m instanceof Metamodel)) {
      final ArrayList<EModelElement> toRemove = CollectionLiterals.<EModelElement>newArrayList();
      final TreeIterator<EObject> i = IterableExtensions.<EPackage>head(copy).eAllContents();
      while (i.hasNext()) {
        {
          final EObject obj = i.next();
          if (hideAspectElements) {
            if ((obj instanceof EModelElement)) {
              boolean _isAspectSpecific = this._ecoreExtensions.isAspectSpecific(((EModelElement)obj));
              if (_isAspectSpecific) {
                if ((obj instanceof EStructuralFeature)) {
                  boolean _isAspectSpecific_1 = this._ecoreExtensions.isAspectSpecific(((EStructuralFeature)obj).getEType());
                  boolean _not = (!_isAspectSpecific_1);
                  if (_not) {
                  } else {
                    toRemove.add(((EStructuralFeature)obj));
                  }
                } else {
                  toRemove.add(((EModelElement)obj));
                }
              }
            }
          }
          if ((obj instanceof EStructuralFeature)) {
            boolean _isVolatile = ((EStructuralFeature)obj).isVolatile();
            if (_isVolatile) {
              ((EStructuralFeature)obj).setVolatile(false);
            }
          }
        }
      }
      final Consumer<EModelElement> _function_3 = new Consumer<EModelElement>() {
        @Override
        public void accept(final EModelElement it) {
          EcoreUtil.delete(it);
        }
      };
      toRemove.forEach(_function_3);
    }
    final Consumer<EPackage> _function_4 = new Consumer<EPackage>() {
      @Override
      public void accept(final EPackage pkg) {
        ModelingElementExtensions.this._ecoreExtensions.replaceLocalEObjectReferencesToEcoreEObjectReferences(pkg);
        final BiConsumer<EObject, Collection<EStructuralFeature.Setting>> _function = new BiConsumer<EObject, Collection<EStructuralFeature.Setting>>() {
          @Override
          public void accept(final EObject obj, final Collection<EStructuralFeature.Setting> settings) {
            final Function1<EStructuralFeature.Setting, Boolean> _function = new Function1<EStructuralFeature.Setting, Boolean>() {
              @Override
              public Boolean apply(final EStructuralFeature.Setting it) {
                return Boolean.valueOf((((it.getEStructuralFeature() != null) && (!it.getEStructuralFeature().isDerived())) && (!it.getEStructuralFeature().isMany())));
              }
            };
            final Consumer<EStructuralFeature.Setting> _function_1 = new Consumer<EStructuralFeature.Setting>() {
              @Override
              public void accept(final EStructuralFeature.Setting ss) {
                if ((obj instanceof EClassifier)) {
                  final Function1<EPackage, EList<EClassifier>> _function = new Function1<EPackage, EList<EClassifier>>() {
                    @Override
                    public EList<EClassifier> apply(final EPackage it) {
                      return it.getEClassifiers();
                    }
                  };
                  final Function1<EClassifier, Boolean> _function_1 = new Function1<EClassifier, Boolean>() {
                    @Override
                    public Boolean apply(final EClassifier it) {
                      String _name = it.getName();
                      String _name_1 = ((EClassifier)obj).getName();
                      return Boolean.valueOf(Objects.equal(_name, _name_1));
                    }
                  };
                  final EClassifier corresponding = IterableExtensions.<EClassifier>findFirst(Iterables.<EClassifier>concat(IterableExtensions.<EPackage, EList<EClassifier>>map(copy, _function)), _function_1);
                  if ((corresponding != null)) {
                    ss.getEObject().eSet(ss.getEStructuralFeature(), corresponding);
                  }
                } else {
                  if ((obj instanceof EReference)) {
                    EReference _eOpposite = ((EReference)obj).getEOpposite();
                    boolean _tripleNotEquals = (_eOpposite != null);
                    if (_tripleNotEquals) {
                      final Function1<EPackage, EList<EClassifier>> _function_2 = new Function1<EPackage, EList<EClassifier>>() {
                        @Override
                        public EList<EClassifier> apply(final EPackage it) {
                          return it.getEClassifiers();
                        }
                      };
                      final Function1<EClassifier, Boolean> _function_3 = new Function1<EClassifier, Boolean>() {
                        @Override
                        public Boolean apply(final EClassifier it) {
                          String _name = it.getName();
                          String _name_1 = ((EReference)obj).getEContainingClass().getName();
                          return Boolean.valueOf(Objects.equal(_name, _name_1));
                        }
                      };
                      EClassifier _findFirst = IterableExtensions.<EClassifier>findFirst(Iterables.<EClassifier>concat(IterableExtensions.<EPackage, EList<EClassifier>>map(copy, _function_2)), _function_3);
                      final EClass correspondingCls = ((EClass) _findFirst);
                      final Function1<EReference, Boolean> _function_4 = new Function1<EReference, Boolean>() {
                        @Override
                        public Boolean apply(final EReference it) {
                          String _name = it.getName();
                          String _name_1 = ((EReference)obj).getName();
                          return Boolean.valueOf(Objects.equal(_name, _name_1));
                        }
                      };
                      final EReference correspondingRef = IterableExtensions.<EReference>findFirst(correspondingCls.getEReferences(), _function_4);
                      if ((correspondingRef != null)) {
                        ss.getEObject().eSet(ss.getEStructuralFeature(), correspondingRef);
                      }
                    }
                  }
                }
              }
            };
            IterableExtensions.<EStructuralFeature.Setting>filter(settings, _function).forEach(_function_1);
          }
        };
        EcoreUtil.ExternalCrossReferencer.find(pkg).forEach(_function);
      }
    };
    copy.forEach(_function_4);
    if ((m instanceof Metamodel)) {
      boolean _isGeneratedByMelange = this._languageExtensions.isGeneratedByMelange(((Metamodel)m).getOwningLanguage());
      if (_isGeneratedByMelange) {
        final Function1<EPackage, List<EClass>> _function_5 = new Function1<EPackage, List<EClass>>() {
          @Override
          public List<EClass> apply(final EPackage it) {
            return ModelingElementExtensions.this._ecoreExtensions.getAllClasses(it);
          }
        };
        final Set<EClass> classes = IterableExtensions.<EClass>toSet(Iterables.<EClass>concat(IterableExtensions.<EPackage, List<EClass>>map(copy, _function_5)));
        final Function1<EClass, EList<EOperation>> _function_6 = new Function1<EClass, EList<EOperation>>() {
          @Override
          public EList<EOperation> apply(final EClass it) {
            return it.getEOperations();
          }
        };
        final Consumer<EOperation> _function_7 = new Consumer<EOperation>() {
          @Override
          public void accept(final EOperation op) {
            EAnnotation _eAnnotation = op.getEAnnotation("http://www.eclipse.org/emf/2002/GenModel");
            EMap<String, String> _details = null;
            if (_eAnnotation!=null) {
              _details=_eAnnotation.getDetails();
            }
            if (_details!=null) {
              _details.removeKey("body");
            }
          }
        };
        Iterables.<EOperation>concat(IterableExtensions.<EClass, EList<EOperation>>map(classes, _function_6)).forEach(_function_7);
      }
    }
    EList<EObject> _contents = res.getContents();
    Iterables.<EObject>addAll(_contents, copy);
    try {
      final HashMap<String, String> options = CollectionLiterals.<String, String>newHashMap();
      options.put(Resource.OPTION_SAVE_ONLY_IF_CHANGED, 
        Resource.OPTION_SAVE_ONLY_IF_CHANGED_MEMORY_BUFFER);
      res.save(options);
    } catch (final Throwable _t) {
      if (_t instanceof IOException) {
        final IOException e = (IOException)_t;
        ModelingElementExtensions.log.error(("Error while serializing new Ecore for " + m), e);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
    return IterableExtensions.<EPackage>head(rootPkgs);
  }
}
