/**
 * Copyright (c) 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package fr.inria.diverse.melange.utils;

import com.google.common.base.Objects;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Iterables;
import com.google.common.collect.SetMultimap;
import com.google.inject.Inject;
import fr.inria.diverse.melange.ast.AspectExtensions;
import fr.inria.diverse.melange.ast.LanguageExtensions;
import fr.inria.diverse.melange.eclipse.EclipseProjectHelper;
import fr.inria.diverse.melange.metamodel.melange.Aspect;
import fr.inria.diverse.melange.metamodel.melange.Language;
import java.io.BufferedWriter;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.function.Consumer;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.xtext.common.types.JvmDeclaredType;
import org.eclipse.xtext.common.types.JvmFormalParameter;
import org.eclipse.xtext.common.types.JvmOperation;
import org.eclipse.xtext.common.types.JvmType;
import org.eclipse.xtext.common.types.JvmTypeReference;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

@SuppressWarnings("all")
public class AspectOverrider {
  @Inject
  @Extension
  private LanguageExtensions _languageExtensions;
  
  @Inject
  @Extension
  private AspectExtensions _aspectExtensions;
  
  @Inject
  private EclipseProjectHelper eclipseHelper;
  
  public void generateAspectJ(final Language l) {
    try {
      final String packageName = this._languageExtensions.getAspectsNamespace(l);
      final String aspectName = "_MelangeDispatcher";
      final IProject targetProject = ResourcesPlugin.getWorkspace().getRoot().getProject(this._languageExtensions.getExternalRuntimeName(l));
      String _path = targetProject.getLocationURI().getPath();
      String _plus = (_path + "/src-gen/");
      String _replaceAll = packageName.replaceAll("\\.", "/");
      final String targetAspectFolder = (_plus + _replaceAll);
      final Function1<Aspect, JvmDeclaredType> _function = new Function1<Aspect, JvmDeclaredType>() {
        @Override
        public JvmDeclaredType apply(final Aspect it) {
          JvmType _type = it.getAspectTypeRef().getType();
          return ((JvmDeclaredType) _type);
        }
      };
      final List<JvmDeclaredType> aspects = ListExtensions.<Aspect, JvmDeclaredType>map(l.getSemantics(), _function);
      final String advices = this.generateAdvices(aspects);
      boolean _isEmpty = advices.isEmpty();
      boolean _not = (!_isEmpty);
      if (_not) {
        final StringBuilder fileContent = new StringBuilder();
        fileContent.append((("package " + packageName) + ";\n\n"));
        fileContent.append((("public aspect " + aspectName) + " {\n"));
        fileContent.append(advices);
        fileContent.append("}\n");
        final Path path = Paths.get((((targetAspectFolder + "/") + aspectName) + ".aj"));
        final BufferedWriter writer = Files.newBufferedWriter(path);
        writer.write(fileContent.toString());
        writer.flush();
        this.convertToAspectJ(targetProject);
        targetProject.refreshLocal(IResource.DEPTH_INFINITE, null);
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  /**
   * Generate AspectJ advice to intercept methods calls when some
   * K3 aspects override others K3 aspects
   */
  private String generateAdvices(final List<JvmDeclaredType> aspects) {
    final StringBuilder content = new StringBuilder();
    final SetMultimap<JvmOperation, JvmOperation> res = HashMultimap.<JvmOperation, JvmOperation>create();
    final SetMultimap<JvmDeclaredType, JvmDeclaredType> overriders = HashMultimap.<JvmDeclaredType, JvmDeclaredType>create();
    final Consumer<JvmDeclaredType> _function = new Consumer<JvmDeclaredType>() {
      @Override
      public void accept(final JvmDeclaredType cls) {
        final List<JvmDeclaredType> ancestors = AspectOverrider.this.getAllSuperAspects(cls);
        boolean _isEmpty = ancestors.isEmpty();
        boolean _not = (!_isEmpty);
        if (_not) {
          overriders.putAll(cls, ancestors);
        }
      }
    };
    aspects.forEach(_function);
    final Consumer<JvmDeclaredType> _function_1 = new Consumer<JvmDeclaredType>() {
      @Override
      public void accept(final JvmDeclaredType childCls) {
        final Set<JvmDeclaredType> ancestors = overriders.get(childCls);
        final Consumer<JvmOperation> _function = new Consumer<JvmOperation>() {
          @Override
          public void accept(final JvmOperation m) {
            final Function1<JvmDeclaredType, Boolean> _function = new Function1<JvmDeclaredType, Boolean>() {
              @Override
              public Boolean apply(final JvmDeclaredType it) {
                return Boolean.valueOf(AspectOverrider.this.hasMethod(it, m));
              }
            };
            final Consumer<JvmDeclaredType> _function_1 = new Consumer<JvmDeclaredType>() {
              @Override
              public void accept(final JvmDeclaredType superCls) {
                final Function1<JvmOperation, Boolean> _function = new Function1<JvmOperation, Boolean>() {
                  @Override
                  public Boolean apply(final JvmOperation it) {
                    return Boolean.valueOf(AspectOverrider.this.isEqual(it, m));
                  }
                };
                final JvmOperation target = IterableExtensions.<JvmOperation>findFirst(Iterables.<JvmOperation>filter(superCls.getMembers(), JvmOperation.class), _function);
                AspectOverrider.this.storeRule(res, m, target);
              }
            };
            IterableExtensions.<JvmDeclaredType>filter(ancestors, _function).forEach(_function_1);
          }
        };
        Iterables.<JvmOperation>filter(childCls.getMembers(), JvmOperation.class).forEach(_function);
      }
    };
    overriders.keySet().forEach(_function_1);
    int _size = res.keySet().size();
    boolean _greaterThan = (_size > 0);
    if (_greaterThan) {
      final Consumer<JvmOperation> _function_2 = new Consumer<JvmOperation>() {
        @Override
        public void accept(final JvmOperation m1) {
          final String returnType = m1.getReturnType().getType().getQualifiedName();
          final String calledMethodFqn = m1.getQualifiedName();
          final Function1<JvmFormalParameter, String> _function = new Function1<JvmFormalParameter, String>() {
            @Override
            public String apply(final JvmFormalParameter it) {
              String _qualifiedName = it.getParameterType().getType().getQualifiedName();
              String _plus = (_qualifiedName + " ");
              String _name = it.getName();
              return (_plus + _name);
            }
          };
          final String parameters = IterableExtensions.join(ListExtensions.<JvmFormalParameter, String>map(m1.getParameters(), _function), ",");
          final Function1<JvmFormalParameter, String> _function_1 = new Function1<JvmFormalParameter, String>() {
            @Override
            public String apply(final JvmFormalParameter it) {
              return it.getParameterType().getType().getQualifiedName();
            }
          };
          final String parameterTypes = IterableExtensions.join(ListExtensions.<JvmFormalParameter, String>map(m1.getParameters(), _function_1), ",");
          final Function1<JvmFormalParameter, String> _function_2 = new Function1<JvmFormalParameter, String>() {
            @Override
            public String apply(final JvmFormalParameter it) {
              return it.getName();
            }
          };
          final String parameterNames = IterableExtensions.join(ListExtensions.<JvmFormalParameter, String>map(m1.getParameters(), _function_2), ",");
          final Consumer<JvmOperation> _function_3 = new Consumer<JvmOperation>() {
            @Override
            public void accept(final JvmOperation m2) {
              final String targetMethodFqn = m2.getQualifiedName();
              content.append((((((((((((("\t" + returnType) + " around(") + parameters) + ") : call(") + returnType) + " ") + targetMethodFqn) + "(") + parameterTypes) + ")) && args(") + parameterNames) + ") {\n"));
              boolean _notEquals = (!Objects.equal(returnType, "void"));
              if (_notEquals) {
                content.append((((("\t\treturn " + calledMethodFqn) + "(") + parameterNames) + ");\n"));
              } else {
                content.append((((("\t\t" + calledMethodFqn) + "(") + parameterNames) + ");\n"));
              }
              content.append("\t}\n");
            }
          };
          res.get(m1).forEach(_function_3);
        }
      };
      res.keySet().forEach(_function_2);
      return content.toString();
    }
    return "";
  }
  
  private void storeRule(final SetMultimap<JvmOperation, JvmOperation> storage, final JvmOperation called, final JvmOperation hidden) {
    final Function1<JvmOperation, Boolean> _function = new Function1<JvmOperation, Boolean>() {
      @Override
      public Boolean apply(final JvmOperation it) {
        return Boolean.valueOf(AspectOverrider.this.isEqual(it, called));
      }
    };
    final JvmOperation colider = IterableExtensions.<JvmOperation>findFirst(storage.keySet(), _function);
    if (((colider == null) || (colider == called))) {
      storage.put(called, hidden);
    } else {
      boolean _contains = this.getAllSuperAspects(called.getDeclaringType()).contains(colider.getDeclaringType());
      if (_contains) {
        final Set<JvmOperation> toMove = storage.get(colider);
        storage.removeAll(colider);
        storage.putAll(called, toMove);
        storage.put(called, hidden);
      }
    }
  }
  
  private boolean hasMethod(final JvmDeclaredType cls, final JvmOperation method) {
    final Function1<JvmOperation, Boolean> _function = new Function1<JvmOperation, Boolean>() {
      @Override
      public Boolean apply(final JvmOperation it) {
        return Boolean.valueOf(AspectOverrider.this.isEqual(it, method));
      }
    };
    return IterableExtensions.<JvmOperation>exists(Iterables.<JvmOperation>filter(cls.getMembers(), JvmOperation.class), _function);
  }
  
  private boolean isEqual(final JvmOperation m1, final JvmOperation m2) {
    return ((Objects.equal(m1.getSimpleName(), m2.getSimpleName()) && (m1.getParameters().size() == m2.getParameters().size())) && IterableExtensions.<JvmFormalParameter>forall(m1.getParameters(), new Function1<JvmFormalParameter, Boolean>() {
      @Override
      public Boolean apply(final JvmFormalParameter param) {
        boolean _xblockexpression = false;
        {
          final int index = m1.getParameters().indexOf(param);
          JvmType _type = m2.getParameters().get(index).getParameterType().getType();
          JvmType _type_1 = param.getParameterType().getType();
          _xblockexpression = Objects.equal(_type, _type_1);
        }
        return Boolean.valueOf(_xblockexpression);
      }
    }));
  }
  
  private List<JvmDeclaredType> getAllSuperAspects(final JvmDeclaredType cls) {
    final ArrayList<JvmDeclaredType> res = CollectionLiterals.<JvmDeclaredType>newArrayList();
    boolean _hasAspectAnnotation = this._aspectExtensions.hasAspectAnnotation(cls);
    boolean _not = (!_hasAspectAnnotation);
    if (_not) {
      return res;
    }
    JvmType _type = IterableExtensions.<JvmTypeReference>head(cls.getSuperTypes()).getType();
    JvmDeclaredType current = ((JvmDeclaredType) _type);
    while (((current != null) && this._aspectExtensions.hasAspectAnnotation(current))) {
      {
        res.add(current);
        JvmType _type_1 = IterableExtensions.<JvmTypeReference>head(current.getSuperTypes()).getType();
        current = ((JvmDeclaredType) _type_1);
      }
    }
    return res;
  }
  
  private void convertToAspectJ(final IProject prj) {
    this.eclipseHelper.addDependencies(prj, Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("org.aspectj.runtime")));
    this.eclipseHelper.addNature(prj, "org.eclipse.ajdt.ui.ajnature");
    this.eclipseHelper.addBuilder(prj, "org.eclipse.ajdt.core.ajbuilder");
  }
}
