/**
 * Copyright (c) 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package fr.inria.diverse.melange.ast;

import com.google.inject.Inject;
import fr.inria.diverse.melange.metamodel.melange.Metamodel;
import java.io.IOException;
import java.util.List;
import java.util.function.Consumer;
import org.apache.log4j.Logger;
import org.eclipse.emf.codegen.ecore.genmodel.GenJDKLevel;
import org.eclipse.emf.codegen.ecore.genmodel.GenModel;
import org.eclipse.emf.codegen.ecore.genmodel.GenModelFactory;
import org.eclipse.emf.codegen.ecore.genmodel.GenPackage;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.xtext.naming.IQualifiedNameProvider;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.eclipse.xtext.xbase.lib.StringExtensions;

/**
 * A collection of utilities around {@link Metamodel}s
 */
@SuppressWarnings("all")
public class MetamodelExtensions {
  @Inject
  @Extension
  private IQualifiedNameProvider _iQualifiedNameProvider;

  @Inject
  @Extension
  private LanguageExtensions _languageExtensions;

  @Inject
  @Extension
  private ModelingElementExtensions _modelingElementExtensions;

  private static final Logger log = Logger.getLogger(MetamodelExtensions.class);

  /**
   * Checks whether the given {@link Metamodel} {@code mm} is well-formed,
   * ie. it has at least one package and associated genmodel.
   */
  public boolean isValid(final Metamodel mm) {
    return ((!IterableExtensions.isEmpty(IterableExtensions.<EPackage>filterNull(this._modelingElementExtensions.getPkgs(mm)))) && (!IterableExtensions.isEmpty(IterableExtensions.<GenModel>filterNull(this._modelingElementExtensions.getGenmodels(mm)))));
  }

  /**
   * Generates and serializes a {@link GenModel} at the location {@code gmUri}
   * for the {@link Metamodel} {@code mm} whose Ecore file is located at
   * {@code ecoreUri}, using {@code targetModelDirectory} as its model directory.
   */
  public void createGenmodel(final Metamodel mm, final String ecoreUri, final String gmUri, final String targetModelDirectory) {
    final ResourceSetImpl resSet = new ResourceSetImpl();
    final Resource pkgRes = resSet.getResource(URI.createURI(ecoreUri), true);
    final Function1<EObject, EPackage> _function = (EObject it) -> {
      return ((EPackage) it);
    };
    final List<EPackage> pkgs = ListExtensions.<EObject, EPackage>map(pkgRes.getContents(), _function);
    GenModel _createGenModel = GenModelFactory.eINSTANCE.createGenModel();
    final Procedure1<GenModel> _function_1 = (GenModel it) -> {
      it.setComplianceLevel(GenJDKLevel.JDK70_LITERAL);
      it.setModelDirectory(targetModelDirectory.replaceFirst("platform:/resource", "").replaceFirst("..", ""));
      EList<String> _foreignModel = it.getForeignModel();
      _foreignModel.add(ecoreUri);
      it.setModelName(mm.getOwningLanguage().getName());
      it.setModelPluginID(this._languageExtensions.getExternalRuntimeName(mm.getOwningLanguage()));
      it.initialize(pkgs);
      final Consumer<GenPackage> _function_2 = (GenPackage gp) -> {
        gp.setBasePackage(this._iQualifiedNameProvider.getFullyQualifiedName(mm.getOwningLanguage()).toLowerCase().toString());
        boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(mm.getOwningLanguage().getFileExtension());
        boolean _not = (!_isNullOrEmpty);
        if (_not) {
          gp.setFileExtensions(mm.getOwningLanguage().getFileExtension());
        } else {
          gp.setFileExtensions(mm.getOwningLanguage().getName().toLowerCase().toString());
        }
      };
      it.getGenPackages().forEach(_function_2);
    };
    final GenModel genmodel = ObjectExtensions.<GenModel>operator_doubleArrow(_createGenModel, _function_1);
    final Resource res = resSet.createResource(URI.createURI(gmUri));
    EList<EObject> _contents = res.getContents();
    _contents.add(genmodel);
    try {
      res.save(null);
    } catch (final Throwable _t) {
      if (_t instanceof IOException) {
        final IOException e = (IOException)_t;
        MetamodelExtensions.log.error("Error while serializing new genmodel", e);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }
}
