/**
 * Copyright (c) 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package fr.inria.diverse.melange.processors;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import fr.inria.diverse.melange.ast.ASTHelper;
import fr.inria.diverse.melange.ast.ModelTypeExtensions;
import fr.inria.diverse.melange.metamodel.melange.Element;
import fr.inria.diverse.melange.metamodel.melange.ExternalLanguage;
import fr.inria.diverse.melange.metamodel.melange.Import;
import fr.inria.diverse.melange.metamodel.melange.Language;
import fr.inria.diverse.melange.metamodel.melange.Mapping;
import fr.inria.diverse.melange.metamodel.melange.MelangeFactory;
import fr.inria.diverse.melange.metamodel.melange.ModelType;
import fr.inria.diverse.melange.metamodel.melange.ModelTypingSpace;
import java.util.Arrays;
import java.util.List;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

/**
 * For each {@link Language} in the current {@link Resource}, extract its
 * exact {@link ModelType}. The new model type is named $LanguageName$MT, or
 * can be explicitly specified by the user using the 'exactType' clause.
 */
@SuppressWarnings("all")
public class ExactTypeInferrer extends DispatchMelangeProcessor {
  @Inject
  @Extension
  private ASTHelper _aSTHelper;

  @Inject
  @Extension
  private ModelTypeExtensions _modelTypeExtensions;

  private List<ModelType> inferredMTs = CollectionLiterals.<ModelType>newArrayList();

  protected void _preProcess(final ModelTypingSpace root, final boolean preLinkingPhase) {
    this.inferredMTs = CollectionLiterals.<ModelType>newArrayList();
    final Consumer<Language> _function = (Language l) -> {
      String _exactTypeName = l.getExactTypeName();
      boolean _tripleEquals = (_exactTypeName == null);
      if (_tripleEquals) {
        String _name = l.getName();
        String _plus = (_name + "MT");
        l.setExactTypeName(_plus);
      }
      ModelType _createModelType = MelangeFactory.eINSTANCE.createModelType();
      final Procedure1<ModelType> _function_1 = (ModelType it) -> {
        it.setName(l.getExactTypeName());
      };
      final ModelType newMT = ObjectExtensions.<ModelType>operator_doubleArrow(_createModelType, _function_1);
      final Function1<ModelType, Boolean> _function_2 = (ModelType it) -> {
        String _name_1 = it.getName();
        String _name_2 = newMT.getName();
        return Boolean.valueOf(Objects.equal(_name_1, _name_2));
      };
      boolean _exists = IterableExtensions.<ModelType>exists(this._aSTHelper.getModelTypes(root), _function_2);
      boolean _not = (!_exists);
      if (_not) {
        this.inferredMTs.add(newMT);
      }
    };
    this._aSTHelper.getLanguages(root).forEach(_function);
    EList<Element> _elements = root.getElements();
    Iterables.<Element>addAll(_elements, this.inferredMTs);
    final Consumer<Language> _function_1 = (Language l) -> {
      final Function1<ModelType, Boolean> _function_2 = (ModelType it) -> {
        String _name = it.getName();
        String _exactTypeName = l.getExactTypeName();
        return Boolean.valueOf(Objects.equal(_name, _exactTypeName));
      };
      l.setExactType(IterableExtensions.<ModelType>findFirst(this._aSTHelper.getModelTypes(root), _function_2));
      this.initializeExactType(l);
    };
    this._aSTHelper.getLanguages(root).forEach(_function_1);
    final Consumer<Mapping> _function_2 = (Mapping m) -> {
      final Function1<ModelType, Boolean> _function_3 = (ModelType it) -> {
        String _name = it.getName();
        ModelType _to = m.getTo();
        String _name_1 = null;
        if (_to!=null) {
          _name_1=_to.getName();
        }
        return Boolean.valueOf(Objects.equal(_name, _name_1));
      };
      m.setTo(IterableExtensions.<ModelType>findFirst(this._aSTHelper.getModelTypes(root), _function_3));
    };
    this._aSTHelper.getMappings(root).forEach(_function_2);
  }

  protected void _postProcess(final ModelTypingSpace root) {
    final Consumer<ModelType> _function = (ModelType it) -> {
      EcoreUtil.remove(it);
    };
    this.inferredMTs.forEach(_function);
  }

  /**
   * Initializes the #ecoreUri of the exact {@link ModelType} of the supplied
   * {@code language}.
   */
  private void initializeExactType(final Language language) {
    if ((language instanceof ExternalLanguage)) {
      ModelType _exactType = ((ExternalLanguage)language).getExactType();
      Import _head = IterableExtensions.<Import>head(Iterables.<Import>filter(((ExternalLanguage)language).getOperators(), Import.class));
      String _ecoreUri = null;
      if (_head!=null) {
        _ecoreUri=_head.getEcoreUri();
      }
      _exactType.setEcoreUri(_ecoreUri);
    } else {
      ModelType _exactType_1 = language.getExactType();
      boolean _tripleNotEquals = (_exactType_1 != null);
      if (_tripleNotEquals) {
        ModelType _exactType_2 = language.getExactType();
        _exactType_2.setEcoreUri(this._modelTypeExtensions.getInferredEcoreUri(language.getExactType()));
      }
    }
  }

  public void preProcess(final EObject root, final boolean preLinkingPhase) {
    if (root instanceof ModelTypingSpace) {
      _preProcess((ModelTypingSpace)root, preLinkingPhase);
      return;
    } else if (root != null) {
      _preProcess(root, preLinkingPhase);
      return;
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(root, preLinkingPhase).toString());
    }
  }

  public void postProcess(final EObject root) {
    if (root instanceof ModelTypingSpace) {
      _postProcess((ModelTypingSpace)root);
      return;
    } else if (root != null) {
      _postProcess(root);
      return;
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(root).toString());
    }
  }
}
