/**
 * Copyright (c) 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package fr.inria.diverse.melange.ast;

import com.google.common.base.Objects;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.SetMultimap;
import com.google.inject.Inject;
import fr.inria.diverse.melange.lib.EcoreExtensions;
import fr.inria.diverse.melange.metamodel.melange.Metamodel;
import fr.inria.diverse.melange.metamodel.melange.ModelType;
import fr.inria.diverse.melange.metamodel.melange.ModelTypingSpace;
import fr.inria.diverse.melange.metamodel.melange.ModelingElement;
import fr.inria.diverse.melange.metamodel.melange.Transformation;
import java.util.function.Consumer;
import org.eclipse.emf.codegen.ecore.genmodel.GenPackage;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EDataType;
import org.eclipse.emf.ecore.EFactory;
import org.eclipse.emf.ecore.EOperation;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EParameter;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.common.types.JvmOperation;
import org.eclipse.xtext.naming.IQualifiedNameConverter;
import org.eclipse.xtext.naming.IQualifiedNameProvider;
import org.eclipse.xtext.naming.QualifiedName;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure2;
import org.eclipse.xtext.xbase.lib.StringExtensions;

/**
 * A collection of utilities around naming conventions in Melange
 * (generated code, adapters, UML specificities, etc.)
 */
@SuppressWarnings("all")
public class NamingHelper {
  @Inject
  @Extension
  private ModelingElementExtensions _modelingElementExtensions;

  @Inject
  @Extension
  private EcoreExtensions _ecoreExtensions;

  @Inject
  @Extension
  private IQualifiedNameProvider _iQualifiedNameProvider;

  @Inject
  @Extension
  private IQualifiedNameConverter _iQualifiedNameConverter;

  /**
   * Returns the namespace corresponding to the first {@link EPackage}
   * in the {@link ModelingElement} {@code m}.
   */
  public String getRootPackageNamespace(final ModelingElement m) {
    return this.getPackageNamespace(IterableExtensions.<GenPackage>head(this._modelingElementExtensions.getAllGenPkgs(m)));
  }

  /**
   * Return the list of packages from {@link m} associated with their java qualified name
   */
  public SetMultimap<String, String> getRootPackageNamespaces(final ModelingElement m) {
    final SetMultimap<String, String> res = HashMultimap.<String, String>create();
    final Consumer<GenPackage> _function = (GenPackage it) -> {
      EPackage _eSuperPackage = it.getEcorePackage().getESuperPackage();
      boolean _tripleEquals = (_eSuperPackage == null);
      if (_tripleEquals) {
        final String syntaxPackage = this._ecoreExtensions.getUniqueId(it.getEcorePackage());
        final String javaPackage = this.getPackageNamespace(it);
        res.put(syntaxPackage, javaPackage);
      }
    };
    this._modelingElementExtensions.getAllGenPkgs(m).forEach(_function);
    return res;
  }

  /**
   * Returns the (simple) name of the class generated for the first
   * {@link EPackage} of {@code m}.
   */
  public String getRootPackageName(final ModelingElement m) {
    return IterableExtensions.<GenPackage>head(this._modelingElementExtensions.getAllGenPkgs(m)).getPackageInterfaceName();
  }

  /**
   * Returns the fully qualified name of the class generated for the first
   * {@link EPackage} of {@code m}.
   */
  public String getRootPackageFqn(final ModelingElement m) {
    return IterableExtensions.<GenPackage>head(this._modelingElementExtensions.getAllGenPkgs(m)).getQualifiedPackageInterfaceName();
  }

  /**
   * Returns the namespace corresponding to the {@link EPackage} pointed
   * by the {@link GenPackage} {@code gp}.
   */
  public String getPackageNamespace(final GenPackage gp) {
    return this._iQualifiedNameConverter.toQualifiedName(gp.getQualifiedPackageInterfaceName()).skipLast(1).toString();
  }

  /**
   * Returns the fully qualified name of the {@link EFactory} generated
   * for the first {@link EPackage} of {@code m}.
   */
  public String getRootFactoryFqn(final ModelingElement m) {
    return IterableExtensions.<GenPackage>head(this._modelingElementExtensions.getAllGenPkgs(m)).getQualifiedFactoryInterfaceName();
  }

  /**
   * Returns the fully qualified name of the {@link EFactory} generated
   * for the {@link EPackage} {@code pkg}.
   */
  public String getFactoryFqnFor(final ModelingElement m, final EPackage pkg) {
    return this._modelingElementExtensions.getGenPkgFor(m, pkg).getQualifiedFactoryInterfaceName();
  }

  /**
   * Returns the (Java) fully qualified name of the {@link EPackage}
   * {@code pkg}.
   */
  public String getFqnFor(final ModelingElement m, final EPackage pkg) {
    return this._modelingElementExtensions.getGenPkgFor(m, pkg).getQualifiedPackageInterfaceName();
  }

  /**
   * Returns the (Java) fully qualified name of the {@link EClassifier}
   * {@code cls}.
   */
  public String getFqnFor(final ModelingElement m, final EClassifier cls) {
    String _switchResult = null;
    boolean _matched = false;
    if (cls instanceof EClass) {
      _matched=true;
      _switchResult = this._modelingElementExtensions.getGenClsFor(m, ((EClass)cls)).getQualifiedInterfaceName();
    }
    if (!_matched) {
      if (cls instanceof EDataType) {
        _matched=true;
        _switchResult = this._modelingElementExtensions.getGenDataTypeFor(m, ((EDataType)cls)).getQualifiedInstanceClassName();
      }
    }
    return _switchResult;
  }

  /**
   * Returns the fully qualified name of the generated factory that creates
   * adapters from the {@link Metamodel} {@code mm} to the {@link ModelType}
   * {@code mt}.
   */
  public String getAdaptersFactoryNameFor(final Metamodel mm, final ModelType mt) {
    QualifiedName _lowerCase = this._iQualifiedNameProvider.getFullyQualifiedName(mm.getOwningLanguage()).append("adapters").append(this._iQualifiedNameProvider.getFullyQualifiedName(mt).getLastSegment()).toLowerCase();
    String _lastSegment = this._iQualifiedNameConverter.toQualifiedName(mt.getName()).getLastSegment();
    String _plus = (_lastSegment + "AdaptersFactory");
    return this.normalize(_lowerCase.append(_plus)).toString();
  }

  /**
   * @deprecated
   */
  public String getMappersFactoryNameFor(final Metamodel sourceModel, final ModelType targetMT) {
    QualifiedName _lowerCase = this._iQualifiedNameProvider.getFullyQualifiedName(sourceModel.getOwningLanguage()).append("mappers").append(this._iQualifiedNameProvider.getFullyQualifiedName(targetMT).getLastSegment()).toLowerCase();
    String _lastSegment = this._iQualifiedNameConverter.toQualifiedName(targetMT.getName()).getLastSegment();
    String _plus = (_lastSegment + "MappersFactory");
    return this.normalize(_lowerCase.append(_plus)).toString();
  }

  /**
   * Returns the fully qualified name of the adapter class generated between
   * {@code mm} and {@code mt} for the {@link EClass} {@code cls}.
   */
  public String adapterNameFor(final Metamodel mm, final ModelType mt, final EClass cls) {
    return this.adapterNameFor(mm, mt, this._iQualifiedNameProvider.getFullyQualifiedName(cls).toString());
  }

  /**
   * Returns the fully qualified name of the adapter class generated between
   * {@code mm} and {@code mt} for the type named {@code name}.
   */
  public String adapterNameFor(final Metamodel mm, final ModelType mt, final String name) {
    return this.normalize(this._iQualifiedNameProvider.getFullyQualifiedName(mm.getOwningLanguage()).append("adapters").append(this._iQualifiedNameProvider.getFullyQualifiedName(mt).getLastSegment()).toLowerCase().append((name + "Adapter"))).toString();
  }

  /**
   * Returns the fully qualified name of the in-the-large adapter class
   * generated between {@code mm} and {@code mt}.
   */
  public String adapterNameFor(final Metamodel mm, final ModelType mt) {
    QualifiedName _lowerCase = this._iQualifiedNameProvider.getFullyQualifiedName(mm.getOwningLanguage()).append("adapters").append(this._iQualifiedNameProvider.getFullyQualifiedName(mt).getLastSegment()).toLowerCase();
    String _lastSegment = this._iQualifiedNameConverter.toQualifiedName(mm.getOwningLanguage().getName()).getLastSegment();
    String _plus = (_lastSegment + "Adapter");
    return this.normalize(_lowerCase.append(_plus)).toString();
  }

  /**
   * @deprecated
   */
  public String mapperNameFor(final Metamodel sourceModel, final ModelType targetMT) {
    QualifiedName _lowerCase = this._iQualifiedNameProvider.getFullyQualifiedName(sourceModel).append("mappers").append(this._iQualifiedNameProvider.getFullyQualifiedName(targetMT).getLastSegment()).toLowerCase();
    String _name = sourceModel.getOwningLanguage().getName();
    String _plus = (_name + "Adapter");
    return this.normalize(_lowerCase.append(_plus)).toString();
  }

  /**
   * @deprecated
   */
  public String adapterNameFor(final Metamodel mm, final Metamodel superMM, final EClass cls) {
    QualifiedName _lowerCase = this._iQualifiedNameProvider.getFullyQualifiedName(mm.getOwningLanguage()).append("adapters").append(this._iQualifiedNameConverter.toQualifiedName(superMM.getOwningLanguage().getName()).getLastSegment()).toLowerCase();
    String _name = cls.getName();
    String _plus = (_name + "Adapter");
    return this.normalize(_lowerCase.append(_plus)).toString();
  }

  /**
   * @deprecated
   */
  public String mapperNameFor(final Metamodel sourceModel, final ModelType targetMT, final EClass targetClass) {
    QualifiedName _lowerCase = this._iQualifiedNameProvider.getFullyQualifiedName(sourceModel.getOwningLanguage()).append("mappers").append(this._iQualifiedNameProvider.getFullyQualifiedName(targetMT).getLastSegment()).toLowerCase();
    String _name = targetClass.getName();
    String _plus = (_name + "Mapper");
    return this.normalize(_lowerCase.append(_plus)).toString();
  }

  /**
   * @deprecated
   */
  public String simpleMapperNameFor(final Metamodel sourceModel, final ModelType targetMT, final EClass targetClass) {
    String _name = targetClass.getName();
    return (_name + "Mapper");
  }

  /**
   * Returns the (simple) name of the adapter class generated for the
   * {@link EClass} {@code cls} between {@code mm} and {@code mt}.
   */
  public String simpleAdapterNameFor(final Metamodel mm, final ModelType mt, final EClass cls) {
    return this.simpleAdapterNameFor(mm, mt, cls.getName());
  }

  /**
   * Returns the (simple) name of the adapter class generated for the
   * type named {@code name} between {@code mm} and {@code mt}.
   */
  public String simpleAdapterNameFor(final Metamodel mm, final ModelType mt, final String name) {
    return this._iQualifiedNameConverter.toQualifiedName(this.adapterNameFor(mm, mt, name)).getLastSegment().toString();
  }

  /**
   * Returns the fully qualified name of the adapter class generated for the
   * {@link EFactory} between {@code mm} and {@code mt}.
   */
  public String factoryAdapterNameFor(final GenPackage pkg, final Metamodel mm, final ModelType mt) {
    QualifiedName _lowerCase = this._iQualifiedNameProvider.getFullyQualifiedName(mm.getOwningLanguage()).append("adapters").append(this._iQualifiedNameProvider.getFullyQualifiedName(mt).getLastSegment()).append(this._iQualifiedNameProvider.getFullyQualifiedName(pkg.getEcorePackage())).toLowerCase();
    String _factoryInterfaceName = pkg.getFactoryInterfaceName();
    String _plus = (_factoryInterfaceName + "Adapter");
    return this.normalize(_lowerCase.append(_plus)).toString();
  }

  /**
   * Returns the name of the Java class generated for the {@link Transformation}
   * {@code t}.
   */
  public String getClassName(final Transformation t) {
    return this._iQualifiedNameProvider.getFullyQualifiedName(t).skipLast(1).toLowerCase().append(t.getName()).toString();
  }

  /**
   * Returns the name of the StandaloneSetup class generated for the project
   * defined by the {@link ModelTypingspace} {@code root}.
   */
  public String getStandaloneSetupClassName(final ModelTypingSpace root) {
    return this._iQualifiedNameProvider.getFullyQualifiedName(root).append("StandaloneSetup").toString();
  }

  /**
   * Returns the name of the Java getter associated to the
   * {@link EStructuralFeature} {@code f}.
   */
  public String getGetterName(final EStructuralFeature f) {
    String _switchResult = null;
    boolean _matched = false;
    boolean _isUml = this._ecoreExtensions.isUml(f.getEContainingClass().getEPackage());
    if (_isUml) {
      _matched=true;
      _switchResult = this.getUmlGetterName(f);
    }
    if (!_matched) {
      if (f instanceof EAttribute) {
        _matched=true;
        String _xifexpression = null;
        if ((Objects.equal(((EAttribute)f).getEAttributeType().getInstanceClassName(), "boolean") && (!((EAttribute)f).isMany()))) {
          StringConcatenation _builder = new StringConcatenation();
          _builder.append("is");
          String _firstUpper = StringExtensions.toFirstUpper(((EAttribute)f).getName());
          _builder.append(_firstUpper);
          _xifexpression = _builder.toString();
        } else {
          StringConcatenation _builder_1 = new StringConcatenation();
          _builder_1.append("get");
          String _firstUpper_1 = StringExtensions.toFirstUpper(((EAttribute)f).getName());
          _builder_1.append(_firstUpper_1);
          _xifexpression = _builder_1.toString();
        }
        _switchResult = _xifexpression;
      }
    }
    if (!_matched) {
      if (f instanceof EReference) {
        _matched=true;
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("get");
        String _firstUpper = StringExtensions.toFirstUpper(((EReference)f).getName());
        _builder.append(_firstUpper);
        _switchResult = _builder.toString();
      }
    }
    return _switchResult;
  }

  /**
   * Returns the name of the Java getter associated to the
   * {@link JvmOperation} {@code op}.
   */
  public String getGetterName(final JvmOperation op) {
    String _xifexpression = null;
    String _simpleName = op.getReturnType().getType().getSimpleName();
    boolean _equals = Objects.equal(_simpleName, "boolean");
    if (_equals) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("is");
      String _firstUpper = StringExtensions.toFirstUpper(op.getSimpleName());
      _builder.append(_firstUpper);
      _xifexpression = _builder.toString();
    } else {
      StringConcatenation _builder_1 = new StringConcatenation();
      _builder_1.append("get");
      String _firstUpper_1 = StringExtensions.toFirstUpper(op.getSimpleName());
      _builder_1.append(_firstUpper_1);
      _xifexpression = _builder_1.toString();
    }
    return _xifexpression;
  }

  /**
   * Returns the name of the Java getter associated to the
   * {@link EStructuralFeature} {@code f}, taking into account the
   * specialized naming conventions of UML.
   */
  public String getUmlGetterName(final EStructuralFeature f) {
    String _switchResult = null;
    boolean _matched = false;
    if (f instanceof EAttribute) {
      _matched=true;
      String _xifexpression = null;
      String _instanceClassName = ((EAttribute)f).getEAttributeType().getInstanceClassName();
      boolean _equals = Objects.equal(_instanceClassName, "boolean");
      if (_equals) {
        String _xifexpression_1 = null;
        if ((((EAttribute)f).getName().startsWith("is") && Character.isUpperCase(((EAttribute)f).getName().charAt(2)))) {
          StringConcatenation _builder = new StringConcatenation();
          String _formatUmlFeatureName = this.formatUmlFeatureName(f);
          _builder.append(_formatUmlFeatureName);
          _xifexpression_1 = _builder.toString();
        } else {
          String _xifexpression_2 = null;
          boolean _isMany = ((EAttribute)f).isMany();
          if (_isMany) {
            StringConcatenation _builder_1 = new StringConcatenation();
            _builder_1.append("get");
            String _firstUpper = StringExtensions.toFirstUpper(this.formatUmlFeatureName(f));
            _builder_1.append(_firstUpper);
            _xifexpression_2 = _builder_1.toString();
          } else {
            StringConcatenation _builder_2 = new StringConcatenation();
            _builder_2.append("is");
            String _firstUpper_1 = StringExtensions.toFirstUpper(this.formatUmlFeatureName(f));
            _builder_2.append(_firstUpper_1);
            _xifexpression_2 = _builder_2.toString();
          }
          _xifexpression_1 = _xifexpression_2;
        }
        _xifexpression = _xifexpression_1;
      } else {
        StringConcatenation _builder_3 = new StringConcatenation();
        _builder_3.append("get");
        String _firstUpper_2 = StringExtensions.toFirstUpper(this.formatUmlFeatureName(f));
        _builder_3.append(_firstUpper_2);
        _xifexpression = _builder_3.toString();
      }
      _switchResult = _xifexpression;
    }
    if (!_matched) {
      if (f instanceof EReference) {
        _matched=true;
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("get");
        String _firstUpper = StringExtensions.toFirstUpper(this.formatUmlFeatureName(f));
        _builder.append(_firstUpper);
        _switchResult = _builder.toString();
      }
    }
    return _switchResult;
  }

  /**
   * Returns the name of the Java setter associated to the
   * {@link EStructuralFeature} {@code f}, taking into account the
   * specialized naming conventions of UML.
   */
  public String getUmlSetterName(final EStructuralFeature f) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("set");
    String _firstUpper = StringExtensions.toFirstUpper(this.formatUmlFeatureName(f));
    _builder.append(_firstUpper);
    return _builder.toString();
  }

  /**
   * Returns the name of the Java setter associated to the
   * {@link EStructuralFeature} {@code f}.
   */
  public String getSetterName(final EStructuralFeature f) {
    String _xifexpression = null;
    boolean _isUml = this._ecoreExtensions.isUml(f.getEContainingClass().getEPackage());
    if (_isUml) {
      _xifexpression = this.getUmlSetterName(f);
    } else {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("set");
      String _firstUpper = StringExtensions.toFirstUpper(f.getName());
      _builder.append(_firstUpper);
      _xifexpression = _builder.toString();
    }
    return _xifexpression;
  }

  /**
   * Returns the name of the Java getter associated to the
   * {@link JvmOperation} {@code op}.
   */
  public String getSetterName(final JvmOperation op) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("set");
    String _firstUpper = StringExtensions.toFirstUpper(op.getSimpleName());
    _builder.append(_firstUpper);
    return _builder.toString();
  }

  /**
   * Returns the name of the Java unsetter ({@code eUnset()}) associated to
   * the {@link EStructuralFeature} {@code f}.
   */
  public String getUnsetterName(final EStructuralFeature f) {
    String _xifexpression = null;
    boolean _isUml = this._ecoreExtensions.isUml(f.getEContainingClass().getEPackage());
    if (_isUml) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("unset");
      String _firstUpper = StringExtensions.toFirstUpper(this.formatUmlFeatureName(f));
      _builder.append(_firstUpper);
      _xifexpression = _builder.toString();
    } else {
      StringConcatenation _builder_1 = new StringConcatenation();
      _builder_1.append("unset");
      String _firstUpper_1 = StringExtensions.toFirstUpper(f.getName());
      _builder_1.append(_firstUpper_1);
      _xifexpression = _builder_1.toString();
    }
    return _xifexpression;
  }

  /**
   * Returns the name of the Java unsetter check ({@code eIsSet()})
   * associated to the {@link EStructuralFeature} {@code f}.
   */
  public String getUnsetterCheckName(final EStructuralFeature f) {
    String _xifexpression = null;
    boolean _isUml = this._ecoreExtensions.isUml(f.getEContainingClass().getEPackage());
    if (_isUml) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("isSet");
      String _firstUpper = StringExtensions.toFirstUpper(this.formatUmlFeatureName(f));
      _builder.append(_firstUpper);
      _xifexpression = _builder.toString();
    } else {
      StringConcatenation _builder_1 = new StringConcatenation();
      _builder_1.append("isSet");
      String _firstUpper_1 = StringExtensions.toFirstUpper(f.getName());
      _builder_1.append(_firstUpper_1);
      _xifexpression = _builder_1.toString();
    }
    return _xifexpression;
  }

  /**
   * Returns the name of the Java operation corresponding to the
   * {@link EOperation} {@code op}, taking into account UML naming conventions.
   */
  public String formatUmlOperationName(final EOperation op) {
    final String opName = this.toCamelCase(op.getName());
    EParameter _head = IterableExtensions.<EParameter>head(op.getEParameters());
    EClassifier _eType = null;
    if (_head!=null) {
      _eType=_head.getEType();
    }
    String _name = null;
    if (_eType!=null) {
      _name=_eType.getName();
    }
    boolean _equals = Objects.equal(_name, "EDiagnosticChain");
    if (_equals) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("validate");
      String _firstUpper = StringExtensions.toFirstUpper(opName);
      _builder.append(_firstUpper);
      return _builder.toString();
    }
    return opName;
  }

  /**
   * Returns the name of the Java operation corresponding to the
   * {@link EStructuralFeature} {@code f}, taking into account UML naming
   * conventions.
   */
  public String formatUmlFeatureName(final EStructuralFeature f) {
    String _xifexpression = null;
    String _name = f.getName();
    boolean _equals = Objects.equal(_name, "class");
    if (_equals) {
      _xifexpression = "class_";
    } else {
      String _xifexpression_1 = null;
      boolean _isMany = f.isMany();
      if (_isMany) {
        String _xifexpression_2 = null;
        boolean _endsWith = f.getName().endsWith("es");
        if (_endsWith) {
          StringConcatenation _builder = new StringConcatenation();
          String _name_1 = f.getName();
          _builder.append(_name_1);
          _builder.append("es");
          _xifexpression_2 = _builder.toString();
        } else {
          String _xifexpression_3 = null;
          if ((f.getName().endsWith("y") && (!f.getName().endsWith("By")))) {
            StringConcatenation _builder_1 = new StringConcatenation();
            String _name_2 = f.getName();
            int _length = f.getName().length();
            int _minus = (_length - 1);
            String _substring = _name_2.substring(0, _minus);
            _builder_1.append(_substring);
            _builder_1.append("ies");
            _xifexpression_3 = _builder_1.toString();
          } else {
            String _xifexpression_4 = null;
            boolean _endsWith_1 = f.getName().endsWith("ex");
            if (_endsWith_1) {
              StringConcatenation _builder_2 = new StringConcatenation();
              String _name_3 = f.getName();
              int _length_1 = f.getName().length();
              int _minus_1 = (_length_1 - 2);
              String _substring_1 = _name_3.substring(0, _minus_1);
              _builder_2.append(_substring_1);
              _builder_2.append("ices");
              _xifexpression_4 = _builder_2.toString();
            } else {
              String _xifexpression_5 = null;
              boolean _endsWith_2 = f.getName().endsWith("ss");
              if (_endsWith_2) {
                StringConcatenation _builder_3 = new StringConcatenation();
                String _name_4 = f.getName();
                _builder_3.append(_name_4);
                _builder_3.append("es");
                _xifexpression_5 = _builder_3.toString();
              } else {
                String _xifexpression_6 = null;
                boolean _endsWith_3 = f.getName().endsWith("Data");
                boolean _not = (!_endsWith_3);
                if (_not) {
                  StringConcatenation _builder_4 = new StringConcatenation();
                  String _name_5 = f.getName();
                  _builder_4.append(_name_5);
                  _builder_4.append("s");
                  _xifexpression_6 = _builder_4.toString();
                } else {
                  _xifexpression_6 = f.getName();
                }
                _xifexpression_5 = _xifexpression_6;
              }
              _xifexpression_4 = _xifexpression_5;
            }
            _xifexpression_3 = _xifexpression_4;
          }
          _xifexpression_2 = _xifexpression_3;
        }
        _xifexpression_1 = _xifexpression_2;
      } else {
        _xifexpression_1 = f.getName();
      }
      _xifexpression = _xifexpression_1;
    }
    return _xifexpression;
  }

  /**
   * Wololo, wololo wololo.
   */
  public String toCamelCase(final String s) {
    final String[] parts = s.split("_");
    final StringBuilder res = new StringBuilder();
    final Procedure2<String, Integer> _function = (String p, Integer i) -> {
      String _xifexpression = null;
      if (((i).intValue() == 0)) {
        _xifexpression = p;
      } else {
        _xifexpression = StringExtensions.toFirstUpper(p);
      }
      res.append(_xifexpression);
    };
    IterableExtensions.<String>forEach(((Iterable<String>)Conversions.doWrapArray(parts)), _function);
    return res.toString();
  }

  /**
   * Wololo, wololo wololo.
   */
  private String normalize(final QualifiedName name) {
    final StringBuilder res = new StringBuilder();
    final Function1<String, String> _function = (String it) -> {
      return it.toLowerCase();
    };
    res.append(IterableExtensions.join(ListExtensions.<String, String>map(name.skipLast(1).getSegments(), _function), "."));
    String _lastSegment = name.getLastSegment();
    String _plus = ("." + _lastSegment);
    res.append(_plus);
    return res.toString();
  }
}
