/*******************************************************************************
 * Copyright (c) 2013, 2016- UT-Battelle, LLC.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Initial API and implementation and/or initial documentation - Jay Jay Billings,
 *   Jordan H. Deyton, Dasha Gorin, Alexander J. McCaskey, Taylor Patterson,
 *   Claire Saunders, Matthew Wang, Anna Wojtowicz
 *******************************************************************************/

package org.eclipse.january.form;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;

import org.eclipse.eavp.viz.service.IVizServiceFactory;
import org.eclipse.january.form.internal.VizServiceFactoryHolder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * <p>
 * This class acts as a manager for discovering and creating resource objects in
 * ICE. Resources in ICE can be used in Model Items for things such as
 * displaying the contents of a mesh file, or in Launcher Items for tasks such
 * as plotting the contents of a post-processed CSV file.
 * </p>
 *
 * <p>
 * Some resources contain visualization data and can be rendered via the
 * visualization service. A private list of all file extensions that have a
 * IVizService implementation is maintained and used for cross reference. If a
 * file extension matches one found in this private list, the associated
 * recourse will be a {@link VizResource}, otherwise it will be just a regular
 * {@link ICEResource}.
 * </p>
 *
 * <p>
 * This class' methods are intended to be called by Items directly, using the
 * {@code Item.getProcess(...)} methods. method.
 * </p>
 *
 * @author Anna Wojtowicz
 */

public class ResourceHandler {

	/**
	 * Logger for handling event messages and other information.
	 */
	private static final Logger logger = LoggerFactory
			.getLogger(ResourceHandler.class);

	/**
	 * The list of file extensions that indicate a resource should be made into
	 * a {@link VizResource}.
	 */
	private static ArrayList<String> vizFileExtensions;

	/**
	 * Nullary constructor. This constructor is also responsible for initiating
	 * and constructing the {@link #vizFileExtensions} list.
	 */
	public ResourceHandler() {

		// Initiate the list of VizResource file extensions if it hasn't been
		// already
		if (vizFileExtensions == null) {
			vizFileExtensions = new ArrayList<String>(3);
		}

		return;
	}

	/**
	 * <p>
	 * This method is the star of the {@link ResourceHandler} class and does the
	 * majority of the heavy-lifting.
	 * </p>
	 *
	 * <p>
	 * Based on the filepath passed in, it will create and return a
	 * {@link VizResource} if it has a file extension found in the
	 * {@link #vizFileExtensions} list. If the file extension is not found in
	 * the list, then it will create and return a regular {@link ICEResource}.
	 * If no valid file extension was found, it will return null.
	 *
	 * @param filePath
	 *            The file path to the resource file.
	 * @return Returns a {@link VizResource} or {@link ICEResource} depending on
	 *         the file path, or null if the file path was invalid.
	 * @throws IOException
	 */
	public ICEResource getResource(String filePath) throws IOException {

		// Populate the list with all the file extensions which can be handled
		// by any viz service
		if (vizFileExtensions.size() == 0) {
			IVizServiceFactory factory = VizServiceFactoryHolder.getFactory();

			// If there was no factory, then we cannot handle visualizations for
			// any file types
			if (factory != null) {
				for (String service : factory.getServiceNames()) {
					vizFileExtensions.addAll(
							factory.get(service).getSupportedExtensions());
				}
			}
		}

		// Local declarations
		ICEResource resource = null;
		File file = null;

		// Make sure the file path is valid first
		if (!filePath.isEmpty()) {
			file = new File(filePath);
		} else {
			// If the file path is empty, complain and exit
			logger.info(
					"ResourceHandler Message: The file path was " + "empty!");

			return resource;
		}

		// Check the file path is valid
		if (file != null && file.exists()) {

			// Get the file extension of the file
			int lastDot = filePath.lastIndexOf(".");
			String fileExt = "";
			if (lastDot != -1) {
				// If a proper file extension is found, grab it (not including
				// the dot)
				fileExt = filePath.substring(lastDot + 1);
			} else {
				// If no proper file extension was found, complain and exit
				logger.info("ResourceHandler Message: The file path "
						+ "does not have a valid file extension: " + filePath);

				return resource;
			}

			// Use the file extension to determine what kind of ICE resource
			// to create
			if (vizFileExtensions.contains(fileExt.toLowerCase())) {

				// If the file extension was found in the vizFileExtensions
				// list, create a VizResource object.
				resource = new VizResource(file);

			} else {

				// If the file extension wasn't found in the vizFileExtensions
				// list, create a regular ICEResource.
				resource = new ICEResource(file);
			}

		}

		return resource;

	}

	/**
	 * This method calls {@link #getResource(String)} and then sets the
	 * resource's name.
	 *
	 * @param filePath
	 *            The file path to the resource file.
	 * @param name
	 *            The name of the resource object.
	 * @return Returns a {@link VizResource} or {@link ICEResource} depending on
	 *         the file path, or null if the file path was invalid.
	 * @throws IOException
	 */
	public ICEResource getResource(String filePath, String name)
			throws IOException {

		// Get the resource and set its name
		ICEResource resource = getResource(filePath);

		if (resource != null) {
			resource.setName(name);
		}

		return resource;
	}

	/**
	 * This method calls {@link #getResource(String, String)} and then sets the
	 * resource's ID.
	 *
	 * @param filePath
	 *            The file path to the resource file.
	 * @param name
	 *            The name of the resource object.
	 * @param id
	 *            The ID of the resource object.
	 * @return Returns a {@link VizResource} or {@link ICEResource} depending on
	 *         the file path, or null if the file path was invalid.
	 * @throws IOException
	 */
	public ICEResource getResource(String filePath, String name, int id)
			throws IOException {

		// Get the resource and set its name and ID
		ICEResource resource = getResource(filePath, name);

		if (resource != null) {
			resource.setId(id);
		}

		return resource;
	}

	/**
	 * This method calls {@link #getResource(String, String, int)} and then sets
	 * the resource's description.
	 *
	 * @param filePath
	 *            The file path to the resource file.
	 * @param name
	 *            The name of the resource object.
	 * @param id
	 *            The ID of the resource object.
	 * @param desc
	 *            The description of the resource object.
	 * @return Returns a {@link VizResource} or {@link ICEResource} depending on
	 *         the file path, or null if the file path was invalid.
	 * @throws IOException
	 */
	public ICEResource getResource(String filePath, String name, int id,
			String desc) throws IOException {

		// Get the resource and set it's name, ID and description
		ICEResource resource = getResource(filePath, name, id);

		if (resource != null) {
			resource.setDescription(desc);
		}

		return resource;
	}
}
