/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jkube.kit.build.service.docker;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Properties;
import org.apache.commons.text.StrSubstitutor;
import org.eclipse.jkube.kit.build.api.model.Container;
import org.eclipse.jkube.kit.build.service.docker.QueryService;
import org.eclipse.jkube.kit.build.service.docker.access.DockerAccess;
import org.eclipse.jkube.kit.build.service.docker.access.DockerAccessException;
import org.eclipse.jkube.kit.build.service.docker.access.log.DefaultLogCallback;
import org.eclipse.jkube.kit.build.service.docker.access.log.LogDispatcher;
import org.eclipse.jkube.kit.build.service.docker.access.log.LogOutputSpec;
import org.eclipse.jkube.kit.build.service.docker.wait.ExitCodeChecker;
import org.eclipse.jkube.kit.build.service.docker.wait.HealthCheckChecker;
import org.eclipse.jkube.kit.build.service.docker.wait.HttpPingChecker;
import org.eclipse.jkube.kit.build.service.docker.wait.LogWaitChecker;
import org.eclipse.jkube.kit.build.service.docker.wait.PreconditionFailedException;
import org.eclipse.jkube.kit.build.service.docker.wait.TcpPortChecker;
import org.eclipse.jkube.kit.build.service.docker.wait.WaitChecker;
import org.eclipse.jkube.kit.build.service.docker.wait.WaitTimeoutException;
import org.eclipse.jkube.kit.build.service.docker.wait.WaitUtil;
import org.eclipse.jkube.kit.common.KitLogger;
import org.eclipse.jkube.kit.config.image.ImageConfiguration;
import org.eclipse.jkube.kit.config.image.WaitConfiguration;

public class WaitService {
    private final QueryService queryService;
    private DockerAccess dockerAccess;
    private KitLogger log;

    public WaitService(DockerAccess dockerAccess, QueryService queryService, KitLogger log) {
        this.dockerAccess = dockerAccess;
        this.log = log;
        this.queryService = queryService;
    }

    public void wait(ImageConfiguration imageConfig, Properties projectProperties, String containerId) throws IOException {
        List<WaitChecker> checkers = this.prepareWaitCheckers(imageConfig, projectProperties, containerId);
        int timeout = this.getTimeOut(imageConfig);
        if (checkers.isEmpty()) {
            if (timeout > 0) {
                this.log.info("%s: Pausing for %d ms", new Object[]{imageConfig.getDescription(), timeout});
                WaitUtil.sleep(timeout);
            }
            return;
        }
        String logLine = this.extractCheckerLog(checkers);
        ContainerRunningPrecondition precondition = new ContainerRunningPrecondition(this.dockerAccess, containerId);
        try {
            long waited = WaitUtil.wait((WaitUtil.Precondition)precondition, timeout, checkers);
            this.log.info("%s: Waited %s %d ms", new Object[]{imageConfig.getDescription(), logLine, waited});
        }
        catch (WaitTimeoutException exp) {
            String desc = String.format("%s: Timeout after %d ms while waiting %s", imageConfig.getDescription(), exp.getWaited(), logLine);
            this.log.error(desc, new Object[0]);
            throw new IOException(desc);
        }
        catch (PreconditionFailedException exp) {
            String desc = String.format("%s: Container stopped with exit code %d unexpectedly after %d ms while waiting %s", imageConfig.getDescription(), precondition.getExitCode(), exp.getWaited(), logLine);
            this.log.error(desc, new Object[0]);
            throw new IOException(desc);
        }
    }

    private int getTimeOut(ImageConfiguration imageConfig) {
        WaitConfiguration wait = this.getWaitConfiguration(imageConfig);
        return wait != null && wait.getTime() != null ? wait.getTime() : 0;
    }

    private String extractCheckerLog(List<WaitChecker> checkers) {
        ArrayList<String> logOut = new ArrayList<String>();
        for (WaitChecker checker : checkers) {
            logOut.add(checker.getLogLabel());
        }
        return String.join((CharSequence)" and ", logOut);
    }

    private List<WaitChecker> prepareWaitCheckers(ImageConfiguration imageConfig, Properties projectProperties, String containerId) throws IOException {
        WaitConfiguration wait = this.getWaitConfiguration(imageConfig);
        if (wait == null) {
            return Collections.emptyList();
        }
        ArrayList<WaitChecker> checkers = new ArrayList<WaitChecker>();
        if (wait.getUrl() != null) {
            checkers.add(this.getUrlWaitChecker(imageConfig.getDescription(), projectProperties, wait));
        }
        if (wait.getLog() != null) {
            this.log.debug("LogWaitChecker: Waiting on %s", new Object[]{wait.getLog()});
            checkers.add(new LogWaitChecker(wait.getLog(), this.dockerAccess, containerId, this.log));
        }
        if (wait.getTcp() != null) {
            try {
                Container container = this.queryService.getMandatoryContainer(containerId);
                checkers.add(this.getTcpWaitChecker(container, imageConfig.getDescription(), projectProperties, wait.getTcp()));
            }
            catch (DockerAccessException e) {
                throw new IOException("Unable to access container " + containerId, e);
            }
        }
        if (wait.getHealthy() == Boolean.TRUE) {
            checkers.add(new HealthCheckChecker(this.dockerAccess, containerId, imageConfig.getDescription(), this.log));
        }
        if (wait.getExit() != null) {
            checkers.add(new ExitCodeChecker(wait.getExit(), this.queryService, containerId));
        }
        return checkers;
    }

    private WaitConfiguration getWaitConfiguration(ImageConfiguration imageConfig) {
        return imageConfig.getRunConfiguration().getWait();
    }

    private WaitChecker getUrlWaitChecker(String imageConfigDesc, Properties projectProperties, WaitConfiguration wait) {
        HttpPingChecker checker;
        String waitUrl = StrSubstitutor.replace((Object)wait.getUrl(), (Properties)projectProperties);
        WaitConfiguration.HttpConfiguration httpConfig = wait.getHttp();
        if (httpConfig != null) {
            checker = new HttpPingChecker(waitUrl, httpConfig.getMethod(), httpConfig.getStatus(), httpConfig.isAllowAllHosts());
            this.log.info("%s: Waiting on url %s with method %s for status %s.", new Object[]{imageConfigDesc, waitUrl, httpConfig.getMethod(), httpConfig.getStatus()});
        } else {
            checker = new HttpPingChecker(waitUrl);
            this.log.info("%s: Waiting on url %s.", new Object[]{imageConfigDesc, waitUrl});
        }
        return checker;
    }

    private WaitChecker getTcpWaitChecker(Container container, String imageConfigDesc, Properties projectProperties, WaitConfiguration.TcpConfiguration tcpConfig) {
        List<Object> ports = new ArrayList();
        List<Integer> portsConfigured = this.getTcpPorts(tcpConfig);
        String host = this.getTcpHost(tcpConfig, projectProperties);
        WaitConfiguration.TcpConfigMode mode = this.getTcpMode(tcpConfig, host);
        if (mode == WaitConfiguration.TcpConfigMode.mapped) {
            for (int port : portsConfigured) {
                Container.PortBinding binding = (Container.PortBinding)container.getPortBindings().get(port + "/tcp");
                if (binding == null) {
                    throw new IllegalArgumentException(String.format("Cannot watch on port %d, since there is no network binding", port));
                }
                ports.add(binding.getHostPort());
            }
            this.log.info("%s: Waiting for mapped ports %s on host %s", new Object[]{imageConfigDesc, ports, host});
        } else {
            String networkMode = container.getNetworkMode();
            this.log.info("%s: Network mode: %s", new Object[]{imageConfigDesc, networkMode});
            if (networkMode == null || networkMode.isEmpty() || "bridge".equals(networkMode)) {
                host = container.getIPAddress();
            } else if (!"host".equals(networkMode)) {
                host = (String)container.getCustomNetworkIpAddresses().get(networkMode);
            }
            ports = portsConfigured;
            this.log.info("%s: Waiting for ports %s directly on container with IP (%s).", new Object[]{imageConfigDesc, ports, host});
        }
        return new TcpPortChecker(host, ports);
    }

    private List<Integer> getTcpPorts(WaitConfiguration.TcpConfiguration tcpConfig) {
        List portsConfigured = tcpConfig.getPorts();
        if (portsConfigured == null || portsConfigured.isEmpty()) {
            throw new IllegalArgumentException("TCP wait config given but no ports to wait on");
        }
        return portsConfigured;
    }

    private WaitConfiguration.TcpConfigMode getTcpMode(WaitConfiguration.TcpConfiguration tcpConfig, String host) {
        WaitConfiguration.TcpConfigMode mode = tcpConfig.getMode();
        if (mode == null) {
            return "localhost".equals(host) ? WaitConfiguration.TcpConfigMode.direct : WaitConfiguration.TcpConfigMode.mapped;
        }
        return mode;
    }

    private String getTcpHost(WaitConfiguration.TcpConfiguration tcpConfig, Properties projectProperties) {
        String host = tcpConfig.getHost();
        if (host == null) {
            host = projectProperties.getProperty("docker.host.address");
        }
        return host;
    }

    private class ContainerRunningPrecondition
    implements WaitUtil.Precondition {
        private final String containerId;
        private final DockerAccess dockerAccess;
        private Integer exitCode;

        ContainerRunningPrecondition(DockerAccess dockerAccess, String containerId) {
            this.dockerAccess = dockerAccess;
            this.containerId = containerId;
        }

        @Override
        public boolean isOk() {
            try {
                this.exitCode = this.dockerAccess.getContainer(this.containerId).getExitCode();
                return this.exitCode == null;
            }
            catch (DockerAccessException e) {
                return false;
            }
        }

        @Override
        public void cleanup() {
            if (this.exitCode != null && WaitService.this.log.isVerboseEnabled()) {
                new LogDispatcher(this.dockerAccess).fetchContainerLog(this.containerId, LogOutputSpec.DEFAULT);
                this.dockerAccess.getLogSync(this.containerId, new DefaultLogCallback(LogOutputSpec.builder().colorString("black", true).prefix(this.containerId.substring(0, 6)).useColor(true).logStdout(true).build()));
            }
        }

        Integer getExitCode() {
            return this.exitCode;
        }
    }
}

