
package org.mandas.docker.client.messages.swarm;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link Version}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableVersion.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableVersion implements Version {
  private final @Nullable Long index;

  private ImmutableVersion(@Nullable Long index) {
    this.index = index;
  }

  /**
   * @return The value of the {@code index} attribute
   */
  @JsonProperty("Index")
  @Override
  public @Nullable Long index() {
    return index;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Version#index() index} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for index (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableVersion withIndex(@Nullable Long value) {
    if (Objects.equals(this.index, value)) return this;
    return new ImmutableVersion(value);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableVersion} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableVersion
        && equalsByValue((ImmutableVersion) another);
  }

  private boolean equalsByValue(ImmutableVersion another) {
    return Objects.equals(index, another.index);
  }

  /**
   * Computes a hash code from attributes: {@code index}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + Objects.hashCode(index);
    return h;
  }

  /**
   * Prints the immutable value {@code Version} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "Version{"
        + "index=" + index
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link Version} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable Version instance
   */
  public static ImmutableVersion copyOf(Version instance) {
    if (instance instanceof ImmutableVersion) {
      return (ImmutableVersion) instance;
    }
    return ImmutableVersion.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableVersion ImmutableVersion}.
   * <pre>
   * ImmutableVersion.builder()
   *    .index(Long | null) // nullable {@link Version#index() index}
   *    .build();
   * </pre>
   * @return A new ImmutableVersion builder
   */
  public static ImmutableVersion.Builder builder() {
    return new ImmutableVersion.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableVersion ImmutableVersion}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder {
    private Long index;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code Version} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(Version instance) {
      Objects.requireNonNull(instance, "instance");
      @Nullable Long indexValue = instance.index();
      if (indexValue != null) {
        index(indexValue);
      }
      return this;
    }

    /**
     * Initializes the value for the {@link Version#index() index} attribute.
     * @param index The value for index (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Index")
    public final Builder index(@Nullable Long index) {
      this.index = index;
      return this;
    }

    /**
     * Builds a new {@link ImmutableVersion ImmutableVersion}.
     * @return An immutable instance of Version
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableVersion build() {
      return new ImmutableVersion(index);
    }
  }
}
