/*******************************************************************************
 * Copyright (c) 2016, 2022 Red Hat Inc. and others.
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *  Mickael Istria (Red Hat Inc.) - initial implementation
 *  Michał Niewrzał (Rogue Wave Software Inc.)
 *  Rubén Porras Campo (Avaloq Evolution AG) - documentAboutToBeSaved implementation
 *******************************************************************************/
package org.eclipse.lsp4e;

import java.io.File;
import java.net.URI;
import java.util.Collections;
import java.util.ConcurrentModificationException;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import org.eclipse.core.filebuffers.IFileBuffer;
import org.eclipse.core.filesystem.EFS;
import org.eclipse.core.filesystem.IFileStore;
import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.content.IContentType;
import org.eclipse.jdt.annotation.NonNull;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.DocumentEvent;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.IDocumentListener;
import org.eclipse.jface.text.IRegion;
import org.eclipse.jface.text.ITextSelection;
import org.eclipse.jface.text.MultiTextSelection;
import org.eclipse.lsp4e.format.IFormatRegionsProvider;
import org.eclipse.lsp4e.internal.DocumentUtil;
import org.eclipse.lsp4e.operations.format.LSPFormatter;
import org.eclipse.lsp4e.ui.Messages;
import org.eclipse.lsp4j.DidChangeTextDocumentParams;
import org.eclipse.lsp4j.DidCloseTextDocumentParams;
import org.eclipse.lsp4j.DidOpenTextDocumentParams;
import org.eclipse.lsp4j.DidSaveTextDocumentParams;
import org.eclipse.lsp4j.FormattingOptions;
import org.eclipse.lsp4j.MessageParams;
import org.eclipse.lsp4j.MessageType;
import org.eclipse.lsp4j.Range;
import org.eclipse.lsp4j.ServerCapabilities;
import org.eclipse.lsp4j.TextDocumentContentChangeEvent;
import org.eclipse.lsp4j.TextDocumentIdentifier;
import org.eclipse.lsp4j.TextDocumentItem;
import org.eclipse.lsp4j.TextDocumentSaveReason;
import org.eclipse.lsp4j.TextDocumentSyncKind;
import org.eclipse.lsp4j.TextDocumentSyncOptions;
import org.eclipse.lsp4j.TextEdit;
import org.eclipse.lsp4j.VersionedTextDocumentIdentifier;
import org.eclipse.lsp4j.WillSaveTextDocumentParams;
import org.eclipse.lsp4j.jsonrpc.messages.Either;
import org.eclipse.lsp4j.services.LanguageServer;
import org.eclipse.osgi.util.NLS;
import org.osgi.framework.FrameworkUtil;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.framework.ServiceReference;

final class DocumentContentSynchronizer implements IDocumentListener {

	private final @NonNull LanguageServerWrapper languageServerWrapper;
	private final @NonNull IDocument document;
	private final @NonNull URI fileUri;
	private final TextDocumentSyncKind syncKind;

	private int version = 0;
	private DidChangeTextDocumentParams changeParams;
	private long openSaveStamp;
	private IPreferenceStore store;
	private IFormatRegionsProvider formatRegionsProvider;

	public DocumentContentSynchronizer(@NonNull LanguageServerWrapper languageServerWrapper,
			@NonNull LanguageServer languageServer,
			@NonNull IDocument document, TextDocumentSyncKind syncKind) {
		this.languageServerWrapper = languageServerWrapper;
		URI uri = LSPEclipseUtils.toUri(document);
		if (uri == null) {
			throw new NullPointerException();
		}
		this.fileUri = uri;
		try {
			IFileStore store = EFS.getStore(fileUri);
			this.openSaveStamp = store.fetchInfo().getLastModified();
		} catch (CoreException e) {
			try {
				this.openSaveStamp = new File(fileUri).lastModified();
			} catch (IllegalArgumentException iae) {
				this.openSaveStamp = 0L;
			}
		}
		this.syncKind = syncKind != null ? syncKind : TextDocumentSyncKind.Full;

		this.document = document;
		this.store = LanguageServerPlugin.getDefault().getPreferenceStore();

		// add a document buffer
		final var textDocument = new TextDocumentItem();
		textDocument.setUri(fileUri.toASCIIString());
		textDocument.setText(document.get());

		List<IContentType> contentTypes = LSPEclipseUtils.getDocumentContentTypes(this.document);

		String languageId = languageServerWrapper.getLanguageId(contentTypes.toArray(new IContentType[0]));

		IPath fromPortableString = Path.fromPortableString(this.fileUri.getPath());
		if (languageId == null) {
			languageId = fromPortableString.getFileExtension();
			if (languageId == null) {
				languageId = fromPortableString.lastSegment();
			}
		}

		textDocument.setLanguageId(languageId);
		textDocument.setVersion(++version);
		languageServer.getTextDocumentService().didOpen(new DidOpenTextDocumentParams(textDocument));
	}


	@Override
	public void documentChanged(DocumentEvent event) {
		checkEvent(event);
		if (syncKind == TextDocumentSyncKind.Full) {
			createChangeEvent(event);
		}

		if (changeParams != null) {
			final DidChangeTextDocumentParams changeParamsToSend = changeParams;
			changeParams = null;

			changeParamsToSend.getTextDocument().setVersion(++version);
			languageServerWrapper.sendNotification(ls -> ls.getTextDocumentService().didChange(changeParamsToSend));
		}
	}

	@Override
	public void documentAboutToBeChanged(DocumentEvent event) {
		checkEvent(event);
		if (syncKind == TextDocumentSyncKind.Incremental) {
			// this really needs to happen before event gets actually
			// applied, to properly compute positions
			createChangeEvent(event);
		}
	}

	/**
	 * Convert Eclipse {@link DocumentEvent} to LS according
	 * {@link TextDocumentSyncKind}. {@link TextDocumentContentChangeEventImpl}.
	 *
	 * @param event
	 *            Eclipse {@link DocumentEvent}
	 * @return true if change event is ready to be sent
	 */
	private boolean createChangeEvent(DocumentEvent event) {
		Assert.isTrue(changeParams == null);
		changeParams = new DidChangeTextDocumentParams(new VersionedTextDocumentIdentifier(),
				Collections.singletonList(new TextDocumentContentChangeEvent()));
		changeParams.getTextDocument().setUri(fileUri.toASCIIString());

		IDocument document = event.getDocument();
		TextDocumentContentChangeEvent changeEvent = null;
		TextDocumentSyncKind syncKind = getTextDocumentSyncKind();
		switch (syncKind) {
		case None:
			return false;
		case Full:
			changeParams.getContentChanges().get(0).setText(event.getDocument().get());
			break;
		case Incremental:
			changeEvent = changeParams.getContentChanges().get(0);
			String newText = event.getText();
			int offset = event.getOffset();
			int length = event.getLength();
			try {
				// try to convert the Eclipse start/end offset to LS range.
				final var range = new Range(LSPEclipseUtils.toPosition(offset, document),
						LSPEclipseUtils.toPosition(offset + length, document));
				changeEvent.setRange(range);
				changeEvent.setText(newText);
				changeEvent.setRangeLength(length);
			} catch (BadLocationException e) {
				// error while conversion (should never occur)
				// set the full document text as changes.
				changeEvent.setText(document.get());
			}
			break;
		}
		return true;
	}

	private boolean serverSupportsWillSaveWaitUntil() {
		ServerCapabilities serverCapabilities = languageServerWrapper.getServerCapabilities();
		if(serverCapabilities != null ) {
			Either<TextDocumentSyncKind, TextDocumentSyncOptions> textDocumentSync = serverCapabilities.getTextDocumentSync();
			if(textDocumentSync.isRight()) {
				TextDocumentSyncOptions saveOptions = textDocumentSync.getRight();
				return saveOptions != null && Boolean.TRUE.equals(saveOptions.getWillSaveWaitUntil());
			}
		}
		return false;
	}

	private static final String WILL_SAVE_WAIT_UNTIL_TIMEOUT__KEY = "timeout.willSaveWaitUntil"; //$NON-NLS-1$

	private static final int WILL_SAVE_WAIT_UNTIL_COUNT_THRESHOLD = 3;
	private static final Map<String, Integer> WILL_SAVE_WAIT_UNTIL_TIMEOUT_MAP = new ConcurrentHashMap<>();

	/**
	 * Converts a language server ID to the preference ID to define a timeout
	 * for willSaveWaitUntil
	 *
	 * @return language server's preference ID to define a timeout for willSaveWaitUntil
	 */
	private static @NonNull String lsToWillSaveWaitUntilTimeoutKey(String serverId) {
		return serverId + '.' + WILL_SAVE_WAIT_UNTIL_TIMEOUT__KEY;
	}

	private int lsToWillSaveWaitUntilTimeout() {
		int defaultWillSaveWaitUntilTimeoutInSeconds = 5;
		int willSaveWaitUntilTimeout = store.getInt(lsToWillSaveWaitUntilTimeoutKey(languageServerWrapper.serverDefinition.id));
		return willSaveWaitUntilTimeout != 0 ? willSaveWaitUntilTimeout : defaultWillSaveWaitUntilTimeoutInSeconds;
	}

	public void documentAboutToBeSaved() {
		if (!serverSupportsWillSaveWaitUntil()) {
			// format document if service has been provided:
			formatDocument();
			return;
		}

		final var identifier = LSPEclipseUtils.toTextDocumentIdentifier(fileUri);
		if (WILL_SAVE_WAIT_UNTIL_TIMEOUT_MAP.getOrDefault(identifier.getUri(), 0) > WILL_SAVE_WAIT_UNTIL_COUNT_THRESHOLD) {
			return;
		}

		// Use @link{TextDocumentSaveReason.Manual} as the platform does not give enough information to be accurate
		final var params = new WillSaveTextDocumentParams(identifier, TextDocumentSaveReason.Manual);


		try {
			List<TextEdit> edits = languageServerWrapper.executeImpl(ls -> ls.getTextDocumentService().willSaveWaitUntil(params))
				.get(lsToWillSaveWaitUntilTimeout(), TimeUnit.SECONDS);
			try {
				LSPEclipseUtils.applyEdits(document, edits);
			} catch (BadLocationException e) {
				LanguageServerPlugin.logError(e);
			}
		} catch (ExecutionException e) {
			LanguageServerPlugin.logError(e);
		} catch (TimeoutException e) {
			Integer timeoutCount = WILL_SAVE_WAIT_UNTIL_TIMEOUT_MAP.compute(identifier.getUri(),
					(k, v) -> v == null ? 1 : Integer.valueOf(v + 1));
			String message = timeoutCount > WILL_SAVE_WAIT_UNTIL_COUNT_THRESHOLD ?
					Messages.DocumentContentSynchronizer_TimeoutThresholdMessage:
						Messages.DocumentContentSynchronizer_TimeoutMessage;
			String boundMessage = NLS.bind(message, Integer.toString(lsToWillSaveWaitUntilTimeout()), identifier.getUri());
			ServerMessageHandler.showMessage(Messages.DocumentContentSynchronizer_OnSaveActionTimeout, new MessageParams(MessageType.Error, boundMessage));
		} catch (InterruptedException e) {
			LanguageServerPlugin.logError(e);
			Thread.currentThread().interrupt();
		}
	}

	private void formatDocument() {
		var regions = getFormatRegions();
		if (regions != null && document != null) {
			try {
				var textSelection = new MultiTextSelection(document, regions);
				var edits = requestFormatting(document, textSelection).get(lsToWillSaveWaitUntilTimeout(), TimeUnit.SECONDS);
				if (edits != null) {
					try {
						edits.apply();
					} catch (final ConcurrentModificationException ex) {
						ServerMessageHandler.showMessage(Messages.LSPFormatHandler_DiscardedFormat, new MessageParams(MessageType.Error, Messages.LSPFormatHandler_DiscardedFormatResponse));
					} catch (BadLocationException e) {
						LanguageServerPlugin.logError(e);
					}
				};
			} catch (BadLocationException | InterruptedException | ExecutionException | TimeoutException e) {
				LanguageServerPlugin.logError(e);
			}
		}
	}

	private synchronized IRegion[] getFormatRegions() {
		if (formatRegionsProvider != null) {
			return formatRegionsProvider.getFormattingRegions(document);
		}
		var serverId = "(serverDefinitionId=" + languageServerWrapper.serverDefinition.id + ")";  //$NON-NLS-1$ //$NON-NLS-2$
		var bundleContext = FrameworkUtil.getBundle(this.getClass()).getBundleContext();
		if (bundleContext != null) {
			try {
				ServiceReference<?> reference = null;
				var serviceReferences = bundleContext.getAllServiceReferences(IFormatRegionsProvider.class.getName(), serverId);
				if (serviceReferences != null) {
					reference = serviceReferences[0];
				} else {
					//Use LSP4E default implementation:
					reference = bundleContext.getServiceReference(IFormatRegionsProvider.class.getName());
				}
				if (reference != null) {
					formatRegionsProvider = (IFormatRegionsProvider) bundleContext.getService(reference);
					if (formatRegionsProvider != null) {
						return formatRegionsProvider.getFormattingRegions(document);
					}
				}
			} catch (InvalidSyntaxException e) {
				LanguageServerPlugin.logError(e);
			}
		}
		return null;
	}

	private CompletableFuture<VersionedEdits> requestFormatting(@NonNull IDocument document, @NonNull ITextSelection textSelection) throws BadLocationException {
		long modificationStamp = DocumentUtil.getDocumentModificationStamp(document);

		FormattingOptions formatOptions = LSPFormatter.getFormatOptions();
		TextDocumentIdentifier docId = new TextDocumentIdentifier(fileUri.toString());

		final ServerCapabilities capabilities = languageServerWrapper.getServerCapabilities();
		if (LSPFormatter.isDocumentRangeFormattingSupported(capabilities)
				&& !(LSPFormatter.isDocumentFormattingSupported(capabilities) && textSelection.getLength() == 0)) {
			var rangeParams = LSPFormatter.getRangeFormattingParams(document, textSelection, formatOptions, docId);
			return languageServerWrapper.executeImpl(ls -> ls.getTextDocumentService().rangeFormatting(rangeParams).thenApply(edits -> new VersionedEdits(modificationStamp, edits, document)));
		}
		var params = LSPFormatter.getFullFormatParams(formatOptions, docId);
		return languageServerWrapper.executeImpl(ls -> ls.getTextDocumentService().formatting(params).thenApply(edits -> new VersionedEdits(modificationStamp, edits, document)));
	}

	public void documentSaved(IFileBuffer buffer) {
		if (openSaveStamp >= buffer.getModificationStamp()) {
			return;
		}
		this.openSaveStamp = buffer.getModificationStamp();
		ServerCapabilities serverCapabilities = languageServerWrapper.getServerCapabilities();
		if (serverCapabilities != null) {
			Either<TextDocumentSyncKind, TextDocumentSyncOptions> textDocumentSync = serverCapabilities
					.getTextDocumentSync();
			if (textDocumentSync.isRight() && textDocumentSync.getRight().getSave() == null) {
				return;
			}
		}
		final var identifier = LSPEclipseUtils.toTextDocumentIdentifier(fileUri);
		final var params = new DidSaveTextDocumentParams(identifier, document.get());

		languageServerWrapper.sendNotification(ls -> ls.getTextDocumentService().didSave(params));

	}

	public CompletableFuture<Void> documentClosed() {
	   final var identifier = LSPEclipseUtils.toTextDocumentIdentifier(fileUri);
		WILL_SAVE_WAIT_UNTIL_TIMEOUT_MAP.remove(identifier.getUri());
		// When LS is shut down all documents are being disconnected. No need to send
		// "didClose" message to the LS that is being shut down or not yet started
		if (languageServerWrapper.isActive()) {
			final var params = new DidCloseTextDocumentParams(identifier);
			languageServerWrapper.sendNotification(ls -> ls.getTextDocumentService().didClose(params));
		}
		return CompletableFuture.completedFuture(null);
	}

	/**
	 * Returns the text document sync kind capabilities of the server and
	 * {@link TextDocumentSyncKind#Full} otherwise.
	 *
	 * @return the text document sync kind capabilities of the server and
	 *         {@link TextDocumentSyncKind#Full} otherwise.
	 */
	private TextDocumentSyncKind getTextDocumentSyncKind() {
		return syncKind;
	}

	public IDocument getDocument() {
		return this.document;
	}

	int getVersion() {
		return version;
	}

	private void checkEvent(DocumentEvent event) {
		if (this.document != event.getDocument()) {
			throw new IllegalStateException(
					"Synchronizer should apply to only a single document, which is the one it was instantiated for"); //$NON-NLS-1$
		}
	}
}
