/*******************************************************************************
 * Copyright (c) 2006 - 2008 Ecliptical Software Inc. and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Ecliptical Software Inc. - initial API and implementation
 *******************************************************************************/
package org.eclipse.emf.mint.internal.ui;

import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.mint.IMemberAnnotationListener;
import org.eclipse.emf.mint.MemberAnnotationChangedEvent;
import org.eclipse.emf.mint.MintCore;
import org.eclipse.emf.mint.internal.ui.search.JavaSearchMonitor;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.resource.ImageRegistry;
import org.eclipse.jface.viewers.AbstractTreeViewer;
import org.eclipse.jface.viewers.IContentProvider;
import org.eclipse.jface.viewers.ILazyTreeContentProvider;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.ITreePathContentProvider;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.jface.viewers.TreePath;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.ui.plugin.AbstractUIPlugin;
import org.osgi.framework.BundleContext;

public class MintUI extends AbstractUIPlugin {

	public static final String PLUGIN_ID = "org.eclipse.emf.mint.ui"; //$NON-NLS-1$

	public static final String GENERATED_COLOR = PLUGIN_ID + ".GENERATED_COLOR"; //$NON-NLS-1$

	public static final String GENERATED_NOT_COLOR = PLUGIN_ID
			+ ".GENERATED_NOT_COLOR"; //$NON-NLS-1$

	public static final String IMG_BLANK = "icons/blank.gif"; //$NON-NLS-1$

	private static MintUI instance;

	private final Collection<StructuredViewer> viewers = new HashSet<StructuredViewer>();

	private IMemberAnnotationListener filterUpdater;

	public MintUI() {
		instance = this;
	}

	public static MintUI getDefault() {
		return instance;
	}

	@Override
	public void stop(BundleContext context) throws Exception {
		viewers.clear();
		filterUpdater = null;
		JavaSearchMonitor.stop();
		super.stop(context);
	}

	@Override
	protected void initializeImageRegistry(ImageRegistry reg) {
		super.initializeImageRegistry(reg);
		reg.put(IMG_BLANK, ImageDescriptor.createFromURL(getBundle().getEntry(
				IMG_BLANK)));
	}

	public IStatus logError(String message, Throwable t) {
		if (message == null)
			message = Messages.MintUI_UnexpectedError;

		IStatus status = new Status(IStatus.ERROR, PLUGIN_ID, 0, message, t);
		getLog().log(status);
		return status;
	}

	public void installFilterUpdater(final StructuredViewer viewer) {
		synchronized (viewers) {
			if (viewers.add(viewer)) {
				createFilterUpdater();
				viewer.getControl().addDisposeListener(new DisposeListener() {
					public void widgetDisposed(DisposeEvent e) {
						synchronized (viewers) {
							viewers.remove(viewer);
						}
					}
				});
			}
		}
	}

	private void createFilterUpdater() {
		if (filterUpdater != null)
			return;

		filterUpdater = new IMemberAnnotationListener() {
			public void memberAnnotationChanged(
					final MemberAnnotationChangedEvent event) {
				getWorkbench().getDisplay().asyncExec(new Runnable() {
					public void run() {
						handleChanges(event.getChanges().keySet());
					}
				});
			}
		};

		MintCore.getInstance().getMemberAnnotationManager()
				.addMemberAnnotationListener(filterUpdater);
	}

	private void handleChanges(Collection<?> changes) {
		synchronized (viewers) {
			for (StructuredViewer viewer : viewers) {
				handleChanges(viewer, changes);
			}
		}
	}

	private void handleChanges(StructuredViewer viewer, Collection<?> changes) {
		IContentProvider cp = viewer.getContentProvider();
		if (viewer instanceof AbstractTreeViewer) {
			Collection<Object> parents = new HashSet<Object>();
			if (cp instanceof ITreeContentProvider) {
				ITreeContentProvider tcp = (ITreeContentProvider) cp;
				for (Object element : changes) {
					Object parent = tcp.getParent(element);
					if (parent != null)
						parents.add(parent);
				}
			} else if (cp instanceof ITreePathContentProvider) {
				ITreePathContentProvider tpcp = (ITreePathContentProvider) cp;
				for (Object element : changes) {
					TreePath[] parentPaths = tpcp.getParents(element);
					for (int k = 0; k < parentPaths.length; ++k)
						parents.add(parentPaths[k].getLastSegment());
				}
			} else if (cp instanceof ILazyTreeContentProvider) {
				ILazyTreeContentProvider ltcp = (ILazyTreeContentProvider) cp;
				for (Object element : changes) {
					Object parent = ltcp.getParent(element);
					if (parent != null)
						parents.add(parent);
				}
			}

			for (Object parent : parents)
				viewer.refresh(parent);

			return;
		}

		if (cp instanceof IStructuredContentProvider) {
			Object[] elements = ((IStructuredContentProvider) cp)
					.getElements(viewer.getInput());
			Collection<?> c = Arrays.asList(elements);
			for (Object element : changes) {
				if (c.contains(element)) {
					viewer.refresh();
					break;
				}
			}
		}
	}
}
