/**
 * Copyright (c) 2014, 2023 CEA LIST.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *     CEA LIST - initial API and implementation
 *     Ansgar Radermacher - Bug 581531 - handle return parameter multiplicity
 */
package org.eclipse.papyrus.designer.languages.cpp.codegen.xtend;

import com.google.common.base.Objects;
import java.util.Collection;
import org.eclipse.emf.common.util.EList;
import org.eclipse.papyrus.designer.languages.common.base.GenUtils;
import org.eclipse.papyrus.designer.languages.cpp.codegen.Constants;
import org.eclipse.papyrus.designer.languages.cpp.codegen.utils.CppClassUtils;
import org.eclipse.papyrus.designer.languages.cpp.codegen.utils.CppGenUtils;
import org.eclipse.papyrus.designer.languages.cpp.codegen.utils.Modifier;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.ConstInit;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Default;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Delete;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Explicit;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Inline;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Variadic;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Virtual;
import org.eclipse.papyrus.designer.uml.tools.utils.StereotypeUtil;
import org.eclipse.uml2.uml.Behavior;
import org.eclipse.uml2.uml.Classifier;
import org.eclipse.uml2.uml.DataType;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.Interface;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.OpaqueBehavior;
import org.eclipse.uml2.uml.Operation;
import org.eclipse.uml2.uml.Parameter;
import org.eclipse.uml2.uml.ParameterDirectionKind;
import org.eclipse.uml2.uml.Region;
import org.eclipse.uml2.uml.Type;
import org.eclipse.uml2.uml.profile.standard.Create;
import org.eclipse.uml2.uml.profile.standard.Destroy;
import org.eclipse.uml2.uml.util.UMLUtil;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;

@SuppressWarnings("all")
public class CppOperations {
  public static CharSequence CppOperationImplementation(final Operation operation) {
    StringConcatenation _builder = new StringConcatenation();
    CharSequence _CppOperationDoc = CppDocumentation.CppOperationDoc(operation);
    _builder.append(_CppOperationDoc);
    _builder.newLineIfNotEmpty();
    {
      String _name = operation.getName();
      boolean _equals = Objects.equal(_name, "main");
      if (_equals) {
        String _CppReturnSpec = CppOperations.CppReturnSpec(operation);
        _builder.append(_CppReturnSpec);
        String _name_1 = operation.getName();
        _builder.append(_name_1);
        _builder.append("(");
        CharSequence _CppOperationParameters = CppParameter.CppOperationParameters(operation, false);
        _builder.append(_CppOperationParameters);
        _builder.append(") {");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        String _body = GenUtils.getBody(operation, Constants.supportedLanguages);
        _builder.append(_body, "\t");
        _builder.newLineIfNotEmpty();
        _builder.append("} ");
        _builder.newLine();
      } else {
        CharSequence _templateSignature = CppTemplates.templateSignature(operation);
        _builder.append(_templateSignature);
        String _InlineTxt = CppOperations.InlineTxt(operation);
        _builder.append(_InlineTxt);
        String _modCEQualifier = Modifier.modCEQualifier(operation);
        _builder.append(_modCEQualifier);
        String _CppReturnSpec_1 = CppOperations.CppReturnSpec(operation);
        _builder.append(_CppReturnSpec_1);
        String _nestedOperationFarthestClassifierOwnerNamespace = GenUtils.getNestedOperationFarthestClassifierOwnerNamespace(operation);
        _builder.append(_nestedOperationFarthestClassifierOwnerNamespace);
        CharSequence _templateShortSignature = CppTemplates.templateShortSignature(operation);
        _builder.append(_templateShortSignature);
        _builder.append("::");
        String _destructor = CppOperations.destructor(operation);
        _builder.append(_destructor);
        String _name_2 = operation.getName();
        _builder.append(_name_2);
        _builder.append("(");
        CharSequence _CppOperationParameters_1 = CppParameter.CppOperationParameters(operation, false);
        _builder.append(_CppOperationParameters_1);
        String _variadicParameter = CppOperations.variadicParameter(operation);
        _builder.append(_variadicParameter);
        _builder.append(")");
        String _modCVQualifier = Modifier.modCVQualifier(operation);
        _builder.append(_modCVQualifier);
        CharSequence _throwss = CppOperations.throwss(operation);
        _builder.append(_throwss);
        String _CppConstInit = CppOperations.CppConstInit(operation);
        _builder.append(_CppConstInit);
        _builder.append(" {");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        String _body_1 = GenUtils.getBody(operation, Constants.supportedLanguages);
        _builder.append(_body_1, "\t");
        _builder.newLineIfNotEmpty();
        _builder.append("}");
        _builder.newLine();
      }
    }
    return _builder;
  }

  public static String CppReturnSpec(final Operation operation) {
    String _xifexpression = null;
    if (((operation.getType() == null) || CppOperations.isConsOrDestructor(operation))) {
      _xifexpression = CppOperations.ConsDestructorOrVoid(operation);
    } else {
      String _xblockexpression = null;
      {
        final Parameter retParam = operation.getReturnResult();
        String _modCVQualifier = Modifier.modCVQualifier(retParam);
        String _modSCQualifier = Modifier.modSCQualifier(retParam);
        String _plus = (_modCVQualifier + _modSCQualifier);
        String _cppType = CppTypedElement.cppType(retParam);
        String _plus_1 = (_plus + _cppType);
        String _modPtr = Modifier.modPtr(retParam);
        String _plus_2 = (_plus_1 + _modPtr);
        String _modRef = Modifier.modRef(retParam);
        String _plus_3 = (_plus_2 + _modRef);
        String _modArray = Modifier.modArray(retParam);
        String _plus_4 = (_plus_3 + _modArray);
        _xblockexpression = (_plus_4 + " ");
      }
      _xifexpression = _xblockexpression;
    }
    return _xifexpression;
  }

  public static CharSequence CppReturnSpec(final Behavior behavior) {
    StringConcatenation _builder = new StringConcatenation();
    {
      Parameter _returnResult = GenUtils.returnResult(behavior);
      boolean _tripleEquals = (_returnResult == null);
      if (_tripleEquals) {
        _builder.append("void ");
      } else {
        String _modCVQualifier = Modifier.modCVQualifier(GenUtils.returnResult(behavior));
        _builder.append(_modCVQualifier);
        _builder.append(" ");
        String _cppQualifiedName = CppGenUtils.cgu(behavior).cppQualifiedName(GenUtils.returnResult(behavior).getType());
        _builder.append(_cppQualifiedName);
        String _modPtr = Modifier.modPtr(GenUtils.returnResult(behavior));
        _builder.append(_modPtr);
        String _modRef = Modifier.modRef(GenUtils.returnResult(behavior));
        _builder.append(_modRef);
        _builder.append(" ");
      }
    }
    _builder.newLineIfNotEmpty();
    return _builder;
  }

  public static CharSequence throwss(final Operation operation) {
    StringConcatenation _builder = new StringConcatenation();
    {
      int _length = ((Object[])Conversions.unwrapArray(operation.getRaisedExceptions(), Object.class)).length;
      boolean _greaterThan = (_length > 0);
      if (_greaterThan) {
        _builder.append(" ");
        _builder.append("throw(");
        {
          EList<Type> _raisedExceptions = operation.getRaisedExceptions();
          boolean _hasElements = false;
          for(final Type re : _raisedExceptions) {
            if (!_hasElements) {
              _hasElements = true;
            } else {
              _builder.appendImmediate(",", "");
            }
            String _cppQualifiedName = CppGenUtils.cgu(operation).cppQualifiedName(re);
            _builder.append(_cppQualifiedName);
          }
        }
        _builder.append(")");
      }
    }
    return _builder;
  }

  public static String ConsDestructorOrVoid(final Operation operation) {
    String _xifexpression = null;
    boolean _isConsOrDestructor = CppOperations.isConsOrDestructor(operation);
    if (_isConsOrDestructor) {
      _xifexpression = null;
    } else {
      _xifexpression = "void ";
    }
    return _xifexpression;
  }

  public static boolean isConsOrDestructor(final Operation operation) {
    return (GenUtils.hasStereotype(operation, Create.class) || GenUtils.hasStereotype(operation, Destroy.class));
  }

  public static String CppConstInit(final Operation operation) {
    String _xifexpression = null;
    if ((GenUtils.hasStereotype(operation, ConstInit.class) && GenUtils.hasStereotype(operation, Create.class))) {
      String _initialisation = UMLUtil.<ConstInit>getStereotypeApplication(operation, ConstInit.class).getInitialisation();
      _xifexpression = (": " + _initialisation);
    }
    return _xifexpression;
  }

  public static Collection<Operation> getOwnedOperations(final Classifier cl) {
    Collection<Operation> _xblockexpression = null;
    {
      final EList<Operation> operations = CppOperations.getOwnedOperationsWNull(cl);
      Collection<Operation> _xifexpression = null;
      if ((operations == null)) {
        _xifexpression = CollectionLiterals.<Operation>emptySet();
      } else {
        _xifexpression = operations;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }

  public static EList<Operation> getOwnedOperationsWNull(final Classifier cl) {
    EList<Operation> _xifexpression = null;
    if ((cl instanceof org.eclipse.uml2.uml.Class)) {
      _xifexpression = ((org.eclipse.uml2.uml.Class) cl).getOwnedOperations();
    } else {
      EList<Operation> _xifexpression_1 = null;
      if ((cl instanceof DataType)) {
        _xifexpression_1 = ((DataType) cl).getOwnedOperations();
      } else {
        EList<Operation> _xifexpression_2 = null;
        if ((cl instanceof Interface)) {
          _xifexpression_2 = ((Interface) cl).getOwnedOperations();
        } else {
          _xifexpression_2 = null;
        }
        _xifexpression_1 = _xifexpression_2;
      }
      _xifexpression = _xifexpression_1;
    }
    return _xifexpression;
  }

  public static Collection<Operation> getNestedOperations(final Classifier c1) {
    Collection<Operation> _xblockexpression = null;
    {
      final EList<Operation> operations = CppOperations.getNestedOperationsWNull(c1);
      Collection<Operation> _xifexpression = null;
      if ((operations == null)) {
        _xifexpression = CollectionLiterals.<Operation>emptySet();
      } else {
        _xifexpression = operations;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }

  public static EList<Operation> getNestedOperationsWNull(final Classifier cl) {
    EList<Operation> _xifexpression = null;
    if (((cl instanceof org.eclipse.uml2.uml.Class) || (cl instanceof Interface))) {
      _xifexpression = CppClassUtils.nestedOperations(cl);
    }
    return _xifexpression;
  }

  public static CharSequence CppBehaviorImplementation(final OpaqueBehavior behavior) {
    StringConcatenation _builder = new StringConcatenation();
    CharSequence _CppBehaviorDoc = CppDocumentation.CppBehaviorDoc(behavior);
    _builder.append(_CppBehaviorDoc);
    _builder.newLineIfNotEmpty();
    CharSequence _CppReturnSpec = CppOperations.CppReturnSpec(behavior);
    _builder.append(_CppReturnSpec);
    String _nestedBehaviorFarthestClassifierOwnerNamespace = GenUtils.getNestedBehaviorFarthestClassifierOwnerNamespace(behavior);
    _builder.append(_nestedBehaviorFarthestClassifierOwnerNamespace);
    _builder.append("::");
    String _qualifiedBehaviorName = CppOperations.qualifiedBehaviorName(behavior);
    _builder.append(_qualifiedBehaviorName);
    _builder.append("(");
    CharSequence _CppBehaviorParameters = CppParameter.CppBehaviorParameters(behavior, false);
    _builder.append(_CppBehaviorParameters);
    _builder.append(")");
    String _modCVQualifier = Modifier.modCVQualifier(behavior);
    _builder.append(_modCVQualifier);
    _builder.append(" {");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    String _bodyFromOB = GenUtils.getBodyFromOB(behavior, Constants.supportedLanguages);
    _builder.append(_bodyFromOB, "\t");
    _builder.newLineIfNotEmpty();
    _builder.append("}");
    _builder.newLine();
    return _builder;
  }

  public static CharSequence CppOperationDeclaration(final Operation operation) {
    StringConcatenation _builder = new StringConcatenation();
    CharSequence _CppOperationDoc = CppDocumentation.CppOperationDoc(operation);
    _builder.append(_CppOperationDoc);
    _builder.newLineIfNotEmpty();
    String _explicitTxt = CppOperations.explicitTxt(operation);
    _builder.append(_explicitTxt);
    String _modCEQualifier = Modifier.modCEQualifier(operation);
    _builder.append(_modCEQualifier);
    String _InlineTxt = CppOperations.InlineTxt(operation);
    _builder.append(_InlineTxt);
    String _virtualTxt = CppOperations.virtualTxt(operation);
    _builder.append(_virtualTxt);
    String _staticTxt = CppOperations.staticTxt(operation);
    _builder.append(_staticTxt);
    String _CppReturnSpec = CppOperations.CppReturnSpec(operation);
    _builder.append(_CppReturnSpec);
    String _destructor = CppOperations.destructor(operation);
    _builder.append(_destructor);
    String _name = operation.getName();
    _builder.append(_name);
    _builder.append("(");
    CharSequence _CppOperationParameters = CppParameter.CppOperationParameters(operation, true);
    _builder.append(_CppOperationParameters);
    String _variadicParameter = CppOperations.variadicParameter(operation);
    _builder.append(_variadicParameter);
    _builder.append(")");
    String _modCVQualifier = Modifier.modCVQualifier(operation);
    _builder.append(_modCVQualifier);
    CharSequence _throwss = CppOperations.throwss(operation);
    _builder.append(_throwss);
    String _virtualSuffix = CppOperations.virtualSuffix(operation);
    _builder.append(_virtualSuffix);
    _builder.append(";");
    _builder.newLineIfNotEmpty();
    return _builder;
  }

  public static String InlineTxt(final Element element) {
    String _xifexpression = null;
    boolean _hasStereotype = GenUtils.hasStereotype(element, Inline.class);
    if (_hasStereotype) {
      _xifexpression = "inline ";
    }
    return _xifexpression;
  }

  public static String explicitTxt(final Element element) {
    String _xifexpression = null;
    boolean _hasStereotype = GenUtils.hasStereotype(element, Explicit.class);
    if (_hasStereotype) {
      _xifexpression = "explicit ";
    }
    return _xifexpression;
  }

  public static String virtualTxt(final Operation operation) {
    String _xifexpression = null;
    if ((((operation.getInterface() != null) || operation.isAbstract()) || GenUtils.hasStereotype(operation, Virtual.class))) {
      _xifexpression = "virtual ";
    }
    return _xifexpression;
  }

  public static String staticTxt(final Operation operation) {
    String _xifexpression = null;
    boolean _isStatic = operation.isStatic();
    if (_isStatic) {
      _xifexpression = "static ";
    }
    return _xifexpression;
  }

  public static String destructor(final Operation operation) {
    String _xifexpression = null;
    if ((GenUtils.hasStereotype(operation, Destroy.class) && (!operation.getName().startsWith("~")))) {
      _xifexpression = "~";
    } else {
      _xifexpression = "";
    }
    return _xifexpression;
  }

  /**
   * Common method to handle virtual, default and deleted operations
   */
  public static String virtualSuffix(final Operation operation) {
    String _xifexpression = null;
    if (((operation.getInterface() != null) || operation.isAbstract())) {
      _xifexpression = " = 0";
    } else {
      String _xifexpression_1 = null;
      boolean _isApplied = StereotypeUtil.isApplied(operation, Delete.class);
      if (_isApplied) {
        _xifexpression_1 = " = delete";
      } else {
        String _xifexpression_2 = null;
        boolean _isApplied_1 = StereotypeUtil.isApplied(operation, 
          Default.class);
        if (_isApplied_1) {
          _xifexpression_2 = " = default";
        }
        _xifexpression_1 = _xifexpression_2;
      }
      _xifexpression = _xifexpression_1;
    }
    return _xifexpression;
  }

  public static CharSequence CppBehaviorDeclaration(final Classifier cl, final Behavior behavior) {
    StringConcatenation _builder = new StringConcatenation();
    CharSequence _CppBehaviorDoc = CppDocumentation.CppBehaviorDoc(behavior);
    _builder.append(_CppBehaviorDoc);
    _builder.newLineIfNotEmpty();
    String _InlineTxt = CppOperations.InlineTxt(behavior);
    _builder.append(_InlineTxt);
    CharSequence _CppReturnSpec = CppOperations.CppReturnSpec(behavior);
    _builder.append(_CppReturnSpec);
    String _qualifiedBehaviorName = CppOperations.qualifiedBehaviorName(behavior);
    _builder.append(_qualifiedBehaviorName);
    _builder.append("(");
    CharSequence _CppBehaviorParameters = CppParameter.CppBehaviorParameters(behavior, true);
    _builder.append(_CppBehaviorParameters);
    _builder.append(")");
    String _modCVQualifier = Modifier.modCVQualifier(behavior);
    _builder.append(_modCVQualifier);
    _builder.append(";");
    _builder.newLineIfNotEmpty();
    return _builder;
  }

  public static String qualifiedBehaviorName(final Behavior behavior) {
    NamedElement ne = ((NamedElement) behavior);
    String name = behavior.getName();
    while ((((ne != null) && (!(ne instanceof Classifier))) || (ne instanceof Behavior))) {
      {
        Element _owner = ne.getOwner();
        if ((_owner instanceof NamedElement)) {
          Element _owner_1 = ne.getOwner();
          ne = ((NamedElement) _owner_1);
        }
        Element _owner_2 = ne.getOwner();
        if ((_owner_2 instanceof Region)) {
          String _name = ne.getName();
          String _plus = (_name + "_");
          String _plus_1 = (_plus + name);
          name = _plus_1;
        }
      }
    }
    return name;
  }

  public static String variadicParameter(final Operation operation) {
    String _xblockexpression = null;
    {
      boolean hasParameters = false;
      int i = 0;
      String _xifexpression = null;
      boolean _hasStereotype = GenUtils.hasStereotype(operation, Variadic.class);
      if (_hasStereotype) {
        String _xblockexpression_1 = null;
        {
          while (((i < operation.getOwnedParameters().size()) && (!hasParameters))) {
            {
              ParameterDirectionKind _direction = operation.getOwnedParameters().get(i).getDirection();
              boolean _notEquals = (!Objects.equal(_direction, ParameterDirectionKind.RETURN_LITERAL));
              if (_notEquals) {
                hasParameters = true;
              }
              i++;
            }
          }
          String _xifexpression_1 = null;
          if (hasParameters) {
            _xifexpression_1 = ", ...";
          } else {
            _xifexpression_1 = "...";
          }
          _xblockexpression_1 = _xifexpression_1;
        }
        _xifexpression = _xblockexpression_1;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
}
