/**
 * Copyright (c) 2022 CEA LIST
 * 
 * All rights reserved. This program and the accompanying materials are
 * made available under the terms of the Eclipse License 2.0 which
 * accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *   Ansgar Radermacher - Initial API and implementation
 *   Mohamed Harkat
 */
package org.eclipse.papyrus.designer.languages.python.codegen.transformation;

import java.util.ArrayList;
import org.eclipse.emf.common.util.EList;
import org.eclipse.papyrus.designer.infra.base.StringConstants;
import org.eclipse.papyrus.designer.languages.common.base.ClassUtils;
import org.eclipse.papyrus.designer.languages.common.base.GenUtils;
import org.eclipse.papyrus.designer.languages.common.profile.Codegen.External;
import org.eclipse.papyrus.designer.uml.tools.utils.StereotypeUtil;
import org.eclipse.uml2.uml.Classifier;
import org.eclipse.uml2.uml.Dependency;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.Namespace;
import org.eclipse.uml2.uml.util.UMLUtil;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * Utility method for Python code generation
 */
@SuppressWarnings("all")
public class PythonCodeGenUtils {
  /**
   * A method that returns the qualified name of the element in form of a python
   * pathname (lowercase, dot as path separator).
   */
  public static String getPythonQName(final NamedElement element) {
    final External external = UMLUtil.<External>getStereotypeApplication(element, External.class);
    if (((external != null) && (external.getName() != null))) {
      return external.getName();
    } else {
      return element.getQualifiedName().toLowerCase().replace(Namespace.SEPARATOR, StringConstants.DOT);
    }
  }

  /**
   * Get the relative path of the target Classifier to the source classifier
   * (The classifier type was chosen here because classes, interfaces
   * enumerations and datatypes are all specializations of the classifier).
   * 
   * @param src the importing classifier
   * @param target the element to import
   */
  public static String getRelativePath(final Classifier src, final NamedElement target) {
    String dot = StringConstants.DOT;
    Namespace ns = src.getNamespace();
    while (((!target.allNamespaces().contains(ns)) && (ns != null))) {
      {
        String _dot = dot;
        dot = (_dot + StringConstants.DOT);
        ns = ns.getNamespace();
      }
    }
    if ((ns != null)) {
      String _pythonQName = PythonCodeGenUtils.getPythonQName(target);
      String _pythonQName_1 = PythonCodeGenUtils.getPythonQName(ns);
      String _plus = (_pythonQName_1 + StringConstants.DOT);
      String _replaceFirst = _pythonQName.replaceFirst(_plus, "");
      return (dot + _replaceFirst);
    } else {
      return PythonCodeGenUtils.getPythonQName(target);
    }
  }

  /**
   * Returns an array list which contains all the packages that the classifier depends on.
   */
  public static ArrayList<org.eclipse.uml2.uml.Package> getPackageDep(final Classifier src) {
    ArrayList<org.eclipse.uml2.uml.Package> packDp = CollectionLiterals.<org.eclipse.uml2.uml.Package>newArrayList();
    EList<Dependency> _clientDependencies = src.getClientDependencies();
    for (final Dependency dependency : _clientDependencies) {
      final Function1<Element, Boolean> _function = (Element it) -> {
        return Boolean.valueOf((it instanceof org.eclipse.uml2.uml.Package));
      };
      Iterable<Element> _filter = IterableExtensions.<Element>filter(dependency.getTargets(), _function);
      for (final Element target : _filter) {
        packDp.add(((org.eclipse.uml2.uml.Package) target));
      }
    }
    return packDp;
  }

  public static Iterable<Classifier> pyRequiredClassifiers(final Classifier classifier) {
    final Function1<Classifier, Boolean> _function = (Classifier it) -> {
      return Boolean.valueOf(((!GenUtils.isUMLPrimitiveType(it)) && (!PythonCodeGenUtils.isPythonPrimitiveType(it))));
    };
    return IterableExtensions.<Classifier>filter(ClassUtils.requiredClassifiers(classifier), _function);
  }

  public static boolean isPythonPrimitiveType(final Classifier type) {
    final String owningPkgName = type.getNearestPackage().getName();
    return "PythonStdLib".equals(owningPkgName);
  }

  /**
   * Write all the necessary importations for the classifier for all the relations.
   */
  public static CharSequence writeImports(final Classifier classifier, final boolean useRelativeImports) {
    StringConcatenation _builder = new StringConcatenation();
    {
      boolean _isAbstract = classifier.isAbstract();
      if (_isAbstract) {
        _builder.append("from abc import ABC, abstractmethod");
        _builder.newLine();
      }
    }
    {
      ArrayList<org.eclipse.uml2.uml.Package> _packageDep = PythonCodeGenUtils.getPackageDep(classifier);
      for(final org.eclipse.uml2.uml.Package pkg : _packageDep) {
        _builder.append("import ");
        String _pythonQName = PythonCodeGenUtils.getPythonQName(pkg);
        _builder.append(_pythonQName);
        _builder.newLineIfNotEmpty();
      }
    }
    {
      Iterable<Classifier> _pyRequiredClassifiers = PythonCodeGenUtils.pyRequiredClassifiers(classifier);
      for(final Classifier dependencyClassifier : _pyRequiredClassifiers) {
        final NamedElement mp = PythonCodeGenUtils.modulePkg(dependencyClassifier);
        _builder.newLineIfNotEmpty();
        final org.eclipse.papyrus.designer.languages.python.profile.python.External extPython = UMLUtil.<org.eclipse.papyrus.designer.languages.python.profile.python.External>getStereotypeApplication(dependencyClassifier, org.eclipse.papyrus.designer.languages.python.profile.python.External.class);
        _builder.newLineIfNotEmpty();
        final External extCommon = UMLUtil.<External>getStereotypeApplication(dependencyClassifier, External.class);
        _builder.newLineIfNotEmpty();
        {
          if ((extPython != null)) {
            String _externalImport = PythonCodeGenUtils.externalImport(extPython);
            _builder.append(_externalImport);
            _builder.newLineIfNotEmpty();
          } else {
            if ((extCommon != null)) {
              String _externalImport_1 = PythonCodeGenUtils.externalImport(extCommon);
              _builder.append(_externalImport_1);
              _builder.newLineIfNotEmpty();
            } else {
              if (useRelativeImports) {
                _builder.append("from ");
                String _relativePath = PythonCodeGenUtils.getRelativePath(classifier, mp);
                _builder.append(_relativePath);
                _builder.append(" import ");
                String _name = dependencyClassifier.getName();
                _builder.append(_name);
                _builder.newLineIfNotEmpty();
              } else {
                _builder.append("from ");
                String _pythonQName_1 = PythonCodeGenUtils.getPythonQName(mp);
                _builder.append(_pythonQName_1);
                _builder.append(" import ");
                String _name_1 = dependencyClassifier.getName();
                _builder.append(_name_1);
                _builder.newLineIfNotEmpty();
              }
            }
          }
        }
      }
    }
    return _builder;
  }

  /**
   * Import an external classifier, always assuming that it is in a module
   */
  public static String externalImport(final External external) {
    String name = external.getName();
    if ((name == null)) {
      name = PythonCodeGenUtils.getPythonQName(external.getBase_Classifier());
    }
    name = name.replace(Namespace.SEPARATOR, StringConstants.DOT);
    final int index = name.lastIndexOf(StringConstants.DOT);
    if ((index != (-1))) {
      final String from = name.substring(0, index).toLowerCase();
      final String shortName = name.substring((index + 1));
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("from ");
      _builder.append(from);
      _builder.append(" import ");
      _builder.append(shortName);
      return _builder.toString();
    } else {
      return "";
    }
  }

  /**
   * Module = multiple classifiers in same file (which has the name of the package)
   * If a classifier is defined in a "module" package, return this package, otherwise
   * return the classifier itself.
   */
  public static NamedElement modulePkg(final NamedElement ne) {
    boolean _isApplied = StereotypeUtil.isApplied(ne.getNearestPackage(), org.eclipse.papyrus.designer.languages.python.profile.python.Module.class);
    if (_isApplied) {
      return ne.getNearestPackage();
    }
    return ne;
  }
}
