/*******************************************************************************
 * Copyright (c) 2022 DFKI.
 *
 * This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     DFKI - Tapanta Bhanja <tapanta.bhanja@dfki.de>
 *******************************************************************************/
package org.eclipse.aas.basyx.codegen.generator.submodel.submodelelements;

import java.util.List;

import org.eclipse.aas.api.reference.Key;
import org.eclipse.aas.api.submodel.parts.ConceptDescription;
import org.eclipse.aas.api.submodel.submodelelement.SubModelElementCollection;
import org.eclipse.aas.api.submodel.submodelelement.dataelement.File;
import org.eclipse.aas.basyx.codegen.generator.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileGenerator {
	
	private static final Logger logger = LoggerFactory.getLogger(FileGenerator.class);
	private File fileInstance;
	private List<Key> semanticKeys;
	
	private String parent;
	private ConceptDescription conceptDesc;
	
	/**
	 * Constructor for the FileGenerator Class. It receives a single instance
	 * of File as parameter to generate its respective and necessary BaSyx Code. 
	 * 
	 * @param fileInstance
	 */
	public FileGenerator(File fileInstance) {
		
		this.fileInstance = fileInstance;
		
		try {
			this.semanticKeys = fileInstance.getSemanticIdentifier().getKeys();
		}
		
		catch (NullPointerException e) {
			logger.error("Null Pointer Exception in Semantic ID Declaration "
					+ "while initialising " + this.getClass().getSimpleName());
		}
		
		try {
			this.conceptDesc = this.fileInstance.getSemanticDescription();
		}
		
		catch (NullPointerException e) {
			logger.error("Null Pointer Exception while fetching ConceptDescription for " + 
							"the semantic Id definition of the File: " + 
												this.getClass().getSimpleName());
		}
		
		// Checking for Parents.
		if (fileInstance.getParentSEC()!=null) {
			this.parent = fileInstance.getParentSEC().getIdShort();
		}
		
		else if (fileInstance.getParentSub()!=null) {
			this.parent = fileInstance.getParentSub().getIdShort();
		}
		
		else {
			logger.error("File: " + fileInstance.getIdShort() 
						+ "has no Parent defined");
		}
		
		logger.info("FileGenerator Initialised for File : " + this.fileInstance.getIdShort());
	}
	
	/**
	 * Generates a {@code File} either in a SubModel or inside it's parent 
	 * SubModelElementCollection.
	 * This function generates:
	 * 
	 * 	-	Semantic Reference for the {@code File} in consideration.
	 * 	-	Modeling Info for the {@code File} in consideration. 
	 * 	-	Parental Information for the {@code File} in consideration.
	 * 	-	MimeType Information for the {@code File} in consideration.
	 * 	-	Value Information for the {@code File} in consideration.
	 * 
	 * @return
	 */
	public String generateFile() {
		
		String fileText = ""
				+ "		File " + parent + "_" + fileInstance.getIdShort() + "= new File();\r\n"
				+ "		" + parent + "_" + fileInstance.getIdShort() + ".setIdShort(\"" + fileInstance.getIdShort() +"\");\r\n"
				+ generateMimeType()
				+ generateModelingInfo()
				+ generateSemanticReference()
				+ generateParentalRelation()
				+ generateValueDelegates()
				+ "\r\n\r\n";
		
		logger.info("File Code generated for File: " +
				fileInstance.getIdShort());
		
		return fileText;
	}
	
	/**
	 * Generates the Modeling Info for the File. 
	 * 
	 * @return	Returns a String with generated code containing the Modeling Info. 
	 */
	private String generateModelingInfo() {
		
		String setKind = "";
		
		if (fileInstance.getKind()!=null) {
			setKind = "		" + parent + "_" + fileInstance.getIdShort() + 
					".setKind(ModelingKind." + fileInstance.getKind() + ");\r\n";
			
			logger.info("Generated ModelingKind Info for File: " + 
					fileInstance.getIdShort());
		}
		
		else {
			logger.info("Generated ModelingKind Info for File: " + 
					fileInstance.getIdShort() + " not generated.");
		}
		
		return setKind;
		
	}
	
	/**
	 * Generates the Parental Relation for the File. 
	 * 
	 * @return	Returns a String with generated code containing the parental 
	 * 			information for the File. 
	 */
	private String generateParentalRelation() {
		
		// Thread Check: To Understand who has called the function.
		// Test out this function of stackTrace.
		StackTraceElement[] stackTraceElements = Thread.currentThread().getStackTrace();
		boolean isSubModelFile = true;
				
		if(stackTraceElements[3].getMethodName() == "generateSubModelElementCollection") {
			isSubModelFile = false;
		}
		
		// Defines where the Property belongs to. 
		String addFile = "";
		if (isSubModelFile) {
			addFile = "		addSubmodelElement(" + parent + "_" + fileInstance.getIdShort() + ");\r\n";
			
			logger.info("File " + fileInstance.getIdShort() 
									+ " added to Submodel: " 
									+ fileInstance.getParentSub().getIdShort());
		}
		else {
			SubModelElementCollection parentSEC = fileInstance.getParentSEC();
			String immediateParent = "";
			
			// Checking for Parents.
			if (parentSEC.getParentSEC()!=null) {
				immediateParent = parentSEC.getParentSEC().getIdShort();
			}
			
			else if (parentSEC.getParentSub()!=null) {
				immediateParent = parentSEC.getParentSub().getIdShort();
			}
			
			else {
				logger.error("File: " + parentSEC.getIdShort() 
				+ "has no Parent defined");
			}
			
			addFile = "		" + immediateParent + "_" + parentSEC.getIdShort() + "value.add(" 
								+ parent + "_" + fileInstance.getIdShort() + ");\r\n\r\n"; 
			
			logger.info("File " + fileInstance.getIdShort() 
									+ " added to SubmodelElementCollection: " 
									+ fileInstance.getParentSEC().getIdShort());
		}

		return addFile;
		
	}
	
	/**
	 * Generates the Semantic Reference for the File in consideration. 
	 * 
	 * @return	Returns a String with generated code containing the semantic 
	 * 			references for the file. 
	 */
	private String generateSemanticReference() {
		
		// Adding the Semantic Reference.
		String semanticStr = "";	
		if (this.semanticKeys!=null && this.conceptDesc==null) {
			
			semanticStr += "		List<IKey> " + parent + "_" + fileInstance.getIdShort().toLowerCase() 
					+ "Keys= new ArrayList<IKey>();\r\n";
			
			for (Key key : this.semanticKeys) {
				
				String basyxKeyElement = FileUtils.removeUnderScore(key.getType().toString());  
						
				semanticStr += "		" + parent + "_" + fileInstance.getIdShort().toLowerCase() 
								+ "Keys.add(" + "new Key(KeyElements." + basyxKeyElement
								+ ", " + key.isLocal() + ", " + "\"" + key.getValue() + "\""
								+ ", " + "KeyType." + key.getIdType() + ")); \r\n";
			}
			
			semanticStr += "		Reference " + parent + "_" + fileInstance.getIdShort() 
							+ "Ref = new Reference(" + parent + "_" + fileInstance.getIdShort().toLowerCase() 
							+ "Keys" + ");\r\n"
							+ "		" + parent + "_" + fileInstance.getIdShort() 
							+ ".setSemanticId(" + parent + "_" + fileInstance.getIdShort() + "Ref); \r\n \r\n";
			
			logger.info("Semantic Id reference for File: " + fileInstance.getIdShort() + "generated.");
		}
		
		else if (this.conceptDesc!=null && this.semanticKeys==null) {
			
			semanticStr += "		" + parent + "_" + fileInstance.getIdShort() 
						+ ".setSemanticId(conceptDescriptions." 
						+ this.conceptDesc.getIdShort() + ".getReference()); \r\n \r\n";
			
		}
		
		else {
			
			logger.debug("No Semantic Id reference for File: " + fileInstance.getIdShort() + "found. Thus, not generated.");
			
		}
		
		return semanticStr;
		
	}
	
	/**
	 * Generates the MimeType Information for the  {@code File} in consideration.
	 * 
	 * @return	Returns a String with generated code containing the MimeType 
	 * 			Information for the {@code File} in consideration.
	 */
	private String generateMimeType() {
		
		String setMimeType = "";
		
		if (fileInstance.getMimeType()!=null) {
			setMimeType = "		" + parent + "_" + fileInstance.getIdShort() + 
					".setMimeType(\"" + fileInstance.getMimeType() + "\");\r\n";
			
			logger.info("Generated MimeType Info for File: " + 
					fileInstance.getIdShort());
		}
		
		else {
			logger.info("MimeType Info for File: " + 
					fileInstance.getIdShort() + " not generated.");
		}
		
		return setMimeType;
		
	}
	
	/**
	 * Generates the Value Information for the {@code File} in consideration.
	 * 
	 * @return	Returns a String with generated code containing the Value 
	 * 			Information for the {@code File} in consideration.
	 */
	private String generateValueCode() {
		
		String setValue = "";
		
		if (fileInstance.getValue()!=null) {
			setValue = "		" + parent + "_" + fileInstance.getIdShort() + 
					".setValue(\"" + fileInstance.getValue() + "\");\r\n";
			
			logger.info("Generated Value Info for File: " + 
					fileInstance.getIdShort());
		}
		
		else {
			logger.info("Value Info for File: " + 
					fileInstance.getIdShort() + " not generated.");
		}
		
		return setValue;
		
	}
	
	/**
	 * Generates the Value Delegates for the Property under consideration in case
	 * the Property is declared as "Dynamic". 
	 * 
	 * @return	Returns a String with generated code containing the ValueDelegates
	 * 			installed on the respective Property.  
	 */
	private String generateValueDelegates() {
		
		String valueDelegate = "";
		String valueCode = "";
		
		String getterCode = "";
		String setterCode = "";
		
		if (this.fileInstance.isDynamic()) {
			
			valueDelegate += 
					"		ValueDelegate<" + "String"
					+ "> valDel" + parent + "_" + fileInstance.getIdShort() 
					+ " = ValueDelegate.installOn(" + parent + "_" + fileInstance.getIdShort() 
					+ "); \r\n";
			
			getterCode += 
					"		valDel" + parent + "_" + this.fileInstance.getIdShort() 
					+ ".setGetHandler(dew::get_" + parent + "_" + fileInstance.getIdShort() + "); \r\n";
			
			setterCode += "";
			
			valueDelegate += getterCode + setterCode;
			
			return valueDelegate;
			
		}
		
		else {
			valueCode += generateValueCode();
			
			return valueCode;
		}
		
	}

}
