/*****************************************************************************
 * Copyright (c) 2020 CEA LIST.
 *
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * Contributors:
 *  Ansgar Radermacher  ansgar.radermacher@cea.fr
 *
 *****************************************************************************/

package org.eclipse.papyrus.robotics.ros2.codegen.component

import org.eclipse.uml2.uml.Class

import static extension org.eclipse.papyrus.robotics.core.utils.ParameterUtils.*
import static extension org.eclipse.papyrus.robotics.core.utils.FunctionUtils.getFunction
import static extension org.eclipse.papyrus.robotics.ros2.codegen.utils.ActivityUtils.*
import static extension org.eclipse.papyrus.uml.tools.utils.StereotypeUtil.applyApp
import static extension org.eclipse.papyrus.uml.tools.utils.StereotypeUtil.apply
import org.eclipse.papyrus.designer.languages.common.base.ElementUtils
import org.eclipse.uml2.uml.Type
import org.eclipse.papyrus.robotics.profile.robotics.functions.FunctionKind
import org.eclipse.papyrus.robotics.ros2.codegen.utils.Helpers
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.ManualGeneration
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Include

class CreateMain {
	/**
	 * register a ROS2 components
	 */
	def static registerComponent(Class component) '''
		#include "rclcpp_components/register_node_macro.hpp"
		
		// Register the component with class_loader.
		// This acts as a sort of entry point, allowing the component to be discoverable when its library
		// is being loaded into a running process.
		RCLCPP_COMPONENTS_REGISTER_NODE(«component.qualifiedName»)
	'''

	/**
	 * Create the main entry point for a class
	 */
	def static createMain(Class component) {
		val main = component.nearestPackage.createOwnedClass(component.name + "_main", false);
		component.createDependency(main)
		main.apply(ManualGeneration)
		val include = main.applyApp(Include);
		include.body = component.createMainCode.toString
	}	
	
	def static createMainCode(Class component) '''
		«val instRef = component.instName + "->"»
		«val onActivate = component.onActivate(instRef)»
		«val compClassName = component.qualifiedName + component.postfix»
		#include "«component.nearestPackage.name»/«component.name»«component.postfix».h"

		«IF component.hasPeriodicActivities»
			using namespace std::chrono_literals;
		«ENDIF»
		«component.createTimer»
		// declare options
		rclcpp::NodeOptions «component.instName»_options;

		int main(int argc, char **argv) {
			rclcpp::init(argc, argv);

			auto «component.instName» = std::make_shared<«compClassName»>(«component.instName»_options);

			RCLCPP_INFO(«instRef»get_logger(), "«component.name» has been initialized");

«««			«component.initializeParameters(component)»
			«IF onActivate.length > 0»
				«instRef»register_on_activate(
					«onActivate»
				);
			«ENDIF»
			«IF component.allParameters.size > 0»
				«instRef»declareParameters();
				«instRef»initParameterVars();
			«ENDIF»

			rclcpp::executors::MultiThreadedExecutor executor;

			executor.add_node(«instRef»get_node_base_interface());

			executor.spin();
			rclcpp::shutdown();
		}
	'''

	def static void createTimer(Class component) {
		if (component.functionsToActivate.size > 0) {
			val timerBase = ElementUtils.getQualifiedElementFromRS(component, "ros2Library::rclcpp::timer::TimerBase") as Type;
			val timer = component.createOwnedAttribute("timer_", timerBase);
			Helpers.useSharedPtr(timer)
		}
	}

	/**
	 * If an activity is associated with a port publishing port, call
	 * this activity when the component (node) gets activated
	 */
	def static onActivate(Class component, String instRef) '''
		«FOR activity : component.functionsToActivate»
			«val period = activity.period»
			«val activateFct = activity.getFunction(FunctionKind.ON_ACTIVATE)»
			«val periodicFct = activity.getFunction(FunctionKind.PERIODIC)»
			[«component.instName»](const rclcpp_lifecycle::State&) {
				«IF activateFct !== null»
					«instRef»«activateFct.name»();
				«ENDIF»
				«IF period !== null && periodicFct !== null»
					// periodic execution («period») using a wall timer
					«instRef»timer_ = «instRef»create_wall_timer(«period»,
						std::bind(&«component.nearestPackage.name»::«component.name»«component.postfix»::«periodicFct.name», «component.instName»));
				«ENDIF»
				return rclcpp_lifecycle::node_interfaces::LifecycleNodeInterface::CallbackReturn::SUCCESS;
			}
		«ENDFOR»
	'''

	/**
	 * Return the name of the variable holding the component instance.
	 */
	def static getInstName(Class component) {
		return component.name.toLowerCase
	}
}