/**
 * Copyright (c) 2018, 2021 CEA LIST.
 * 
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * Contributors:
 *  Ansgar Radermacher  ansgar.radermacher@cea.fr
 *  Matteo MORELLI      matteo.morelli@cea.fr - Bug #566899
 */
package org.eclipse.papyrus.robotics.ros2.codegen.cpp.build;

import com.google.common.base.Objects;
import java.util.ArrayList;
import java.util.List;
import org.eclipse.papyrus.designer.languages.common.base.file.IPFileSystemAccess;
import org.eclipse.papyrus.robotics.codegen.common.utils.ComponentUtils;
import org.eclipse.papyrus.robotics.codegen.common.utils.PackageTools;
import org.eclipse.papyrus.robotics.ros2.codegen.common.utils.MessageUtils;
import org.eclipse.papyrus.robotics.ros2.codegen.common.utils.PackageXMLUtils;
import org.eclipse.papyrus.robotics.ros2.codegen.common.utils.SkillUtils;
import org.eclipse.papyrus.uml.tools.utils.PackageUtil;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * Create PackageXML file for a ROS 2 package, containing components and eventually one system
 */
@SuppressWarnings("all")
public class CreateCompPackageXML {
  public static String USER = "USER";

  public static CharSequence createPackageXML(final org.eclipse.uml2.uml.Package model, final List<org.eclipse.uml2.uml.Class> allComponents, final List<org.eclipse.uml2.uml.Class> componentsInPkg, final org.eclipse.uml2.uml.Class system) {
    StringConcatenation _builder = new StringConcatenation();
    final ArrayList<org.eclipse.uml2.uml.Package> entities = CreateCompPackageXML.entityArray(allComponents, system);
    _builder.newLineIfNotEmpty();
    _builder.append("<?xml version=\"1.0\"?>");
    _builder.newLine();
    _builder.append("<package format=\"3\">");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("<name>");
    String _pkgName = PackageTools.pkgName(model);
    _builder.append(_pkgName, "\t");
    _builder.append("</name>");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("<version>0.0.0</version>");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("<description>");
    String _description = PackageXMLUtils.description(entities);
    _builder.append(_description, "\t");
    _builder.append("</description>");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("<maintainer email=\"");
    String _maintainerMail = PackageXMLUtils.getMaintainerMail(entities);
    _builder.append(_maintainerMail, "\t");
    _builder.append("\">");
    String _maintainerName = PackageXMLUtils.getMaintainerName(entities);
    _builder.append(_maintainerName, "\t");
    _builder.append("</maintainer>");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("<license>Apache2.0</license>");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("<url type=\"website\">https://eclipse.org/papyrus</url>");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("<author email=\"");
    String _authorMail = PackageXMLUtils.getAuthorMail(entities);
    _builder.append(_authorMail, "\t");
    _builder.append("\">");
    String _authorName = PackageXMLUtils.getAuthorName(model);
    _builder.append(_authorName, "\t");
    _builder.append("</author>");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("\t");
    _builder.append("<buildtool_depend>ament_cmake</buildtool_depend>");
    _builder.newLine();
    _builder.newLine();
    {
      List<String> _calcDependencies = CreateCompPackageXML.calcDependencies(model, allComponents, componentsInPkg);
      for(final String dependency : _calcDependencies) {
        _builder.append("\t");
        _builder.append("<build_depend>");
        _builder.append(dependency, "\t");
        _builder.append("</build_depend>");
        _builder.newLineIfNotEmpty();
      }
    }
    {
      boolean _isNullOrEmpty = IterableExtensions.isNullOrEmpty(SkillUtils.getUniqueSkills(system));
      boolean _not = (!_isNullOrEmpty);
      if (_not) {
        _builder.append("\t");
        _builder.append("<build_depend>");
        String _realizationPackageName = SkillUtils.realizationPackageName(model);
        _builder.append(_realizationPackageName, "\t");
        _builder.append("</build_depend>");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.newLine();
    {
      List<String> _calcDependencies_1 = CreateCompPackageXML.calcDependencies(model, allComponents, componentsInPkg);
      for(final String dependency_1 : _calcDependencies_1) {
        _builder.append("\t");
        _builder.append("<exec_depend>");
        _builder.append(dependency_1, "\t");
        _builder.append("</exec_depend>");
        _builder.newLineIfNotEmpty();
      }
    }
    {
      boolean _isNullOrEmpty_1 = IterableExtensions.isNullOrEmpty(SkillUtils.getUniqueSkills(system));
      boolean _not_1 = (!_isNullOrEmpty_1);
      if (_not_1) {
        _builder.append("\t");
        _builder.append("<exec_depend>");
        String _realizationPackageName_1 = SkillUtils.realizationPackageName(model);
        _builder.append(_realizationPackageName_1, "\t");
        _builder.append("</exec_depend>");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.newLine();
    _builder.append("\t");
    _builder.append("<export>");
    _builder.newLine();
    _builder.append("\t\t");
    _builder.append("<build_type>ament_cmake</build_type>");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("</export>");
    _builder.newLine();
    _builder.append("</package>");
    _builder.newLine();
    return _builder;
  }

  /**
   * Calculate the dependencies of a list of all components and those defined in
   * this package. The dependencies includes the message dependencies of the
   * components in this package and user-defined dependencies as well as the package
   * defining the component itself for all components.
   */
  public static List<String> calcDependencies(final org.eclipse.uml2.uml.Package model, final List<org.eclipse.uml2.uml.Class> allComponents, final List<org.eclipse.uml2.uml.Class> componentsInPkg) {
    final List<String> dependencies = MessageUtils.calcDependencies(componentsInPkg);
    for (final org.eclipse.uml2.uml.Class component : allComponents) {
      {
        List<String> _dependsPackage = ComponentUtils.getDependsPackage(component);
        for (final String pkgName : _dependsPackage) {
          dependencies.add(pkgName);
        }
        String _pkgName = PackageTools.pkgName(component.getModel());
        String _pkgName_1 = PackageTools.pkgName(model);
        boolean _notEquals = (!Objects.equal(_pkgName, _pkgName_1));
        if (_notEquals) {
          dependencies.add(PackageTools.pkgName(component.getModel()));
        }
      }
    }
    return dependencies;
  }

  public static ArrayList<org.eclipse.uml2.uml.Package> entityArray(final List<org.eclipse.uml2.uml.Class> components, final org.eclipse.uml2.uml.Class system) {
    final ArrayList<org.eclipse.uml2.uml.Package> entities = new ArrayList<org.eclipse.uml2.uml.Package>();
    if ((system != null)) {
      entities.add(PackageUtil.getRootPackage(system));
    }
    for (final org.eclipse.uml2.uml.Class component : components) {
      entities.add(PackageUtil.getRootPackage(component));
    }
    return entities;
  }

  public static void generate(final IPFileSystemAccess fileAccess, final org.eclipse.uml2.uml.Package pkg, final List<org.eclipse.uml2.uml.Class> allComponents, final List<org.eclipse.uml2.uml.Class> componentsInPkg, final org.eclipse.uml2.uml.Class system) {
    fileAccess.generateFile("package.xml", CreateCompPackageXML.createPackageXML(pkg, allComponents, componentsInPkg, system).toString());
  }
}
