/*****************************************************************************
 * Copyright (c) 2023 CEA LIST and others.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *   Vincent Lorenzo (CEA LIST) vincent.lorenzo@cea.fr - Initial API and implementation
 *
 *****************************************************************************/

package org.eclipse.papyrus.model2doc.emf.template2structure.internal.generator;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.osgi.util.NLS;
import org.eclipse.papyrus.model2doc.emf.template2structure.Activator;
import org.eclipse.papyrus.model2doc.emf.template2structure.generator.ITemplate2StructureGenerator;
import org.eclipse.papyrus.model2doc.emf.template2structure.mapping.AbstractTemplateToStructureMapper;

/**
 * Extension point reader for structure generator extension point (org.eclipse.papyrus.model2doc.emf.template2structure.structuregenerator)
 */
public final class Template2StructureGeneratorExtensionReader {

	/**
	 * The name of the extension point
	 */
	private final String EXTENSION_ID = "org.eclipse.papyrus.model2doc.emf.template2structure.structuregenerator"; //$NON-NLS-1$

	/**
	 * name of the mapper extension
	 */
	private static final String MAPPER = "mapper"; //$NON-NLS-1$

	/**
	 * name of the generator extension point
	 */
	private static final String GENERATOR = "generator"; //$NON-NLS-1$

	/**
	 * Fields of the extension
	 */
	private static final String CLASS_ATTRIBUTE = "class"; //$NON-NLS-1$

	private static final String ID_ATTRIBUTE = "id"; //$NON-NLS-1$

	private static final String GENERATOR_ID_ATTRIBUTE = "generatorId"; //$NON-NLS-1$

	private static final String INHERITS_MAPPER_FROM_GENERATOR_ATTRIBUTE = "inheritsMapperFromGenerator"; //$NON-NLS-1$

	/**
	 *
	 * Constructor.
	 *
	 */
	public Template2StructureGeneratorExtensionReader() {
		// nothing to do
	}

	/**
	 *
	 * @return
	 *         a map with the id of the generators as key and a {@link Template2StructureGeneratorDescription} as value
	 */
	public Map<String, Template2StructureGeneratorDescription> readContributions() {
		final IConfigurationElement[] configElements = Platform.getExtensionRegistry().getConfigurationElementsFor(EXTENSION_ID);
		final Collection<IConfigurationElement> generatorContributions = new ArrayList<>();
		final Collection<IConfigurationElement> mapperContributions = new ArrayList<>();

		// 1. we cross all contributions to identify generator contributions and mapper contributions
		for (final IConfigurationElement iConfigurationElement : configElements) {
			if (GENERATOR.equals(iConfigurationElement.getName())) {
				generatorContributions.add(iConfigurationElement);
			} else if (MAPPER.equals(iConfigurationElement.getName())) {
				mapperContributions.add(iConfigurationElement);
			}
		}

		// 2. we read the generator contribution. this action creates the returned map
		final Map<String, Template2StructureGeneratorDescription> generatorDescriptions = readGeneratorConfigurations(generatorContributions);
		// 3. we associate mappers and generators
		associateMappersToGenerators(generatorDescriptions, mapperContributions);
		// 4. we associate parent generator
		associateInheritedGenerator(generatorDescriptions);
		// 5. we check there is no loop in inheritance description
		checkInheritance(generatorDescriptions);
		return generatorDescriptions;
	}

	/**
	 *
	 * @param generatorContributions
	 *            the generator contribution
	 * @return
	 *         a map with the id of the generators as key and a {@link Template2StructureGeneratorDescription} as value
	 */
	private Map<String, Template2StructureGeneratorDescription> readGeneratorConfigurations(Collection<IConfigurationElement> generatorContributions) {
		final Map<String, Template2StructureGeneratorDescription> generatorDescriptions = new HashMap<>();
		for (final IConfigurationElement iConfigurationElement : generatorContributions) {
			final String registeredId = iConfigurationElement.getAttribute(ID_ATTRIBUTE);
			final String inheritedGeneratorId = iConfigurationElement.getAttribute(INHERITS_MAPPER_FROM_GENERATOR_ATTRIBUTE);
			ITemplate2StructureGenerator generator = null;
			try {
				generator = (ITemplate2StructureGenerator) iConfigurationElement.createExecutableExtension(CLASS_ATTRIBUTE);
			} catch (CoreException e) {
				Activator.log.error(NLS.bind("The generator {0} can't be loaded.", iConfigurationElement.getAttribute(CLASS_ATTRIBUTE)), e); //$NON-NLS-1$ }
			}
			if (null != generator) {
				if (null == registeredId || registeredId.isEmpty()) {
					Activator.log.warn(NLS.bind("The contributed generator {0} is ignored because you don't define id for it in your extension point contribution {1}", generator.toString(), EXTENSION_ID + "/" + GENERATOR)); //$NON-NLS-1$ //$NON-NLS-2$
				}
				final String expectedId = generator.getGeneratorId();
				if (false == registeredId.equals(expectedId)) {
					final List<String> messageParameters = new ArrayList<>();
					messageParameters.add(generator.toString());
					messageParameters.add(registeredId);
					messageParameters.add(expectedId);
					Activator.log.warn(NLS.bind("The generator {0} is declared with the id {1} is the extension point contribution, but it should be {2} as declared inside it. We ignore it.", messageParameters.toArray())); //$NON-NLS-1$
				} else if (generatorDescriptions.containsKey(registeredId)) {
					final List<String> messageParameters = new ArrayList<>();
					messageParameters.add(generatorDescriptions.get(registeredId).getGenerator().toString());
					messageParameters.add(generator.toString());
					messageParameters.add(registeredId);
					Activator.log.warn(NLS.bind("The generators {0} and {1} are regitered with the same id {3}. We ignore the second one", messageParameters.toArray())); //$NON-NLS-1$
				} else {
					final Template2StructureGeneratorDescription description = new Template2StructureGeneratorDescription(generator, inheritedGeneratorId);
					generatorDescriptions.put(generator.getGeneratorId(), description);
				}
			}
		}

		return generatorDescriptions;
	}

	/**
	 * this method associates the mappers to the generators
	 *
	 * @param generatorDescriptions
	 * @param mapperContributions
	 */
	private void associateMappersToGenerators(final Map<String, Template2StructureGeneratorDescription> generatorDescriptions, final Collection<IConfigurationElement> mapperContributions) {
		for (final IConfigurationElement iConfigurationElement : mapperContributions) {

			final String contributedGenerator = iConfigurationElement.getAttribute(GENERATOR_ID_ATTRIBUTE);
			AbstractTemplateToStructureMapper<?> mapper = null;
			try {
				mapper = (AbstractTemplateToStructureMapper<?>) iConfigurationElement.createExecutableExtension(CLASS_ATTRIBUTE);
			} catch (CoreException e) {
				Activator.log.error(NLS.bind("The mapper {0} can't be loaded.", iConfigurationElement.getAttribute(CLASS_ATTRIBUTE)), e); //$NON-NLS-1$
			}
			if (null != mapper) {
				if (null == contributedGenerator || contributedGenerator.isEmpty()) {
					Activator.log.warn(NLS.bind("The contributed mapper {0} is ignored because you don't define id for it in your extension point contribution {1}", mapper.toString(), EXTENSION_ID + "/" + MAPPER)); //$NON-NLS-1$ //$NON-NLS-2$
				} else {
					final Template2StructureGeneratorDescription generatorDescription = generatorDescriptions.get(contributedGenerator);
					if (generatorDescription == null) {
						Activator.log.warn(NLS.bind("The contributed mapper {0} is ignored because we can't find generator with id {1}", mapper.toString(), contributedGenerator)); //$NON-NLS-1$
					} else {
						generatorDescription.registerMapper(mapper);
					}
				}
			}
		}
	}

	/**
	 * this method associates generator to their inherited generator
	 *
	 * @param generatorDescriptions
	 */
	private void associateInheritedGenerator(final Map<String, Template2StructureGeneratorDescription> generatorDescriptions) {
		for (final Template2StructureGeneratorDescription current : generatorDescriptions.values()) {
			final String inheritedGeneratorId = current.getInheritsFromGeneratorId();
			if (inheritedGeneratorId != null && !inheritedGeneratorId.isEmpty()) {
				Template2StructureGeneratorDescription parent = generatorDescriptions.get(inheritedGeneratorId);
				if (parent == null) {
					Activator.log.warn(NLS.bind("The generator {0} is not found. It is defines as parent for {1}", inheritedGeneratorId, current.getGeneratorId())); //$NON-NLS-1$
				} else {
					current.setInheritedGenerator(parent);
				}
			}
		}
	}


	/**
	 * this method checks there is not loop in the inheritance
	 *
	 * @param generatorDescriptions
	 */
	private void checkInheritance(final Map<String, Template2StructureGeneratorDescription> generatorDescriptions) {
		List<String> toRemove = new ArrayList<>();
		for (Template2StructureGeneratorDescription current : generatorDescriptions.values()) {
			final IStatus status = validateInheritance(current);
			if (!status.isOK()) {
				Activator.log.warn(NLS.bind("The generator {0} is ignored. {1}", current.getGeneratorId(), status.getMessage())); //$NON-NLS-1$
				toRemove.add(current.getGeneratorId());
			}
		}
		for (final String current : toRemove) {
			generatorDescriptions.remove(current);
		}
	}

	/**
	 *
	 * @param generatorDescription
	 * @return
	 *         a IStatus indicating if there is a problem in the generator inheritance:
	 *         <ul>
	 *         <li>{@link IStatus#OK} : it's OK</li>
	 *         <li>{@link IStatus#ERROR} : there is a loop in the inheritance of the generator</li>
	 *         <ul>
	 */
	private IStatus validateInheritance(final Template2StructureGeneratorDescription generatorDescription) {
		final List<Template2StructureGeneratorDescription> crossedParent = new ArrayList<>();
		Template2StructureGeneratorDescription parent = generatorDescription;
		while (parent != null && !crossedParent.contains(parent)) {
			crossedParent.add(parent);
			parent = parent.getInheritedGenerator();
		}
		if (parent == null) {
			return Status.OK_STATUS;
		}
		final StringBuilder builder = new StringBuilder();
		final Iterator<Template2StructureGeneratorDescription> iter = crossedParent.iterator();
		while (iter.hasNext()) {
			builder.append(iter.next().getGeneratorId());
			builder.append(" -> "); //$NON-NLS-1$
		}
		builder.append(parent.getGeneratorId());

		final String message = NLS.bind("There is an infinite inheritance loop for {0} : {1}", generatorDescription.getGeneratorId(), builder.toString()); //$NON-NLS-1$
		return new Status(IStatus.ERROR, Activator.PLUGIN_ID, message);
	}

}
