/*****************************************************************************
 * Copyright (c) 2014, 2025 CEA LIST.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *  Benoit Maggi (CEA LIST) benoit.maggi@cea.fr - Initial API and implementation
 *  Vincent LORENZO (CEA LIST) vincent.lorenzo@cea.fr - Issue 15
 *****************************************************************************/
package org.eclipse.papyrus.infra.gmfdiag.common.commands;

import org.eclipse.core.runtime.Assert;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.transaction.RecordingCommand;
import org.eclipse.emf.transaction.TransactionalEditingDomain;
import org.eclipse.gmf.runtime.notation.Diagram;
import org.eclipse.papyrus.infra.gmfdiag.common.model.NotationUtils;
import org.eclipse.papyrus.infra.gmfdiag.common.utils.DiagramUtils;


/**
 * Insert a diagram with the specified owner and element
 */
public class InsertDiagramCommand extends RecordingCommand {

	/** FIXME the domain of Recording command should be accessible */
	protected TransactionalEditingDomain transactionalEditingDomain;

	/** diagram to be inserted */
	protected Diagram diagram;

	/** owner of the diagram. This field is used by the ModelExplorer to know which element must be used as parent of the shown diagram */
	protected EObject graphicalOwner;

	/** element of the diagram. This field is mandatory, it is the semantic context of the diagram. The background of the diagram represents this semantic element */
	protected EObject semanticOwner;


	/**
	 * Insert a diagram with the specified owner and element
	 *
	 * @param transactionalEditingDomain
	 * @param label
	 *            of the command
	 * @param diagram
	 *            to be inserted
	 * @param semanticOwner
	 *            of the diagram. This field is mandatory, it is the semantic context of the diagram. The background of the diagram represents this semantic element
	 *            This parameter will also be used as owner of the diagram
	 */
	public InsertDiagramCommand(TransactionalEditingDomain transactionalEditingDomain, String label, Diagram diagram, EObject semanticOwner) {
		this(transactionalEditingDomain, label, diagram, semanticOwner, semanticOwner);
	}

	/**
	 * Insert a diagram with the specified owner and element
	 *
	 * @param transactionalEditingDomain
	 * @param label
	 *            of the command
	 * @param diagram
	 *            to be inserted
	 * @param semanticOwner
	 *            of the diagram. This field is mandatory, it is the semantic context of the diagram. The background of the diagram represents this semantic element
	 * @param graphicalOwner
	 *            of the diagram. This field is used by the ModelExplorer to know which element must be used as parent of the shown diagram
	 *
	 */
	public InsertDiagramCommand(TransactionalEditingDomain transactionalEditingDomain, String label, Diagram diagram, EObject semanticOwner, EObject graphicalOwner) {
		super(transactionalEditingDomain, label);
		this.transactionalEditingDomain = transactionalEditingDomain;
		this.diagram = diagram;
		Assert.isNotNull(semanticOwner);
		this.semanticOwner = semanticOwner;
		this.graphicalOwner = graphicalOwner;
		if (graphicalOwner == null) {
			this.graphicalOwner = this.semanticOwner;
		}
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see org.eclipse.emf.transaction.RecordingCommand#doExecute()
	 */
	@Override
	protected void doExecute() {
		Resource targetResource = NotationUtils.getNotationResourceForDiagram(semanticOwner, transactionalEditingDomain);
		diagram.setElement(semanticOwner);
		DiagramUtils.setOwner(diagram, graphicalOwner);

		if (targetResource != null) {
			targetResource.getContents().add(diagram);
		}
	}

}
