/**
 * Copyright (c) 2020 CEA LIST.
 * 
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * Contributors:
 *  Ansgar Radermacher  ansgar.radermacher@cea.fr
 */
package org.eclipse.papyrus.robotics.ros2.codegen.python.component;

import java.util.List;
import org.eclipse.emf.common.util.EList;
import org.eclipse.papyrus.designer.uml.tools.utils.ElementUtils;
import org.eclipse.papyrus.designer.uml.tools.utils.StereotypeUtil;
import org.eclipse.papyrus.robotics.codegen.common.utils.ActivityUtils;
import org.eclipse.papyrus.robotics.codegen.common.utils.ApplyProfiles;
import org.eclipse.papyrus.robotics.codegen.common.utils.TopicUtils;
import org.eclipse.papyrus.robotics.core.utils.InteractionUtils;
import org.eclipse.papyrus.robotics.core.utils.PortUtils;
import org.eclipse.papyrus.robotics.profile.robotics.commpattern.CommunicationPattern;
import org.eclipse.papyrus.robotics.profile.robotics.components.Activity;
import org.eclipse.papyrus.robotics.profile.robotics.components.ComponentPort;
import org.eclipse.papyrus.robotics.ros2.codegen.common.component.CallbackGroups;
import org.eclipse.papyrus.robotics.ros2.codegen.common.utils.MessageUtils;
import org.eclipse.papyrus.robotics.ros2.codegen.common.utils.RosHelpers;
import org.eclipse.papyrus.robotics.ros2.codegen.python.utils.RosPythonTypes;
import org.eclipse.uml2.uml.Behavior;
import org.eclipse.uml2.uml.Classifier;
import org.eclipse.uml2.uml.Dependency;
import org.eclipse.uml2.uml.Interface;
import org.eclipse.uml2.uml.LiteralString;
import org.eclipse.uml2.uml.OpaqueBehavior;
import org.eclipse.uml2.uml.Operation;
import org.eclipse.uml2.uml.Port;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.Type;
import org.eclipse.uml2.uml.UMLPackage;
import org.eclipse.uml2.uml.ValueSpecification;
import org.eclipse.uml2.uml.profile.standard.Create;
import org.eclipse.uml2.uml.util.UMLUtil;
import org.eclipse.xtend2.lib.StringConcatenation;

/**
 * Manage constructor creation, including port
 */
@SuppressWarnings("all")
public class Constructor {
  public static Dependency createConstructor(final org.eclipse.uml2.uml.Class component) {
    Dependency _xblockexpression = null;
    {
      final Type lcNodeSC = RosPythonTypes.getType(component, "rclpy::lifecycle::Node");
      if ((lcNodeSC instanceof Classifier)) {
        component.createGeneralization(((Classifier)lcNodeSC));
      }
      final Operation op = Constructor.addConstrOp(component);
      Constructor.addConstrMethod(component, op);
      final Type qosRPolicy = RosPythonTypes.getType(component, "rclpy::qos::ReliabilityPolicy");
      final Type qosHPolicy = RosPythonTypes.getType(component, "rclpy::qos::HistoryPolicy");
      component.createDependency(qosRPolicy);
      _xblockexpression = component.createDependency(qosHPolicy);
    }
    return _xblockexpression;
  }

  /**
   * Add an init operation (constructor) that creates the ROS 2 publishers/subscribers/clients/... for ports
   */
  public static Operation addConstrOp(final org.eclipse.uml2.uml.Class component) {
    final Type string = RosPythonTypes.getType(component, "PrimitiveTypes::String");
    final Operation init = component.createOwnedOperation(component.getName(), null, null);
    init.createOwnedParameter("instName", string);
    Create create = StereotypeUtil.<Create>applyApp(init, Create.class);
    if ((create == null)) {
      ApplyProfiles.applyStdProfile(init);
      create = StereotypeUtil.<Create>applyApp(init, Create.class);
    }
    return init;
  }

  /**
   * Add a method body to the constructor operation
   */
  public static boolean addConstrMethod(final org.eclipse.uml2.uml.Class component, final Operation constructorOp) {
    boolean _xblockexpression = false;
    {
      Behavior _createOwnedBehavior = component.createOwnedBehavior(component.getName(), UMLPackage.eINSTANCE.getOpaqueBehavior());
      final OpaqueBehavior ob = ((OpaqueBehavior) _createOwnedBehavior);
      constructorOp.getMethods().add(ob);
      ob.getLanguages().add("Python");
      EList<String> _bodies = ob.getBodies();
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("super().__init__(instName)");
      _builder.newLine();
      {
        List<Activity> _activities = ActivityUtils.getActivities(component);
        for(final Activity activity : _activities) {
          Constructor.createCallbackGroupVar(activity, component);
          _builder.newLineIfNotEmpty();
        }
      }
      {
        EList<Port> _allPorts = PortUtils.getAllPorts(component);
        for(final Port port : _allPorts) {
          {
            CommunicationPattern _communicationPattern = InteractionUtils.getCommunicationPattern(port);
            boolean _tripleNotEquals = (_communicationPattern != null);
            if (_tripleNotEquals) {
              final CommunicationPattern pattern = InteractionUtils.getCommunicationPattern(port);
              _builder.newLineIfNotEmpty();
              {
                if ((InteractionUtils.isPush(pattern) || InteractionUtils.isPubSub(pattern))) {
                  CharSequence _createPush = Constructor.createPush(port);
                  _builder.append(_createPush);
                  _builder.newLineIfNotEmpty();
                  _builder.newLine();
                } else {
                  boolean _isSend = InteractionUtils.isSend(pattern);
                  if (_isSend) {
                    CharSequence _createSend = Constructor.createSend(port);
                    _builder.append(_createSend);
                    _builder.newLineIfNotEmpty();
                    _builder.newLine();
                  } else {
                    boolean _isQuery = InteractionUtils.isQuery(pattern);
                    if (_isQuery) {
                      CharSequence _createQuery = Constructor.createQuery(port);
                      _builder.append(_createQuery);
                      _builder.newLineIfNotEmpty();
                      _builder.newLine();
                    } else {
                      boolean _isAction = InteractionUtils.isAction(pattern);
                      if (_isAction) {
                        CharSequence _createAction = Constructor.createAction(port);
                        _builder.append(_createAction);
                        _builder.newLineIfNotEmpty();
                        _builder.newLine();
                      } else {
                        boolean _isEvent = InteractionUtils.isEvent(pattern);
                        if (_isEvent) {
                        }
                      }
                    }
                  }
                }
              }
            }
          }
        }
      }
      _xblockexpression = _bodies.add(_builder.toString());
    }
    return _xblockexpression;
  }

  /**
   * Create a callback_group attribute for each activity
   */
  public static void createCallbackGroupVar(final Activity activity, final org.eclipse.uml2.uml.Class component) {
    final Type cbgType = RosPythonTypes.getType(component, "rclpy::callback_groups::MutuallyExclusiveCallbackGroup");
    final Property cbgAttr = component.createOwnedAttribute(CallbackGroups.callbackGroupName(activity.getBase_Class()), cbgType);
    ValueSpecification _createDefaultValue = cbgAttr.createDefaultValue(null, null, UMLPackage.eINSTANCE.getLiteralString());
    final LiteralString defaultV = ((LiteralString) _createDefaultValue);
    defaultV.setValue("MutuallyExclusiveCallbackGroup()");
    String _period = ActivityUtils.getPeriod(activity);
    boolean _tripleNotEquals = (_period != null);
    if (_tripleNotEquals) {
      final Property tCbgAttr = component.createOwnedAttribute(CallbackGroups.tCallbackGroupName(activity.getBase_Class()), cbgType);
      ValueSpecification _createDefaultValue_1 = tCbgAttr.createDefaultValue(null, null, UMLPackage.eINSTANCE.getLiteralString());
      final LiteralString tDevaultV = ((LiteralString) _createDefaultValue_1);
      tDevaultV.setValue(defaultV.getValue());
    }
  }

  /**
   * Create a publisher or subscriber
   */
  public static CharSequence createPush(final Port port) {
    StringConcatenation _builder = new StringConcatenation();
    {
      int _size = port.getProvideds().size();
      boolean _greaterThan = (_size > 0);
      if (_greaterThan) {
        _builder.append("self.");
        String _varName = ElementUtils.varName(port);
        _builder.append(_varName);
        _builder.append("_pub_ = self.create_publisher(");
        String _name = InteractionUtils.getCommObject(port).getName();
        _builder.append(_name);
        _builder.append(", \"");
        String _topic = TopicUtils.getTopic(port);
        _builder.append(_topic);
        _builder.append("\",");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        String _qos = Constructor.qos(port, "1");
        _builder.append(_qos, "\t");
        _builder.append(", callback_group = self.");
        String _callbackGroupName = CallbackGroups.callbackGroupName(port);
        _builder.append(_callbackGroupName, "\t");
        _builder.append(")");
        _builder.newLineIfNotEmpty();
      } else {
        int _size_1 = port.getRequireds().size();
        boolean _greaterThan_1 = (_size_1 > 0);
        if (_greaterThan_1) {
          _builder.append("defaultQoSProfile = rclpy.qos.QoSProfile(");
          _builder.newLine();
          _builder.append("\t\t");
          _builder.append("reliability=ReliabilityPolicy.BEST_EFFORT,");
          _builder.newLine();
          _builder.append("\t\t");
          _builder.append("history=HistoryPolicy.KEEP_LAST,");
          _builder.newLine();
          _builder.append("\t\t");
          _builder.append("depth = 100)");
          _builder.newLine();
          _builder.append("self.");
          String _varName_1 = ElementUtils.varName(port);
          _builder.append(_varName_1);
          _builder.append("_sub_ = self.create_subscription(");
          String _name_1 = InteractionUtils.getCommObject(port).getName();
          _builder.append(_name_1);
          _builder.append(", \"");
          String _topic_1 = TopicUtils.getTopic(port);
          _builder.append(_topic_1);
          _builder.append("\", ");
          String _callBackMethodForPush = Callbacks.callBackMethodForPush(port.getClass_(), port);
          _builder.append(_callBackMethodForPush);
          _builder.append(",");
          _builder.newLineIfNotEmpty();
          _builder.append("\t");
          _builder.append("qos_profile = defaultQoSProfile, callback_group = self.");
          String _callbackGroupName_1 = CallbackGroups.callbackGroupName(port);
          _builder.append(_callbackGroupName_1, "\t");
          _builder.append(")");
          _builder.newLineIfNotEmpty();
        }
      }
    }
    return _builder;
  }

  /**
   * Create send (publisher and subscriber as well)
   */
  public static CharSequence createSend(final Port port) {
    StringConcatenation _builder = new StringConcatenation();
    {
      int _size = port.getProvideds().size();
      boolean _greaterThan = (_size > 0);
      if (_greaterThan) {
        _builder.append("defaultQoSProfile = rclpy.qos.QoSProfile(");
        _builder.newLine();
        _builder.append("\t\t");
        _builder.append("reliability=ReliabilityPolicy.BEST_EFFORT,");
        _builder.newLine();
        _builder.append("\t\t");
        _builder.append("history=HistoryPolicy.KEEP_LAST,");
        _builder.newLine();
        _builder.append("\t\t");
        _builder.append("depth = 100)");
        _builder.newLine();
        _builder.append("self.");
        String _varName = ElementUtils.varName(port);
        _builder.append(_varName);
        _builder.append("_recv_ = self.create_subscription(");
        String _name = InteractionUtils.getCommObject(port).getName();
        _builder.append(_name);
        _builder.append(", \"");
        String _topic = TopicUtils.getTopic(port);
        _builder.append(_topic);
        _builder.append("\", ");
        String _callBackMethodForPush = Callbacks.callBackMethodForPush(port.getClass_(), port);
        _builder.append(_callBackMethodForPush);
        _builder.append(", qos_profile = defaultQoSProfile, callback_group = self.");
        String _callbackGroupName = CallbackGroups.callbackGroupName(port);
        _builder.append(_callbackGroupName);
        _builder.append(")");
        _builder.newLineIfNotEmpty();
      } else {
        int _size_1 = port.getRequireds().size();
        boolean _greaterThan_1 = (_size_1 > 0);
        if (_greaterThan_1) {
          _builder.append("self.");
          String _varName_1 = ElementUtils.varName(port);
          _builder.append(_varName_1);
          _builder.append("_send_ = self.create_publisher(");
          String _name_1 = InteractionUtils.getCommObject(port).getName();
          _builder.append(_name_1);
          _builder.append(", \"");
          String _topic_1 = TopicUtils.getTopic(port);
          _builder.append(_topic_1);
          _builder.append("\", ");
          String _qos = Constructor.qos(port, "1");
          _builder.append(_qos);
          _builder.append(", callback_group = self.");
          String _callbackGroupName_1 = CallbackGroups.callbackGroupName(port);
          _builder.append(_callbackGroupName_1);
          _builder.append(")");
          _builder.newLineIfNotEmpty();
          _builder.append("// directly activate a publisher");
          _builder.newLine();
        }
      }
    }
    return _builder;
  }

  /**
   * Create a service client or server
   */
  public static CharSequence createQuery(final Port port) {
    StringConcatenation _builder = new StringConcatenation();
    {
      int _size = port.getProvideds().size();
      boolean _greaterThan = (_size > 0);
      if (_greaterThan) {
        _builder.append("self.");
        String _varName = ElementUtils.varName(port);
        _builder.append(_varName);
        _builder.append("_srv_ = self.create_service(");
        String _srvName = ComponentTransformations.getSrvName(RosHelpers.externalName(MessageUtils.getServiceType(port)));
        _builder.append(_srvName);
        _builder.append(", \"");
        String _topic = TopicUtils.getTopic(port);
        _builder.append(_topic);
        _builder.append("\", ");
        String _serverCallBackMethodForService = Callbacks.serverCallBackMethodForService(port.getClass_(), port);
        _builder.append(_serverCallBackMethodForService);
        _builder.append(", callback_group = self.");
        String _callbackGroupName = CallbackGroups.callbackGroupName(port);
        _builder.append(_callbackGroupName);
        _builder.append(")");
        _builder.newLineIfNotEmpty();
      } else {
        int _size_1 = port.getRequireds().size();
        boolean _greaterThan_1 = (_size_1 > 0);
        if (_greaterThan_1) {
          _builder.append("self.");
          String _varName_1 = ElementUtils.varName(port);
          _builder.append(_varName_1);
          _builder.append("_client_ = self.create_client(");
          String _srvName_1 = ComponentTransformations.getSrvName(RosHelpers.externalName(MessageUtils.getServiceType(port)));
          _builder.append(_srvName_1);
          _builder.append(", \"");
          String _topic_1 = TopicUtils.getTopic(port);
          _builder.append(_topic_1);
          _builder.append("\", callback_group = self.");
          String _callbackGroupName_1 = CallbackGroups.callbackGroupName(port);
          _builder.append(_callbackGroupName_1);
          _builder.append(")");
          _builder.newLineIfNotEmpty();
        }
      }
    }
    return _builder;
  }

  /**
   * Create an action client or server
   */
  public static CharSequence createAction(final Port port) {
    StringConcatenation _builder = new StringConcatenation();
    {
      int _size = port.getProvideds().size();
      boolean _greaterThan = (_size > 0);
      if (_greaterThan) {
        _builder.append("self.");
        String _varName = ElementUtils.varName(port);
        _builder.append(_varName);
        _builder.append("_actsrv_ = ActionServer(");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("self, ");
        String _srvName = ComponentTransformations.getSrvName(RosHelpers.externalName(MessageUtils.getServiceType(port)));
        _builder.append(_srvName, "\t");
        _builder.append(", \'");
        String _topic = TopicUtils.getTopic(port);
        _builder.append(_topic, "\t");
        _builder.append("\',");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        String _serverCallsbacksForAction = Callbacks.serverCallsbacksForAction(port.getClass_(), port);
        _builder.append(_serverCallsbacksForAction, "\t");
        _builder.append(",");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("callback_group = self.");
        String _callbackGroupName = CallbackGroups.callbackGroupName(port);
        _builder.append(_callbackGroupName, "\t");
        _builder.append(")");
        _builder.newLineIfNotEmpty();
        Constructor.createSvcDependency(port.getClass_(), MessageUtils.getServiceType(port));
        _builder.newLineIfNotEmpty();
      } else {
        _builder.append("self.");
        String _varName_1 = ElementUtils.varName(port);
        _builder.append(_varName_1);
        _builder.append("_actcli_ = ActionClient(");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("self, ");
        String _srvName_1 = ComponentTransformations.getSrvName(RosHelpers.externalName(MessageUtils.getServiceType(port)));
        _builder.append(_srvName_1, "\t");
        _builder.append(", \'");
        String _topic_1 = TopicUtils.getTopic(port);
        _builder.append(_topic_1, "\t");
        _builder.append("\',");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("callback_group = self.");
        String _callbackGroupName_1 = CallbackGroups.callbackGroupName(port);
        _builder.append(_callbackGroupName_1, "\t");
        _builder.append(")");
        _builder.newLineIfNotEmpty();
        Callbacks.clientCallsbacksForAction(port.getClass_(), port);
        _builder.newLineIfNotEmpty();
        Constructor.createSvcDependency(port.getClass_(), MessageUtils.getServiceType(port));
        _builder.newLineIfNotEmpty();
      }
    }
    return _builder;
  }

  /**
   * Create an event client and server
   * TODO - not supported
   */
  public static CharSequence createEvent(final Port port) {
    StringConcatenation _builder = new StringConcatenation();
    return _builder;
  }

  /**
   * Return the QoS associated with a port or defaultQoS, if none is specified
   * 
   * The "KEEP_LAST" history setting tells DDS to store a fixed-size buffer of values before they
   * are sent, to aid with recovery in the event of dropped messages.
   * use best effort to avoid blocking during execution.
   */
  public static String qos(final Port port, final String defaultQoS) {
    final ComponentPort compPort = UMLUtil.<ComponentPort>getStereotypeApplication(port, ComponentPort.class);
    if ((((compPort != null) && (compPort.getQos() != null)) && (compPort.getQos().length() > 0))) {
      return compPort.getQos();
    } else {
      return defaultQoS;
    }
  }

  /**
   * Helper to create a dependency to a service interface
   */
  public static void createSvcDependency(final org.eclipse.uml2.uml.Class component, final Interface svcType) {
    EList<Dependency> _clientDependencies = component.getClientDependencies();
    for (final Dependency dep : _clientDependencies) {
      boolean _contains = dep.getTargets().contains(svcType);
      if (_contains) {
        return;
      }
    }
    component.createDependency(svcType);
  }
}
