/**
 * Copyright (c) 2020 CEA LIST.
 * 
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * Contributors:
 *  Matteo MORELLI      matteo.morelli@cea.fr - Bug #566899
 */
package org.eclipse.papyrus.robotics.ros2.codegen.python.skillrealization;

import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.eclipse.core.resources.IProject;
import org.eclipse.emf.common.util.EList;
import org.eclipse.papyrus.designer.languages.common.base.file.FileSystemAccessFactory;
import org.eclipse.papyrus.designer.languages.common.base.file.IPFileSystemAccess;
import org.eclipse.papyrus.designer.languages.common.base.file.ProjectBasedFileAccess;
import org.eclipse.papyrus.designer.transformation.base.utils.TransformationException;
import org.eclipse.papyrus.robotics.core.utils.InteractionUtils;
import org.eclipse.papyrus.robotics.core.utils.PortUtils;
import org.eclipse.papyrus.robotics.profile.robotics.skills.SkillDefinition;
import org.eclipse.papyrus.robotics.profile.robotics.skills.SkillSemantic;
import org.eclipse.papyrus.robotics.ros2.codegen.common.utils.SkillUtils;
import org.eclipse.papyrus.robotics.ros2.codegen.python.build.CreateSkillRealizCMakeLists;
import org.eclipse.papyrus.robotics.ros2.codegen.python.build.CreateSkillRealizPackageXML;
import org.eclipse.papyrus.robotics.ros2.codegen.python.utils.ProjectTools;
import org.eclipse.uml2.uml.Interface;
import org.eclipse.uml2.uml.Port;
import org.eclipse.xtext.xbase.lib.Exceptions;

/**
 * Handle creation of a ROS 2 skill realization package
 */
@SuppressWarnings("all")
public class CreateSkillRelizationPackage {
  private static String msgAbortManyNamesForInterface(final String ifName, final String registeredName, final String newName) {
    return String.format(
      "abort transformation, coordination interface (%s) exposed through different names (%s, %s) is not supported.", ifName, registeredName, newName);
  }

  private static String msgAbortIncompatibleServices(final String refName, final String registeredIfName, final String newIfName) {
    return String.format(
      "abort transformation, the same name (%s) cannot be used to refer different Coordination interfaces (%s, %s).", refName, registeredIfName, newIfName);
  }

  /**
   * List of selected realization semantics for each skill definition
   */
  private Map<SkillDefinition, SkillSemantic> skdefToSemanticsMap;

  /**
   * Map to ensure that each CoordinationService is exposed through the same name
   */
  private Map<Interface, String> serviceToNameMap;

  /**
   * Map to ensure that a given name does not correspond to different incompatible CoordinationServices
   */
  private Map<String, Interface> nameToServiceMap;

  /**
   * Construct
   * 
   * Assuming default semantics of skill is used.
   * TODO. Manage the general case where alternative realization semantics can be specified
   */
  public CreateSkillRelizationPackage(final org.eclipse.uml2.uml.Class system, final List<org.eclipse.uml2.uml.Class> components) {
    try {
      HashMap<SkillDefinition, SkillSemantic> _hashMap = new HashMap<SkillDefinition, SkillSemantic>();
      this.skdefToSemanticsMap = _hashMap;
      LinkedHashSet<SkillDefinition> _uniqueSkills = SkillUtils.getUniqueSkills(system);
      for (final SkillDefinition skill : _uniqueSkills) {
        this.skdefToSemanticsMap.put(skill, skill.getDefaultSemantic());
      }
      HashMap<Interface, String> _hashMap_1 = new HashMap<Interface, String>();
      this.serviceToNameMap = _hashMap_1;
      HashMap<String, Interface> _hashMap_2 = new HashMap<String, Interface>();
      this.nameToServiceMap = _hashMap_2;
      for (final org.eclipse.uml2.uml.Class component : components) {
        EList<Port> _allPorts = PortUtils.getAllPorts(component);
        for (final Port port : _allPorts) {
          if (((InteractionUtils.getCommunicationPattern(port) != null) && InteractionUtils.isAction(InteractionUtils.getCommunicationPattern(port)))) {
            boolean _containsKey = this.serviceToNameMap.containsKey(InteractionUtils.getServiceDefinition(port));
            if (_containsKey) {
              String _get = this.serviceToNameMap.get(InteractionUtils.getServiceDefinition(port));
              String _name = port.getName();
              boolean _notEquals = (!Objects.equals(_get, _name));
              if (_notEquals) {
                String _msgAbortManyNamesForInterface = CreateSkillRelizationPackage.msgAbortManyNamesForInterface(
                  InteractionUtils.getServiceDefinition(port).getName(), 
                  this.serviceToNameMap.get(InteractionUtils.getServiceDefinition(port)), 
                  port.getName());
                throw new TransformationException(_msgAbortManyNamesForInterface);
              }
            }
            this.serviceToNameMap.put(InteractionUtils.getServiceDefinition(port), port.getName());
            boolean _containsKey_1 = this.nameToServiceMap.containsKey(port.getName());
            if (_containsKey_1) {
              Interface _get_1 = this.nameToServiceMap.get(port.getName());
              Interface _serviceDefinition = InteractionUtils.getServiceDefinition(port);
              boolean _tripleNotEquals = (_get_1 != _serviceDefinition);
              if (_tripleNotEquals) {
                String _msgAbortIncompatibleServices = CreateSkillRelizationPackage.msgAbortIncompatibleServices(
                  port.getName(), 
                  this.nameToServiceMap.get(port.getName()).getName(), 
                  InteractionUtils.getServiceDefinition(port).getName());
                throw new TransformationException(_msgAbortIncompatibleServices);
              }
            }
            this.nameToServiceMap.put(port.getName(), InteractionUtils.getServiceDefinition(port));
          }
        }
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  /**
   * Create a skill realization package from the list of skills to be executed by a given system
   */
  public void createSkillRealizationPkg(final org.eclipse.uml2.uml.Package pkg) {
    final IProject project = ProjectTools.getProject(SkillUtils.realizationPackageName(pkg));
    final IPFileSystemAccess fileAccess = FileSystemAccessFactory.create(project);
    CreateSkillRealizCMakeLists.generate(fileAccess, pkg, this.skdefToSemanticsMap);
    CreateSkillRealizPackageXML.generate(fileAccess, pkg, this.skdefToSemanticsMap);
    ProjectBasedFileAccess _projectBasedFileAccess = new ProjectBasedFileAccess(project, "src");
    CreateSkillRealizationPythonCode.genCode(_projectBasedFileAccess, this.skdefToSemanticsMap, this.serviceToNameMap);
  }
}
