/*******************************************************************************
 * Copyright (c) 2015 David Green.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     David Green - initial API and implementation
 *******************************************************************************/

package org.eclipse.mylyn.wikitext.commonmark.internal.inlines;

import static com.google.common.base.MoreObjects.toStringHelper;
import static com.google.common.base.Preconditions.checkNotNull;

import java.util.List;
import java.util.Objects;

import org.eclipse.mylyn.wikitext.commonmark.internal.Line;
import org.eclipse.mylyn.wikitext.commonmark.internal.ToStringHelper;
import org.eclipse.mylyn.wikitext.parser.DocumentBuilder;
import org.eclipse.mylyn.wikitext.parser.DocumentBuilder.SpanType;
import org.eclipse.mylyn.wikitext.parser.LinkAttributes;

public class Link extends InlineWithNestedContents {

	private final String href;

	private final String title;

	public Link(Line line, int offset, int length, String href, String title, List<Inline> contents) {
		super(line, offset, length, contents);
		this.href = checkNotNull(href);
		this.title = title;
	}

	public String getHref() {
		return href;
	}

	@Override
	public void emit(DocumentBuilder builder) {
		LinkAttributes attributes = new LinkAttributes();
		attributes.setTitle(title);
		attributes.setHref(href);
		builder.beginSpan(SpanType.LINK, attributes);

		InlineParser.emit(builder, getContents());

		builder.endSpan();
	}

	@Override
	public int hashCode() {
		return Objects.hash(getOffset(), getLength(), getContents(), href, title);
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj) {
			return true;
		}
		if (!super.equals(obj)) {
			return false;
		}
		Link other = (Link) obj;
		return href.equals(other.href) && getContents().equals(other.getContents())
				&& Objects.equals(title, other.title);
	}

	@Override
	public String toString() {
		return toStringHelper(Link.class).add("offset", getOffset())
				.add("length", getLength())
				.add("href", ToStringHelper.toStringValue(href))
				.add("title", title)
				.add("contents", getContents())
				.toString();
	}
}
