/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.rdf4j.sail.nativerdf.btree;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.eclipse.rdf4j.common.io.ByteArrayUtil;
import org.eclipse.rdf4j.common.io.NioFile;
import org.eclipse.rdf4j.sail.SailException;
import org.eclipse.rdf4j.sail.nativerdf.btree.AllocatedNodesList;
import org.eclipse.rdf4j.sail.nativerdf.btree.ConcurrentNodeCache;
import org.eclipse.rdf4j.sail.nativerdf.btree.DefaultRecordComparator;
import org.eclipse.rdf4j.sail.nativerdf.btree.Node;
import org.eclipse.rdf4j.sail.nativerdf.btree.RangeIterator;
import org.eclipse.rdf4j.sail.nativerdf.btree.RecordComparator;
import org.eclipse.rdf4j.sail.nativerdf.btree.RecordIterator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BTree
implements Closeable {
    static final byte[] MAGIC_NUMBER = new byte[]{98, 116, 102};
    static final byte[] OLD_MAGIC_NUMBER = new byte[]{0, 0, 0};
    static final byte FILE_FORMAT_VERSION = 1;
    static final int HEADER_LENGTH = 16;
    private static final Logger logger = LoggerFactory.getLogger(BTree.class);
    final NioFile nioFile;
    private final boolean forceSync;
    final RecordComparator comparator;
    final ReentrantReadWriteLock btreeLock = new ReentrantReadWriteLock();
    private final ConcurrentNodeCache nodeCache = new ConcurrentNodeCache(id -> {
        Node node = new Node((int)id, this);
        try {
            node.read();
        }
        catch (IOException exc) {
            throw new SailException("Error reading B-tree node", exc);
        }
        return node;
    });
    private final AllocatedNodesList allocatedNodesList;
    final int blockSize;
    final int valueSize;
    final int slotSize;
    final int branchFactor;
    final int minValueCount;
    final int nodeSize;
    private volatile int rootNodeID;
    private volatile int height = -1;
    private final AtomicBoolean closed = new AtomicBoolean(false);

    public BTree(File dataDir, String filenamePrefix, int blockSize, int valueSize) throws IOException {
        this(dataDir, filenamePrefix, blockSize, valueSize, false);
    }

    public BTree(File dataDir, String filenamePrefix, int blockSize, int valueSize, boolean forceSync) throws IOException {
        this(dataDir, filenamePrefix, blockSize, valueSize, new DefaultRecordComparator(), forceSync);
    }

    public BTree(File dataDir, String filenamePrefix, int blockSize, int valueSize, RecordComparator comparator) throws IOException {
        this(dataDir, filenamePrefix, blockSize, valueSize, comparator, false);
    }

    public BTree(File dataDir, String filenamePrefix, int blockSize, int valueSize, RecordComparator comparator, boolean forceSync) throws IOException {
        if (dataDir == null) {
            throw new IllegalArgumentException("dataDir must not be null");
        }
        if (filenamePrefix == null) {
            throw new IllegalArgumentException("filenamePrefix must not be null");
        }
        if (blockSize < 16) {
            throw new IllegalArgumentException("block size must be at least 16 bytes");
        }
        if (valueSize <= 0) {
            throw new IllegalArgumentException("value size must be larger than 0");
        }
        if (blockSize < 3 * valueSize + 20) {
            throw new IllegalArgumentException("block size to small; must at least be able to store three values");
        }
        if (comparator == null) {
            throw new IllegalArgumentException("comparator muts not be null");
        }
        File file = new File(dataDir, filenamePrefix + ".dat");
        this.nioFile = new NioFile(file);
        this.comparator = comparator;
        this.forceSync = forceSync;
        File allocFile = new File(dataDir, filenamePrefix + ".alloc");
        this.allocatedNodesList = new AllocatedNodesList(allocFile, this, forceSync);
        if (this.nioFile.size() == 0L) {
            this.blockSize = blockSize;
            this.valueSize = valueSize;
            this.rootNodeID = 0;
            this.height = 0;
            this.writeFileHeader();
        } else {
            ByteBuffer buf = ByteBuffer.allocate(16);
            this.nioFile.read(buf, 0L);
            buf.rewind();
            byte[] magicNumber = new byte[MAGIC_NUMBER.length];
            buf.get(magicNumber);
            byte version = buf.get();
            this.blockSize = buf.getInt();
            this.valueSize = buf.getInt();
            this.rootNodeID = buf.getInt();
            if (Arrays.equals(MAGIC_NUMBER, magicNumber)) {
                if (version > 1) {
                    throw new IOException("Unable to read BTree file " + file + "; it uses a newer file format");
                }
                if (version != 1) {
                    throw new IOException("Unable to read BTree file " + file + "; invalid file format version: " + version);
                }
            } else if (Arrays.equals(OLD_MAGIC_NUMBER, magicNumber)) {
                if (version != 1) {
                    throw new IOException("Unable to read BTree file " + file + "; invalid file format version: " + version);
                }
                logger.info("Updating file header for btree file '{}'", (Object)file.getAbsolutePath());
                this.writeFileHeader();
            } else {
                throw new IOException("File doesn't contain (compatible) BTree data: " + file);
            }
            if (this.valueSize != valueSize) {
                throw new IOException("Specified value size (" + valueSize + ") is different from what is stored on disk (" + this.valueSize + ") in " + file);
            }
        }
        this.slotSize = 4 + this.valueSize;
        this.branchFactor = 1 + (this.blockSize - 8) / this.slotSize;
        this.minValueCount = (this.branchFactor - 1) / 2;
        this.nodeSize = 8 + (this.branchFactor - 1) * this.slotSize;
    }

    public File getFile() {
        return this.nioFile.getFile();
    }

    public boolean delete() throws IOException {
        if (this.closed.compareAndSet(false, true)) {
            this.close(false);
            boolean success = this.allocatedNodesList.delete();
            return success &= this.nioFile.delete();
        }
        return false;
    }

    @Override
    public void close() throws IOException {
        if (this.closed.compareAndSet(false, true)) {
            this.close(true);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void close(boolean syncChanges) throws IOException {
        this.btreeLock.writeLock().lock();
        try {
            try {
                if (syncChanges) {
                    this.sync();
                }
            }
            finally {
                try {
                    this.nodeCache.clear();
                }
                finally {
                    try {
                        this.nioFile.close();
                    }
                    finally {
                        this.allocatedNodesList.close(syncChanges);
                    }
                }
            }
        }
        finally {
            this.btreeLock.writeLock().unlock();
        }
    }

    public void sync() throws IOException {
        this.btreeLock.readLock().lock();
        try {
            this.nodeCache.flush();
            if (this.forceSync) {
                this.nioFile.force(false);
            }
            this.allocatedNodesList.sync();
        }
        finally {
            this.btreeLock.readLock().unlock();
        }
    }

    public byte[] get(byte[] key) throws IOException {
        this.btreeLock.readLock().lock();
        try {
            Node node = this.readRootNode();
            if (node == null) {
                byte[] byArray = null;
                return byArray;
            }
            while (true) {
                int valueIdx;
                if ((valueIdx = node.search(key)) >= 0) {
                    byte[] result = node.getValue(valueIdx);
                    node.release();
                    byte[] byArray = result;
                    return byArray;
                }
                if (node.isLeaf()) {
                    node.release();
                    byte[] byArray = null;
                    return byArray;
                }
                Node childNode = node.getChildNode(-valueIdx - 1);
                node.release();
                node = childNode;
            }
        }
        finally {
            this.btreeLock.readLock().unlock();
        }
    }

    public RecordIterator iterateAll() {
        return new RangeIterator(this, null, null, null, null);
    }

    public RecordIterator iterateRange(byte[] minValue, byte[] maxValue) {
        return new RangeIterator(this, null, null, minValue, maxValue);
    }

    public RecordIterator iterateValues(byte[] searchKey, byte[] searchMask) {
        return new RangeIterator(this, searchKey, searchMask, null, null);
    }

    public RecordIterator iterateRangedValues(byte[] searchKey, byte[] searchMask, byte[] minValue, byte[] maxValue) {
        return new RangeIterator(this, searchKey, searchMask, minValue, maxValue);
    }

    public long getValueCountEstimate() throws IOException {
        int allocatedNodesCount = this.allocatedNodesList.getNodeCount();
        return (long)((double)(allocatedNodesCount * (this.branchFactor - 1)) * 0.5);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public long getValueCountEstimate(byte[] minValue, byte[] maxValue) throws IOException {
        List<PathSegment> maxValuePath;
        List<PathSegment> minValuePath;
        assert (minValue != null) : "minValue must not be null";
        assert (maxValue != null) : "maxValue must not be null";
        this.btreeLock.readLock().lock();
        try {
            minValuePath = this.getPath(minValue);
            maxValuePath = this.getPath(maxValue);
        }
        finally {
            this.btreeLock.readLock().unlock();
        }
        return this.getValueCountEstimate(minValuePath, maxValuePath);
    }

    private List<PathSegment> getPath(byte[] key) throws IOException {
        assert (key != null) : "key must not be null";
        ArrayList<PathSegment> path = new ArrayList<PathSegment>(this.height());
        Node currentNode = this.readRootNode();
        if (currentNode != null) {
            while (true) {
                int keyIndex = currentNode.search(key);
                path.add(new PathSegment(keyIndex, currentNode.getValueCount()));
                if (keyIndex >= 0 || currentNode.isLeaf()) break;
                Node childNode = currentNode.getChildNode(-keyIndex - 1);
                currentNode.release();
                currentNode = childNode;
            }
            currentNode.release();
        }
        return path;
    }

    private long getValueCountEstimate(List<PathSegment> minValuePath, List<PathSegment> maxValuePath) throws IOException {
        PathSegment ps;
        int i;
        int splitIdx;
        int commonListSize = Math.min(minValuePath.size(), maxValuePath.size());
        if (commonListSize == 0) {
            return 0L;
        }
        PathSegment minNode = null;
        PathSegment maxNode = null;
        for (splitIdx = 0; splitIdx < commonListSize; ++splitIdx) {
            minNode = minValuePath.get(splitIdx);
            maxNode = maxValuePath.get(splitIdx);
            if (minNode.valueIndex != maxNode.valueIndex) break;
        }
        if (splitIdx >= commonListSize) {
            return minNode.valueIndex >= 0 ? 1L : 0L;
        }
        int minValueIndex = minNode.getMinValueIndex();
        int maxValueIndex = maxNode.getMaxValueIndex();
        long valueCount = (long)(maxValueIndex - minValueIndex) * this.getTreeSizeEstimate(splitIdx + 2);
        valueCount += (long)(maxValueIndex - minValueIndex + 1);
        for (i = splitIdx + 1; i < minValuePath.size(); ++i) {
            ps = minValuePath.get(i);
            minValueIndex = ps.getMinValueIndex();
            valueCount += (long)(ps.valueCount - minValueIndex);
            valueCount += (long)(ps.valueCount - minValueIndex) * this.getTreeSizeEstimate(i + 2);
        }
        for (i = splitIdx + 1; i < maxValuePath.size(); ++i) {
            ps = maxValuePath.get(i);
            maxValueIndex = ps.getMaxValueIndex();
            valueCount += (long)(maxValueIndex + 1);
            valueCount += (long)(maxValueIndex + 1) * this.getTreeSizeEstimate(i + 2);
        }
        return valueCount;
    }

    private long getTreeSizeEstimate(int nodeDepth) throws IOException {
        int fanOut = this.branchFactor / 2;
        long valueCount = 0L;
        for (int i = this.height() - nodeDepth; i >= 0; --i) {
            ++valueCount;
            valueCount *= (long)fanOut;
        }
        return valueCount;
    }

    private int height() throws IOException {
        if (this.height >= 0) {
            return this.height;
        }
        int nodeDepth = 0;
        Node currentNode = this.readRootNode();
        if (currentNode != null) {
            nodeDepth = 1;
            while (!currentNode.isLeaf()) {
                Node childNode = currentNode.getChildNode(0);
                currentNode.release();
                currentNode = childNode;
                ++nodeDepth;
            }
            currentNode.release();
        }
        this.height = nodeDepth;
        return this.height;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public byte[] insert(byte[] value) throws IOException {
        this.btreeLock.writeLock().lock();
        try {
            Node rootNode = this.readRootNode();
            if (rootNode == null) {
                rootNode = this.createNewNode();
                this.rootNodeID = rootNode.getID();
                this.writeFileHeader();
                this.height = 1;
            }
            InsertResult insertResult = this.insertInTree(value, 0, rootNode);
            if (insertResult.overflowValue != null) {
                Node newRootNode = this.createNewNode();
                newRootNode.setChildNodeID(0, rootNode.getID());
                newRootNode.insertValueNodeIDPair(0, insertResult.overflowValue, insertResult.overflowNodeID);
                this.rootNodeID = newRootNode.getID();
                this.writeFileHeader();
                newRootNode.release();
                if (this.height >= 0) {
                    ++this.height;
                }
            }
            rootNode.release();
            byte[] byArray = insertResult.oldValue;
            return byArray;
        }
        finally {
            this.btreeLock.writeLock().unlock();
        }
    }

    private InsertResult insertInTree(byte[] value, int nodeID, Node node) throws IOException {
        InsertResult insertResult = null;
        int valueIdx = node.search(value);
        if (valueIdx >= 0) {
            insertResult = new InsertResult();
            insertResult.oldValue = node.getValue(valueIdx);
            if (!Arrays.equals(value, insertResult.oldValue)) {
                node.setValue(valueIdx, value);
            }
        } else {
            valueIdx = -valueIdx - 1;
            if (node.isLeaf()) {
                insertResult = this.insertInNode(value, nodeID, valueIdx, node);
            } else {
                Node childNode = node.getChildNode(valueIdx);
                insertResult = this.insertInTree(value, nodeID, childNode);
                childNode.release();
                if (insertResult.overflowValue != null) {
                    byte[] oldValue = insertResult.oldValue;
                    insertResult = this.insertInNode(insertResult.overflowValue, insertResult.overflowNodeID, valueIdx, node);
                    insertResult.oldValue = oldValue;
                }
            }
        }
        return insertResult;
    }

    private InsertResult insertInNode(byte[] value, int nodeID, int valueIdx, Node node) throws IOException {
        InsertResult insertResult = new InsertResult();
        if (node.isFull()) {
            Node newNode = this.createNewNode();
            insertResult.overflowValue = node.splitAndInsert(value, nodeID, valueIdx, newNode);
            insertResult.overflowNodeID = newNode.getID();
            newNode.release();
        } else {
            node.insertValueNodeIDPair(valueIdx, value, nodeID);
        }
        return insertResult;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public byte[] remove(byte[] key) throws IOException {
        this.btreeLock.writeLock().lock();
        try {
            byte[] result = null;
            Node rootNode = this.readRootNode();
            if (rootNode != null) {
                result = this.removeFromTree(key, rootNode);
                if (rootNode.isEmpty()) {
                    if (rootNode.isLeaf()) {
                        this.rootNodeID = 0;
                    } else {
                        this.rootNodeID = rootNode.getChildNodeID(0);
                        rootNode.setChildNodeID(0, 0);
                    }
                    this.writeFileHeader();
                    if (this.height >= 0) {
                        --this.height;
                    }
                }
                rootNode.release();
            }
            byte[] byArray = result;
            return byArray;
        }
        finally {
            this.btreeLock.writeLock().unlock();
        }
    }

    private byte[] removeFromTree(byte[] key, Node node) throws IOException {
        byte[] value = null;
        int valueIdx = node.search(key);
        if (valueIdx >= 0) {
            if (node.isLeaf()) {
                value = node.removeValueRight(valueIdx);
            } else {
                value = node.getValue(valueIdx);
                Node leftChildNode = node.getChildNode(valueIdx);
                byte[] largestValue = this.removeLargestValueFromTree(leftChildNode);
                node.setValue(valueIdx, largestValue);
                this.balanceChildNode(node, leftChildNode, valueIdx);
                leftChildNode.release();
            }
        } else if (!node.isLeaf()) {
            valueIdx = -valueIdx - 1;
            Node childNode = node.getChildNode(valueIdx);
            value = this.removeFromTree(key, childNode);
            this.balanceChildNode(node, childNode, valueIdx);
            childNode.release();
        }
        return value;
    }

    private byte[] removeLargestValueFromTree(Node node) throws IOException {
        int nodeValueCount = node.getValueCount();
        if (node.isLeaf()) {
            if (node.isEmpty()) {
                throw new IllegalArgumentException("Trying to remove largest value from an empty node in " + this.getFile());
            }
            return node.removeValueRight(nodeValueCount - 1);
        }
        Node childNode = node.getChildNode(nodeValueCount);
        byte[] value = this.removeLargestValueFromTree(childNode);
        this.balanceChildNode(node, childNode, nodeValueCount);
        childNode.release();
        return value;
    }

    private void balanceChildNode(Node parentNode, Node childNode, int childIdx) throws IOException {
        if (childNode.getValueCount() < this.minValueCount) {
            Node rightSibling;
            Node node = rightSibling = childIdx < parentNode.getValueCount() ? parentNode.getChildNode(childIdx + 1) : null;
            if (rightSibling != null && rightSibling.getValueCount() > this.minValueCount) {
                parentNode.rotateLeft(childIdx, childNode, rightSibling);
            } else {
                Node leftSibling;
                Node node2 = leftSibling = childIdx > 0 ? parentNode.getChildNode(childIdx - 1) : null;
                if (leftSibling != null && leftSibling.getValueCount() > this.minValueCount) {
                    parentNode.rotateRight(childIdx, leftSibling, childNode);
                } else if (leftSibling != null) {
                    leftSibling.mergeWithRightSibling(parentNode.removeValueRight(childIdx - 1), childNode);
                } else {
                    childNode.mergeWithRightSibling(parentNode.removeValueRight(childIdx), rightSibling);
                }
                if (leftSibling != null) {
                    leftSibling.release();
                }
            }
            if (rightSibling != null) {
                rightSibling.release();
            }
        }
    }

    public void clear() throws IOException {
        this.btreeLock.writeLock().lock();
        try {
            this.nodeCache.clear();
            this.nioFile.truncate(16L);
            if (this.rootNodeID != 0) {
                this.rootNodeID = 0;
                this.writeFileHeader();
            }
            this.allocatedNodesList.clear();
        }
        finally {
            this.btreeLock.writeLock().unlock();
        }
    }

    private Node createNewNode() throws IOException {
        int newNodeID = this.allocatedNodesList.allocateNode();
        Node node = new Node(newNodeID, this);
        node.use();
        this.nodeCache.put(node);
        return node;
    }

    Node readRootNode() throws IOException {
        if (this.rootNodeID > 0) {
            return this.readNode(this.rootNodeID);
        }
        return null;
    }

    Node readNode(int id) throws IOException {
        if (id <= 0) {
            throw new IllegalArgumentException("id must be larger than 0, is: " + id + " in " + this.getFile());
        }
        return this.nodeCache.readAndUse(id);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void releaseNode(Node node) throws IOException {
        if (node.isEmpty() && node.isLeaf() && this.nodeCache.discardEmptyUnused(node.getID())) {
            AllocatedNodesList allocatedNodesList = this.allocatedNodesList;
            synchronized (allocatedNodesList) {
                this.allocatedNodesList.freeNode(node.getID());
                int maxNodeID = this.allocatedNodesList.getMaxNodeID();
                if (node.getID() > maxNodeID) {
                    this.nioFile.truncate(this.nodeID2offset(maxNodeID) + (long)this.nodeSize);
                }
            }
        } else {
            this.nodeCache.release(node, this.forceSync);
        }
    }

    private void writeFileHeader() throws IOException {
        ByteBuffer buf = ByteBuffer.allocate(16);
        buf.put(MAGIC_NUMBER);
        buf.put((byte)1);
        buf.putInt(this.blockSize);
        buf.putInt(this.valueSize);
        buf.putInt(this.rootNodeID);
        buf.rewind();
        this.nioFile.write(buf, 0L);
    }

    long nodeID2offset(int id) {
        return (long)this.blockSize * (long)id;
    }

    private int offset2nodeID(long offset) {
        return (int)(offset / (long)this.blockSize);
    }

    public void print(PrintStream out) throws IOException {
        out.println("---contents of BTree file---");
        out.println("Stored parameters:");
        out.println("block size   = " + this.blockSize);
        out.println("value size   = " + this.valueSize);
        out.println("root node ID = " + this.rootNodeID);
        out.println();
        out.println("Derived parameters:");
        out.println("slot size       = " + this.slotSize);
        out.println("branch factor   = " + this.branchFactor);
        out.println("min value count = " + this.minValueCount);
        out.println("node size       = " + this.nodeSize);
        out.println();
        int nodeCount = 0;
        int valueCount = 0;
        ByteBuffer buf = ByteBuffer.allocate(this.nodeSize);
        for (long offset = (long)this.blockSize; offset < this.nioFile.size(); offset += (long)this.blockSize) {
            this.nioFile.read(buf, offset);
            buf.rewind();
            int nodeID = this.offset2nodeID(offset);
            int count = buf.getInt();
            ++nodeCount;
            valueCount += count;
            out.print("node " + nodeID + ": ");
            out.print("count=" + count + " ");
            byte[] value = new byte[this.valueSize];
            for (int i = 0; i < count; ++i) {
                out.print(buf.getInt());
                buf.get(value);
                out.print("[" + ByteArrayUtil.toHexString(value) + "]");
            }
            out.println(buf.getInt());
            buf.clear();
        }
        out.println("#nodes          = " + nodeCount);
        out.println("#values         = " + valueCount);
        out.println("---end of BTree file---");
    }

    private static class InsertResult {
        byte[] oldValue = null;
        byte[] overflowValue = null;
        int overflowNodeID = 0;

        private InsertResult() {
        }
    }

    private static class PathSegment {
        public final int valueIndex;
        public final int valueCount;

        public PathSegment(int valueIndex, int valueCount) {
            this.valueIndex = valueIndex;
            this.valueCount = valueCount;
        }

        public int getMinValueIndex() {
            if (this.valueIndex < 0) {
                return -this.valueIndex - 1;
            }
            return this.valueIndex;
        }

        public int getMaxValueIndex() {
            if (this.valueIndex < 0) {
                return -this.valueIndex - 2;
            }
            return this.valueIndex;
        }

        public String toString() {
            return this.valueIndex + ":" + this.valueCount;
        }
    }
}

