/**
 * Copyright (c) 2015 Codetrails GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.epp.logging.aeri.core;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.ImmutableList.of;
import static org.eclipse.e4.core.contexts.ContextInjectionFactory.invoke;
import static org.eclipse.epp.logging.aeri.core.Constants.*;
import static org.eclipse.epp.logging.aeri.core.l10n.LogMessages.WARN_AERI_FAILURE;

import org.eclipse.e4.core.contexts.ContextInjectionFactory;
import org.eclipse.e4.core.contexts.IEclipseContext;
import org.eclipse.e4.core.di.annotations.Execute;
import org.eclipse.e4.core.internal.contexts.EclipseContext;
import org.eclipse.e4.core.internal.contexts.IContextDisposalListener;
import org.eclipse.epp.logging.aeri.core.handlers.ResetSendModeHandler;
import org.eclipse.epp.logging.aeri.core.handlers.SetSendModeHandler;
import org.eclipse.epp.logging.aeri.core.util.Logs;
import org.eclipse.jdt.annotation.Nullable;

@SuppressWarnings("restriction")
public class SystemControl {

    private static final IEclipseContext SYSTEM_CONTEXT = init();
    private static volatile boolean DISPOSED;

    private static EclipseContext init() {
        EclipseContext local = new EclipseContext(null);
        local.set(EclipseContext.DEBUG_STRING, "AERI System Context");
        local.set(ISystemSettings.class, IModelFactory.eINSTANCE.createSystemSettings());

        // register the default handlers
        for (Class<?> clazz : of(SetSendModeHandler.class, ResetSendModeHandler.class)) {
            Object handler = ContextInjectionFactory.make(clazz, local);
            local.declareModifiable(clazz);
            local.set(clazz.getName(), handler);
        }

        // respond to disposal events:
        local.notifyOnDisposal(new IContextDisposalListener() {

            @Override
            public void disposed(IEclipseContext context) {
                setDisposed(true);
            }
        });
        return local;
    }

    /**
     * Returns the DI context used by the error reporting system. Applications may set the parent context of the returned
     * {@link IEclipseContext} on system initialization.
     *
     * @see IEclipseContext#setParent(IEclipseContext)
     */
    public static IEclipseContext getSystemContext() {
        return SYSTEM_CONTEXT;
    }

    public static ISystemSettings getSystemSettings() {
        return checkNotNull(get(ISystemSettings.class));
    }

    @SuppressWarnings("unchecked")
    @Nullable
    public static <T> T get(String key) {
        return (T) SYSTEM_CONTEXT.get(key);
    }

    @Nullable
    public static <T> T get(Class<T> key) {
        return SYSTEM_CONTEXT.get(key);
    }

    public static boolean isActive() {
        try {
            if (Boolean.getBoolean(SYSPROP_DISABLE_AERI) || Boolean.getBoolean(SYSPROP_DISABLE_AERI_V1) || isDisposed()) {
                return false;
            }
            SendMode mode = (SendMode) executeHandler(ResetSendModeHandler.class);
            switch (mode) {
            case NEVER:
                return false;
            case NOTIFY:
            case BACKGROUND:
            default:
                return true;
            }
        } catch (Exception e) {
            Logs.log(WARN_AERI_FAILURE, e, e.getMessage());
            return false;
        }
    }

    public static boolean isDebug() {
        return Constants.DEBUG || getSystemSettings().isDebugEnabled();
    }

    public static void registerHandlers(Class<?>... handlers) {
        for (Class<?> clazz : handlers) {
            Object handler = ContextInjectionFactory.make(clazz, SYSTEM_CONTEXT);
            SYSTEM_CONTEXT.declareModifiable(clazz);
            SYSTEM_CONTEXT.set(clazz.getName(), handler);
        }
    }

    /**
     * Looks up a handler from the system context and executes its @Execute method.
     *
     * @param clazz
     *            the class to instantiate
     * @return the return value of the handler's execute method - or null if void.
     */
    @Nullable
    public static Object executeHandler(Class<?> clazz) {
        Object handler = SYSTEM_CONTEXT.get(clazz);
        return ContextInjectionFactory.invoke(handler, Execute.class, SYSTEM_CONTEXT);
    }

    @Nullable
    public static Object executeHandler(Class<?> clazz, IEclipseContext localContext) {
        Object handler = SYSTEM_CONTEXT.get(clazz);
        return invoke(handler, Execute.class, SYSTEM_CONTEXT, localContext, null);
    }

    private static void setDisposed(boolean value) {
        DISPOSED = value;
    }

    private static boolean isDisposed() {
        return DISPOSED;
    }
}
