/**
 * Copyright (c) 2010-2012, Abel Hegedus, Istvan Rath and Daniel Varro
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   Abel Hegedus - initial API and implementation
 */
package org.eclipse.viatra.query.testing.core;

import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import com.google.inject.Injector;
import java.util.List;
import java.util.Set;
import org.eclipse.core.resources.IFile;
import org.eclipse.emf.common.notify.Notifier;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.viatra.query.patternlanguage.emf.helper.PatternLanguageHelper;
import org.eclipse.viatra.query.patternlanguage.emf.specification.SpecificationBuilder;
import org.eclipse.viatra.query.patternlanguage.emf.vql.Pattern;
import org.eclipse.viatra.query.patternlanguage.emf.vql.PatternModel;
import org.eclipse.viatra.query.runtime.api.IPatternMatch;
import org.eclipse.viatra.query.runtime.api.IQuerySpecification;
import org.eclipse.viatra.query.runtime.api.ViatraQueryEngine;
import org.eclipse.viatra.query.runtime.api.ViatraQueryMatcher;
import org.eclipse.viatra.query.runtime.emf.EMFScope;
import org.eclipse.viatra.query.runtime.registry.IDefaultRegistryView;
import org.eclipse.viatra.query.runtime.registry.QuerySpecificationRegistry;
import org.eclipse.viatra.query.testing.core.XmiModelUtil;
import org.eclipse.viatra.query.testing.snapshot.MatchSetRecord;
import org.eclipse.viatra.query.testing.snapshot.QuerySnapshot;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * Helper methods for loading models from files or URIs.
 */
@SuppressWarnings("all")
public class ModelLoadHelper {
  /**
   * Load an instance EMF model from the given file to a new resource set.
   */
  public Resource loadModelFromFile(final IFile file) {
    return this.loadModelFromUri(file.getFullPath().toString());
  }
  
  /**
   * Load an instance EMF model from the given platform URI to a new resource set.
   */
  public Resource loadModelFromUri(final String platformUri) {
    Resource _xblockexpression = null;
    {
      final ResourceSetImpl resourceSet = new ResourceSetImpl();
      _xblockexpression = this.loadAdditionalResourceFromUri(resourceSet, platformUri);
    }
    return _xblockexpression;
  }
  
  /**
   * Load an additional resource into the resource set from a given file.
   * Works for both pattern and target model resource sets.
   */
  public Resource loadAdditionalResourceFromFile(final ResourceSet resourceSet, final IFile file) {
    return this.loadAdditionalResourceFromUri(resourceSet, file.getFullPath().toString());
  }
  
  /**
   * Load an additional resource into the resource set from a given platform URI.
   * Works for both pattern and target model resource sets.
   */
  public Resource loadAdditionalResourceFromUri(final ResourceSet resourceSet, final String platformUri) {
    Resource _xblockexpression = null;
    {
      final URI modelURI = XmiModelUtil.resolvePlatformURI(XmiModelUtil.XmiModelUtilRunningOptionEnum.BOTH, platformUri);
      Resource _xifexpression = null;
      if ((modelURI != null)) {
        _xifexpression = resourceSet.getResource(modelURI, true);
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  /**
   * Load a pattern model from the given file into a new resource set.
   */
  public PatternModel loadPatternModelFromFile(final IFile file, final Injector injector) {
    return this.loadPatternModelFromUri(file.getFullPath().toString(), injector);
  }
  
  /**
   * Load a pattern model from the given platform URI into a new resource set.
   */
  public PatternModel loadPatternModelFromUri(final String platformUri, final Injector injector, final String... additionalDependencyPlatformUris) {
    PatternModel _xblockexpression = null;
    {
      final ResourceSet resourceSet = XmiModelUtil.prepareXtextResource(injector);
      if (((additionalDependencyPlatformUris != null) && (((List<String>)Conversions.doWrapArray(additionalDependencyPlatformUris)).size() > 0))) {
        for (final String additionalUri : additionalDependencyPlatformUris) {
          this.loadAdditionalResourceFromUri(resourceSet, additionalUri);
        }
      }
      final Resource resource = this.loadAdditionalResourceFromUri(resourceSet, platformUri);
      PatternModel _xifexpression = null;
      int _size = resource.getContents().size();
      boolean _greaterThan = (_size > 0);
      if (_greaterThan) {
        PatternModel _xifexpression_1 = null;
        EObject _get = resource.getContents().get(0);
        if ((_get instanceof PatternModel)) {
          EObject _get_1 = resource.getContents().get(0);
          _xifexpression_1 = ((PatternModel) _get_1);
        }
        _xifexpression = _xifexpression_1;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  /**
   * Initialize a matcher for the pattern with the given name from the pattern model on the selected EMF root.
   */
  public ViatraQueryMatcher<? extends IPatternMatch> initializeMatcherFromModel(final PatternModel model, final ViatraQueryEngine engine, final String patternName) {
    ViatraQueryMatcher<? extends IPatternMatch> _xblockexpression = null;
    {
      final Function1<Pattern, Boolean> _function = (Pattern it) -> {
        String _fullyQualifiedName = PatternLanguageHelper.getFullyQualifiedName(it);
        return Boolean.valueOf(Objects.equal(patternName, _fullyQualifiedName));
      };
      final Iterable<Pattern> patterns = IterableExtensions.<Pattern>filter(model.getPatterns(), _function);
      int _size = IterableExtensions.size(patterns);
      boolean _equals = (_size == 1);
      Preconditions.checkState(_equals, "No pattern found with name %s", patternName);
      Set<IQuerySpecification<? extends ViatraQueryMatcher<? extends IPatternMatch>>> _registeredQuerySpecifications = engine.getRegisteredQuerySpecifications();
      final SpecificationBuilder builder = new SpecificationBuilder(_registeredQuerySpecifications);
      _xblockexpression = engine.getMatcher(builder.getOrCreateSpecification(patterns.iterator().next()));
    }
    return _xblockexpression;
  }
  
  public ViatraQueryMatcher<? extends IPatternMatch> initializeMatcherFromModel(final PatternModel model, final Notifier emfRoot, final String patternName) {
    ViatraQueryMatcher<? extends IPatternMatch> _xblockexpression = null;
    {
      EMFScope _eMFScope = new EMFScope(emfRoot);
      final ViatraQueryEngine engine = ViatraQueryEngine.on(_eMFScope);
      _xblockexpression = this.initializeMatcherFromModel(model, engine, patternName);
    }
    return _xblockexpression;
  }
  
  /**
   * Initialize a registered matcher for the pattern FQN on the selected EMF root.
   */
  public ViatraQueryMatcher<? extends IPatternMatch> initializeMatcherFromRegistry(final Notifier emfRoot, final String patternFQN) {
    ViatraQueryMatcher<? extends IPatternMatch> _xblockexpression = null;
    {
      final IDefaultRegistryView view = QuerySpecificationRegistry.getInstance().getDefaultView();
      final IQuerySpecification<?> querySpecification = view.getEntry(patternFQN).get();
      EMFScope _eMFScope = new EMFScope(emfRoot);
      _xblockexpression = querySpecification.getMatcher(ViatraQueryEngine.on(_eMFScope));
    }
    return _xblockexpression;
  }
  
  /**
   * Load the recorded match set into an existing resource set form the given file.
   */
  public QuerySnapshot loadExpectedResultsFromFile(final ResourceSet resourceSet, final IFile file) {
    return this.loadExpectedResultsFromUri(resourceSet, file.getFullPath().toString());
  }
  
  /**
   * Load the recorded match set into an existing resource set form the given platform URI.
   */
  public QuerySnapshot loadExpectedResultsFromUri(final ResourceSet resourceSet, final String platformUri) {
    QuerySnapshot _xblockexpression = null;
    {
      final Resource resource = this.loadAdditionalResourceFromUri(resourceSet, platformUri);
      QuerySnapshot _xifexpression = null;
      if ((resource != null)) {
        QuerySnapshot _xifexpression_1 = null;
        int _size = resource.getContents().size();
        boolean _greaterThan = (_size > 0);
        if (_greaterThan) {
          QuerySnapshot _xifexpression_2 = null;
          EObject _get = resource.getContents().get(0);
          if ((_get instanceof QuerySnapshot)) {
            EObject _get_1 = resource.getContents().get(0);
            _xifexpression_2 = ((QuerySnapshot) _get_1);
          }
          _xifexpression_1 = _xifexpression_2;
        }
        _xifexpression = _xifexpression_1;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  /**
   * Load the recorded match set into a new resource set form the given file.
   */
  public QuerySnapshot loadExpectedResultsFromFile(final IFile file) {
    return this.loadExpectedResultsFromUri(file.getFullPath().toString());
  }
  
  /**
   * Load the recorded match set into a new resource set form the given platform URI.
   */
  public QuerySnapshot loadExpectedResultsFromUri(final String platformUri) {
    QuerySnapshot _xblockexpression = null;
    {
      final Resource resource = this.loadModelFromUri(platformUri);
      QuerySnapshot _xifexpression = null;
      if ((resource != null)) {
        QuerySnapshot _xifexpression_1 = null;
        int _size = resource.getContents().size();
        boolean _greaterThan = (_size > 0);
        if (_greaterThan) {
          QuerySnapshot _xifexpression_2 = null;
          EObject _get = resource.getContents().get(0);
          if ((_get instanceof QuerySnapshot)) {
            EObject _get_1 = resource.getContents().get(0);
            _xifexpression_2 = ((QuerySnapshot) _get_1);
          }
          _xifexpression_1 = _xifexpression_2;
        }
        _xifexpression = _xifexpression_1;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  /**
   * Returns the match set record for a given pattern name after it loads the snapshot from the given file.
   */
  public MatchSetRecord loadExpectedResultsForPatternFromFile(final ResourceSet resourceSet, final IFile file, final String patternFQN) {
    return this.loadExpectedResultsForPatternFromUri(resourceSet, file.getFullPath().toString(), patternFQN);
  }
  
  /**
   * Returns the match set record for a given pattern name after it loads the snapshot from the given platform URI.
   */
  public MatchSetRecord loadExpectedResultsForPatternFromUri(final ResourceSet resourceSet, final String platformUri, final String patternFQN) {
    Resource _loadAdditionalResourceFromUri = this.loadAdditionalResourceFromUri(resourceSet, platformUri);
    final QuerySnapshot snapshot = ((QuerySnapshot) _loadAdditionalResourceFromUri);
    final Function1<MatchSetRecord, Boolean> _function = (MatchSetRecord it) -> {
      return Boolean.valueOf(it.getPatternQualifiedName().equals(patternFQN));
    };
    final Iterable<MatchSetRecord> matchsetrecord = IterableExtensions.<MatchSetRecord>filter(snapshot.getMatchSetRecords(), _function);
    int _size = IterableExtensions.size(matchsetrecord);
    boolean _equals = (_size == 1);
    if (_equals) {
      return matchsetrecord.iterator().next();
    }
    return null;
  }
}
