/**
 * Copyright (c) 2010-2016, Zoltan Ujhelyi, IncQuery Labs Ltd.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-v20.html.
 * 
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.viatra.query.patternlanguage.emf.types;

import com.google.inject.Inject;
import java.util.Arrays;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.viatra.query.patternlanguage.emf.helper.PatternLanguageHelper;
import org.eclipse.viatra.query.patternlanguage.emf.types.BottomTypeKey;
import org.eclipse.viatra.query.patternlanguage.emf.types.EMFTypeSystem;
import org.eclipse.viatra.query.patternlanguage.emf.types.PatternLanguageTypeRules;
import org.eclipse.viatra.query.patternlanguage.emf.types.TypeInformation;
import org.eclipse.viatra.query.patternlanguage.emf.types.judgements.TypeJudgement;
import org.eclipse.viatra.query.patternlanguage.emf.vql.AggregatedValue;
import org.eclipse.viatra.query.patternlanguage.emf.vql.BoolValue;
import org.eclipse.viatra.query.patternlanguage.emf.vql.CheckConstraint;
import org.eclipse.viatra.query.patternlanguage.emf.vql.CompareConstraint;
import org.eclipse.viatra.query.patternlanguage.emf.vql.EClassifierConstraint;
import org.eclipse.viatra.query.patternlanguage.emf.vql.EnumValue;
import org.eclipse.viatra.query.patternlanguage.emf.vql.Expression;
import org.eclipse.viatra.query.patternlanguage.emf.vql.FunctionEvaluationValue;
import org.eclipse.viatra.query.patternlanguage.emf.vql.JavaConstantValue;
import org.eclipse.viatra.query.patternlanguage.emf.vql.ListValue;
import org.eclipse.viatra.query.patternlanguage.emf.vql.NumberValue;
import org.eclipse.viatra.query.patternlanguage.emf.vql.PathExpressionConstraint;
import org.eclipse.viatra.query.patternlanguage.emf.vql.Pattern;
import org.eclipse.viatra.query.patternlanguage.emf.vql.PatternCompositionConstraint;
import org.eclipse.viatra.query.patternlanguage.emf.vql.StringValue;
import org.eclipse.viatra.query.patternlanguage.emf.vql.TypeCheckConstraint;
import org.eclipse.viatra.query.patternlanguage.emf.vql.VariableReference;
import org.eclipse.viatra.query.runtime.matchers.context.IInputKey;

/**
 * @author Zoltan Ujhelyi
 * @since 1.3
 */
@SuppressWarnings("all")
public class EMFPatternLanguageTypeRules extends PatternLanguageTypeRules {
  @Inject
  private EMFTypeSystem typeSystem;
  
  /**
   * @since 2.0
   */
  protected void _inferTypes(final EClassifierConstraint constraint, final TypeInformation information) {
    boolean _isNonSimpleConstraint = PatternLanguageHelper.isNonSimpleConstraint(constraint);
    if (_isNonSimpleConstraint) {
      return;
    }
    IInputKey _xifexpression = null;
    boolean _isValidType = this.typeSystem.isValidType(constraint.getType());
    if (_isValidType) {
      _xifexpression = this.typeSystem.extractTypeDescriptor(constraint.getType());
    } else {
      _xifexpression = BottomTypeKey.INSTANCE;
    }
    final IInputKey type = _xifexpression;
    VariableReference _var = constraint.getVar();
    TypeJudgement _typeJudgement = new TypeJudgement(_var, type);
    information.provideType(_typeJudgement);
  }
  
  /**
   * @since 2.0
   */
  protected void _inferTypes(final EnumValue reference, final TypeInformation information) {
    IInputKey _xifexpression = null;
    if ((((reference.getEnumeration() == null) || (reference.getLiteral() == null)) || (reference.getLiteral().getEEnum() == null))) {
      _xifexpression = BottomTypeKey.INSTANCE;
    } else {
      _xifexpression = this.typeSystem.classifierToInputKey(reference.getLiteral().getEEnum());
    }
    final IInputKey type = _xifexpression;
    TypeJudgement _typeJudgement = new TypeJudgement(reference, type);
    information.provideType(_typeJudgement);
  }
  
  public void inferTypes(final EObject constraint, final TypeInformation information) {
    if (constraint instanceof AggregatedValue) {
      _inferTypes((AggregatedValue)constraint, information);
      return;
    } else if (constraint instanceof BoolValue) {
      _inferTypes((BoolValue)constraint, information);
      return;
    } else if (constraint instanceof FunctionEvaluationValue) {
      _inferTypes((FunctionEvaluationValue)constraint, information);
      return;
    } else if (constraint instanceof ListValue) {
      _inferTypes((ListValue)constraint, information);
      return;
    } else if (constraint instanceof NumberValue) {
      _inferTypes((NumberValue)constraint, information);
      return;
    } else if (constraint instanceof StringValue) {
      _inferTypes((StringValue)constraint, information);
      return;
    } else if (constraint instanceof EClassifierConstraint) {
      _inferTypes((EClassifierConstraint)constraint, information);
      return;
    } else if (constraint instanceof EnumValue) {
      _inferTypes((EnumValue)constraint, information);
      return;
    } else if (constraint instanceof JavaConstantValue) {
      _inferTypes((JavaConstantValue)constraint, information);
      return;
    } else if (constraint instanceof TypeCheckConstraint) {
      _inferTypes((TypeCheckConstraint)constraint, information);
      return;
    } else if (constraint instanceof VariableReference) {
      _inferTypes((VariableReference)constraint, information);
      return;
    } else if (constraint instanceof CheckConstraint) {
      _inferTypes((CheckConstraint)constraint, information);
      return;
    } else if (constraint instanceof CompareConstraint) {
      _inferTypes((CompareConstraint)constraint, information);
      return;
    } else if (constraint instanceof PathExpressionConstraint) {
      _inferTypes((PathExpressionConstraint)constraint, information);
      return;
    } else if (constraint instanceof PatternCompositionConstraint) {
      _inferTypes((PatternCompositionConstraint)constraint, information);
      return;
    } else if (constraint instanceof Expression) {
      _inferTypes((Expression)constraint, information);
      return;
    } else if (constraint instanceof Pattern) {
      _inferTypes((Pattern)constraint, information);
      return;
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(constraint, information).toString());
    }
  }
}
