/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0, which accompanies this distribution
 * and is available at https://www.eclipse.org/legal/epl-2.0/.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 ******************************************************************************/
package org.eclipse.jpt.jpa.core.internal.facet;

import java.util.Set;
import org.eclipse.core.runtime.IProduct;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jpt.jpa.core.JpaPlatform;
import org.eclipse.jpt.jpa.db.ConnectionProfile;
import org.eclipse.jpt.jpa.db.Database;
import org.eclipse.jpt.jpa.db.SchemaContainer;
import org.eclipse.wst.common.frameworks.datamodel.DataModelPropertyDescriptor;
import org.eclipse.wst.common.frameworks.datamodel.IDataModel;

public class JpaFacetInstallDataModelProvider
	extends JpaFacetDataModelProvider
	implements JpaFacetInstallDataModelProperties
{
	
	/** used for adopter product customization */
	private static final String DISCOVER_ANNOTATED_CLASSES_DEFAULT = "discoverAnnotatedClassesDefault"; //$NON-NLS-1$
	
	/**
	 * required default constructor
	 */
	public JpaFacetInstallDataModelProvider() {
		super();
	}
	
	@Override
	public Set<String> getPropertyNames() {
		Set<String> propertyNames = super.getPropertyNames();
		propertyNames.add(USER_WANTS_TO_ADD_DB_DRIVER_JARS_TO_CLASSPATH);
		propertyNames.add(DB_DRIVER_NAME);
		return propertyNames;
	}
	
	
	// ********** properties **********
	
	protected boolean userWantsToAddDbDriverJarsToClasspath() {
		return this.getBooleanProperty(USER_WANTS_TO_ADD_DB_DRIVER_JARS_TO_CLASSPATH);
	}
	
	protected String getDriverName() {
		return (String) this.getProperty(DB_DRIVER_NAME);
	}
	
	
	// ********** enabled **********
	
	@Override
	public boolean isPropertyEnabled(String propertyName) {
		if (propertyName.equals(USER_WANTS_TO_ADD_DB_DRIVER_JARS_TO_CLASSPATH)) {
			return (this.getConnectionProfile() != null);
		}
		if (propertyName.equals(DB_DRIVER_NAME)) {
			return this.userWantsToAddDbDriverJarsToClasspath();
		}
		
		return super.isPropertyEnabled(propertyName);
	}
	
	
	// ********** defaults **********
	
	@Override
	public Object getDefaultProperty(String propertyName) {
		if (propertyName.equals(USER_WANTS_TO_ADD_DB_DRIVER_JARS_TO_CLASSPATH)) {
			return Boolean.FALSE;
		}
		if (propertyName.equals(DB_DRIVER_NAME)) {
			return this.getDefaultDriverName();
		}
		
		return super.getDefaultProperty(propertyName);
	}
	
	@Override
	protected JpaPlatform.Config getDefaultPlatformConfig() {
		return this.getJpaPlatformManager().getDefaultJpaPlatformConfig(this.getProjectFacetVersion());
	}
	
	@Override
	protected String getDefaultConnectionName() {
		return null;
	}
	
	@Override
	protected Boolean getDefaultUserWantsToOverrideDefaultCatalog() {
		return Boolean.FALSE;
	}
	
	@Override
	protected String getDefaultCatalogIdentifier() {
		Database db = this.getDatabase();
		return (db == null) ? null : db.getDefaultCatalogIdentifier();
	}
	
	@Override
	protected Boolean getDefaultUserWantsToOverrideDefaultSchema() {
		return Boolean.FALSE;
	}
	
	@Override
	protected String getDefaultSchemaIdentifier()  {
		SchemaContainer sc = this.getSchemaContainer();
		return (sc == null) ? null : sc.getDefaultSchemaIdentifier();
	}
	
	protected String getDefaultDriverName() {
		ConnectionProfile cp = this.getConnectionProfile();
		return (cp == null) ? null : cp.getDriverName();
	}
	
	@Override
	protected Boolean getDefaultDiscoverAnnotatedClasses() {
		//Check for product preference first to determine default setting
		IProduct product = Platform.getProduct();
		String discoverAnnotatedClassesDefaultPreference = (product == null) ? null : product.getProperty(DISCOVER_ANNOTATED_CLASSES_DEFAULT);
		//if no product preference is defined check the runtime for support, otherwise check the preference to determine override
		return Boolean.valueOf((discoverAnnotatedClassesDefaultPreference == null) ? 
				this.runtimeSupportsEjb30() : discoverAnnotatedClassesDefaultPreference.equals("true")); //$NON-NLS-1$
	}
	
	
	// ********** synchronize data model **********
	
	/**
	 * The specified property's value has changed to the specified value.
	 * Return whether to fire a VALUE_CHG DataModelEvent.
	 */
	@Override
	public boolean propertySet(String propertyName, Object propertyValue) {
		boolean ok = super.propertySet(propertyName, propertyValue);
		
		if (propertyName.equals(CONNECTION)) {
			// db driver
			if (propertyValue == null) {  // connection set to '<None>'
				this.setBooleanProperty(USER_WANTS_TO_ADD_DB_DRIVER_JARS_TO_CLASSPATH, false);
			}
			this.model.notifyPropertyChange(USER_WANTS_TO_ADD_DB_DRIVER_JARS_TO_CLASSPATH, IDataModel.ENABLE_CHG);
			this.model.notifyPropertyChange(DB_DRIVER_NAME, IDataModel.DEFAULT_CHG);
			this.model.notifyPropertyChange(DB_DRIVER_NAME, IDataModel.VALID_VALUES_CHG);
		}
		else if (propertyName.equals(USER_WANTS_TO_ADD_DB_DRIVER_JARS_TO_CLASSPATH)) {
			this.model.notifyPropertyChange(DB_DRIVER_NAME, IDataModel.ENABLE_CHG);
			if (this.propertyValueIsFalse(propertyValue)) {
				this.setProperty(DB_DRIVER_NAME, null);
			}
		}
		
		return ok;
	}
	
	
	// ********** property descriptors **********

	@Override
	public DataModelPropertyDescriptor[] getValidPropertyDescriptors(String propertyName) {
		if (propertyName.equals(PLATFORM)) {
			return this.buildValidPlatformDescriptors();
		}

		if (propertyName.equals(CONNECTION)) {
			return this.buildValidConnectionDescriptors();
		}

		if (propertyName.equals(DB_DRIVER_NAME)) {
			return this.buildValidDriverDescriptors();
		}

		if (propertyName.equals(USER_OVERRIDE_DEFAULT_CATALOG)) {
			return this.buildValidCatalogDescriptors();
		}

		if (propertyName.equals(USER_OVERRIDE_DEFAULT_SCHEMA)) {
			return this.buildValidSchemaDescriptors();
		}

		return super.getValidPropertyDescriptors(propertyName);
	}
	
	protected DataModelPropertyDescriptor[] buildValidDriverDescriptors() {
		return new DataModelPropertyDescriptor[] { new DataModelPropertyDescriptor(this.getDriverName()) };
	}
	
	
	// ********** validation **********
	
	@Override
	public IStatus validate(String propertyName) {
		if (propertyName.equals(USER_WANTS_TO_ADD_DB_DRIVER_JARS_TO_CLASSPATH)
				|| propertyName.equals(DB_DRIVER_NAME)) {
			return this.validateDbDriverName();
		}
		
		return super.validate(propertyName);
	}
	
	protected IStatus validateDbDriverName() {
		return OK_STATUS;
	}
}
