/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.tools.mapping.orm.ExternalAssociationOverride;
import org.eclipse.persistence.tools.mapping.orm.ExternalAttributeOverride;
import org.eclipse.persistence.tools.mapping.orm.ExternalEmbeddedMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalNoSqlField;
import org.eclipse.persistence.tools.utility.iterable.ListIterable;
import org.eclipse.persistence.tools.utility.iterable.ListListIterable;
import org.w3c.dom.Element;

/**
 * The external form for a embedded mapping, which is a child of an entity.
 *
 * @see MappedSuperClassEntity
 *
 * @version 2.5
 * @author Les Davis
 * @author Pascal Filion
 */
@SuppressWarnings("nls")
final class EmbeddedMapping extends NonTransientMapping
                            implements ExternalEmbeddedMapping {

	/**
	 * The attribute name used to store and retrieve the access property.
	 */
	static final String ACCESS = "access";

	/**
	 * The node name used to store and retrieve the {@link Element} encapsulated by this external form.
	 */
	static final String EMBEDDED = "embedded";

	/**
	 * Creates a new <code>EmbeddedMapping</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with
	 * the same type owned by the parent
	 */
	EmbeddedMapping(EmbeddableEntity parent, int index) {
		super(parent, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalAssociationOverride addAssociationOverride(int index) {
		AssociationOverride associationOverride = buildAssociationOverride(index);
		associationOverride.addSelf();
		return associationOverride;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalAttributeOverride addAttributeOverride(int index) {
		AttributeOverride attributeOverride = buildAttributeOverride(index);
		attributeOverride.addSelf();
		return attributeOverride;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalAssociationOverride> associationOverrides() {

		int count = associationOverridesSize();
		List<ExternalAssociationOverride> associationOverrides = new ArrayList<ExternalAssociationOverride>(count);

		for (int index = count; --index >= 0;) {
			ExternalAssociationOverride associationOverride = buildAssociationOverride(index);
			associationOverrides.add(0, associationOverride);
		}

		return new ListListIterable<ExternalAssociationOverride>(associationOverrides);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int associationOverridesSize() {
		return getChildrenSize(AssociationOverride.ASSOCIATION_OVERRIDE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalAttributeOverride> attributeOverrides() {

		int count = attributeOverridesSize();
		List<ExternalAttributeOverride> attributeOverrides = new ArrayList<ExternalAttributeOverride>(count);

		for (int index = count; --index >= 0;) {
			ExternalAttributeOverride attributeOverride = buildAttributeOverride(index);
			attributeOverrides.add(0, attributeOverride);
		}

		return new ListListIterable<ExternalAttributeOverride>(attributeOverrides);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int attributeOverridesSize() {
		return getChildrenSize(AttributeOverride.ATTRIBUTE_OVERRIDE);
	}

	private AssociationOverride buildAssociationOverride(int index) {
		return new AssociationOverride(this, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(NAME);
		names.add(ACCESS);
		names.add(ATTRIBUTE_TYPE);
		return names;
	}

	private AttributeOverride buildAttributeOverride(int index) {
		return new AttributeOverride(this, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(AttributeOverride.ATTRIBUTE_OVERRIDE);
		names.add(AssociationOverride.ASSOCIATION_OVERRIDE);
		names.add(Property.PROPERTY);
		names.add(AccessMethods.ACCESS_METHODS);
		return names;
	}

	private ExternalNoSqlField buildNoSqlField() {
		return new NoSqlField(this);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalAssociationOverride getAssociationOverride(int index) {

		Element element = getChild(AssociationOverride.ASSOCIATION_OVERRIDE, index);

		if (element == null) {
			return null;
		}

		return buildAssociationOverride(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalAttributeOverride getAttributeOverride(int index) {

		Element element = getChild(AttributeOverride.ATTRIBUTE_OVERRIDE, index);

		if (element == null) {
			return null;
		}

		return buildAttributeOverride(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return EMBEDDED;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getNoSqlField() {
		ExternalNoSqlField field = buildNoSqlField();
		return field.getName();
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public final boolean isEmbeddedMapping() {
		return true;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeAssociationOverride(int index) {
		AssociationOverride associationOverride = buildAssociationOverride(index);
		associationOverride.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeAttributeOverride(int index) {
		AttributeOverride attributeOverride = buildAttributeOverride(index);
		attributeOverride.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setAttributeType(String attributeType) {
		setAttribute(ATTRIBUTE_TYPE, attributeType);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setNoSqlField(String field) {
		ExternalNoSqlField noSqlField = buildNoSqlField();
		noSqlField.setName(field);
	}
}