/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.tools.mapping.orm.ExternalJoinColumn;
import org.eclipse.persistence.tools.mapping.orm.ExternalNoSqlJoinField;
import org.eclipse.persistence.tools.mapping.orm.ExternalOneToManyMapping;
import org.eclipse.persistence.tools.utility.iterable.ListIterable;
import org.eclipse.persistence.tools.utility.iterable.ListListIterable;
import org.w3c.dom.Element;

/**
 * The external form for a 1:M mapping, which is a child of an entity.
 *
 * @see MappedSuperClassEntity
 *
 * @version 2.5
 * @author Les Davis
 * @author Pascal Filion
 */
@SuppressWarnings("nls")
final class OneToManyMapping extends ObjectCollectionMapping
                             implements ExternalOneToManyMapping {

	/**
	 * The attribute name used to store and retrieve the maps-id property.
	 */
	static final String MAPS_ID = "maps-id";

	/**
	 * The node name used to store and retrieve the {@link Element} encapsulated by this external form.
	 */
	static final String ONE_TO_MANY = "one-to-many";

	/**
	 * The attribute name used to store and retrieve the orphan-removal property.
	 */
	static final String ORPHAN_REMOVAL = "orphan-removal";

	/**
	 * The attribute name used to store and retrieve the private-owned property.
	 */
	static final String PRIVATE_OWNED = "private-owned";

	/**
	 * Creates a new <code>OneToManyMapping</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	OneToManyMapping(EmbeddableEntity parent, int index) {
		super(parent, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addJoinColumn(int index, String name) {
		JoinColumn joinColumn = buildJoinColumn(JoinColumn.JOIN_COLUMN, index);
		joinColumn.addSelf();
		joinColumn.setName(name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addJoinField(int index, String name) {
		ExternalNoSqlJoinField joinField = buildJoinField(index);
		joinField.setName(name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(NAME);
		names.add(TARGET_ENTITY);
		names.add(FETCH);
		names.add(ACCESS);
		names.add(MAPPED_BY);
		names.add(MAPS_ID);
		names.add(ORPHAN_REMOVAL);
		return names;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(ORDER_BY);
		names.add(OrderColumn.ORDER_COLUMN);
		names.add(MAP_KEY);
		names.add(MAP_KEY_CLASS);
		names.add(MAP_KEY_TEMPORAL);
		names.add(MAP_KEY_ENUMERATED);
		names.add(MAP_KEY_CONVERT);
		names.add(MAP_KEY_ATTRIBUTE_OVERRIDE);
		names.add(MAP_KEY_ASSOCIATION_OVERRIDE);
		names.add(MAP_KEY_COLUMN);
		names.add(MAP_KEY_JOIN_COLUMN);
		names.add(JoinTable.JOIN_TABLE);
		names.add(JoinColumn.JOIN_COLUMN);
		names.add(CASCADE);
		names.add(PRIVATE_OWNED);
		names.add(JOIN_FETCH);
		names.add(BatchFetch.BATCH_FETCH);
		names.add(Property.PROPERTY);
		names.add(AccessMethods.ACCESS_METHODS);
		return names;
	}

	private ExternalNoSqlJoinField buildJoinField(int index) {
		return new NoSqlJoinField(this, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return ONE_TO_MANY;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Boolean getId() {
		// Should not be used on this mapping
		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalJoinColumn getJoinColumn(int index) {

		Element element = getChild(JoinColumn.JOIN_COLUMN, index);

		if (element == null) {
			return null;
		}

		return buildJoinColumn(JoinColumn.JOIN_COLUMN, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getMappedByMappingName() {
		return getAttribute(MAPPED_BY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getMapsId() {
		return getAttribute(MAPS_ID);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Boolean isOrphanRemoval() {
		return getBooleanAttribute(ORPHAN_REMOVAL);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Boolean isPrivateOwned() {
		return hasChild(PRIVATE_OWNED);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalJoinColumn> joinColumns() {

		int count = joinColumnsSize();
		List<ExternalJoinColumn> joinColumns = new ArrayList<ExternalJoinColumn>(count);

		for (int index = count; --index >= 0;) {
			ExternalJoinColumn joinColumn = buildJoinColumn(JoinColumn.JOIN_COLUMN, index);
			joinColumns.add(0, joinColumn);
		}

		return new ListListIterable<ExternalJoinColumn>(joinColumns);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int joinColumnsSize() {
		return getChildrenSize(JoinColumn.JOIN_COLUMN);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalNoSqlJoinField> joinFields() {

		int count = joinFieldSize();
		List<ExternalNoSqlJoinField> joinFields = new ArrayList<ExternalNoSqlJoinField>(count);

		for (int index = count; --index >= 0;) {
			ExternalNoSqlJoinField joinField = buildJoinField(index);
			joinFields.add(0, joinField);
		}

		return new ListListIterable<ExternalNoSqlJoinField>(joinFields);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int joinFieldSize() {
		return getChildrenSize(NoSqlJoinField.JOIN_FIELD);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeJoinColumn(int index) {
		JoinColumn joinColumn = buildJoinColumn(JoinColumn.JOIN_COLUMN, index);
		joinColumn.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeJoinField(int index) {
		removeChild(NoSqlJoinField.JOIN_FIELD, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setId(Boolean id) {
		// Should not be used on this mapping
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMappedByMappingName(String name) {
		setAttribute(MAPPED_BY, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMapsId(String mapsId) {
		setAttribute(MAPS_ID, mapsId);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setOrphanRemoval(Boolean removeOrphans) {
		setAttribute(ORPHAN_REMOVAL, removeOrphans);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setPrivateOwned(Boolean privateOwned) {
		if (privateOwned == Boolean.TRUE) {
			addChild(PRIVATE_OWNED);
		}
		else {
			removeChild(PRIVATE_OWNED);
		}
	}
}