/*******************************************************************************
 * Copyright (c) 2008, 2012 Oracle. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.gen.db;

/**
 * Database
 * <p>
 * Provisional API: This interface is part of an interim API that is still under development and
 * expected to change significantly before reaching stability. It is available at this early stage
 * to solicit feedback from pioneering adopters on the understanding that any code that uses this
 * API will almost certainly be broken (repeatedly) as the API evolves.
 *
 * @version 2.5
 */
public interface Database extends SchemaContainer {

	/**
	 * Returns the implementation specific database.
	 */

	//TODO Many need a hook here for implementation specific DB object
	//ImplementationDatabase getImplemenationDatabase();


	// ********** properties **********

	/**
	 * Returns the name of the database's vendor.
	 */
	String getVendorName();

	/**
	 * Returns the database's version.
	 */
	String getVersion();


	// ********** catalogs **********

	/**
	 * Returns whether the database supports catalogs. If it does, all database
	 * objects are contained by the database's catalogs; otherwise all database
	 * objects are contained by the database's schemata.
	 * <br>
	 * Practically speaking:<ul>
	 *     <li>If {@link #supportsCatalogs()} returns <code>true</code><ul>
	 *         <li>{@link #getCatalogs()} returns catalogs that contain the database's schemata
	 *         <li>{@link #getSchemata()} returns an empty iterable
	 *     </ul>
	 *     <li>else<ul>
	 *         <li>{@link #getCatalogs()} returns an empty iterable
	 *         <li>{@link #getSchemata()} returns the database's schemata
	 *     </ul>
	 * </ul>
	 * This is complicated by the presence of a "default" catalog that clients can
	 * use to allow the specification of a catalog to be optional; but clients
	 * must manage this explicitly.
	 * @see #getCatalogs()
	 * @see #getSchemata()
	 */
	boolean supportsCatalogs();

	/**
	 * Returns the database's catalogs.
	 * Returns an empty iterable if the database does not support catalogs.
	 * @see #supportsCatalogs()
	 */
	Iterable<Catalog> getCatalogs();

	/**
	 * Returns the number of catalogs the database contains.
	 * Returns zero if the database does not support catalogs.
	 * @see #supportsCatalogs()
	 */
	int getCatalogsSize();

	/**
	 * Returns the database's catalog names, sorted.
	 * Returns an empty iterable if the database does not support catalogs.
	 * This is useful when the user is selecting a catalog from a read-only
	 * combo-box (e.g. in a wizard).
	 * @see #getSortedCatalogIdentifiers()
	 * @see #getCatalogNamed(String)
	 */
	Iterable<String> getSortedCatalogNames();

	/**
	 * Returns the catalog with the specified name. The name must be an exact match
	 * of the catalog's name.
	 * Returns <code>null</code> if the database does not support catalogs.
	 * @see #supportsCatalogs()
	 * @see #getSortedCatalogNames()
	 * @see #getCatalogForIdentifier(String)
	 */
	Catalog getCatalogNamed(String name);

	/**
	 * Returns the database's catalog identifiers, sorted by name.
	 * Returns an empty iterable if the database does not support catalogs.
	 * This is useful when the user is selecting an identifier that will be
	 * placed in a text file (e.g. in a Java annotation).
	 * @see #getSortedCatalogNames()
	 * @see #getCatalogForIdentifier(String)
	 */
	Iterable<String> getSortedCatalogIdentifiers();

	/**
	 * Returns the catalog for the specified identifier. The identifier should
	 * be an SQL identifier (i.e. quoted when case-sensitive or containing
	 * special characters, unquoted otherwise).
	 * Returns <code>null</code> if the database does not support catalogs.
	 * @see #supportsCatalogs()
	 * @see #getSortedCatalogIdentifiers()
	 * @see #getCatalogNamed(String)
	 */
	Catalog getCatalogForIdentifier(String identifier);

	/**
	 * Returns the database's default catalog, as defined by the database vendor.
	 * In most cases the default catalog's name will match the user name.
	 * Returns <code>null</code> if the database does not support catalogs or
	 * if the default catalog does not exist (e.g. the database has no catalog
	 * whose name matches the user name).
	 * @see #supportsCatalogs()
	 * @see #getDefaultCatalogIdentifier()
	 */
	Catalog getDefaultCatalog();

	/**
	 * Returns the database's default catalog identifier.
	 * The database may or may not have a catalog with a matching name.
	 * @see #supportsCatalogs()
	 * @see #getDefaultCatalog()
	 */
	String getDefaultCatalogIdentifier();


	// ********** utility methods **********

	/**
	 * Returns from the specified list of tables the
	 * table identified by the specified identifier.
	 * The identifier should be an SQL identifier (i.e. delimited as
	 * appropriate).
	 */
	Table selectTableForIdentifier(Iterable<Table> tables, String identifier);

	/**
	 * Convert the specified name to a database-appropriate SQL identifier
	 * (i.e. delimit the name as appropriate).
	 */
	// it seems we don't need database object-specific conversions here;
	// i.e. separate methods for tables, columns, etc.
	String convertNameToIdentifier(String name);
}