/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.tools.mapping.orm.ExternalAttributeOverride;
import org.eclipse.persistence.tools.mapping.orm.ExternalEntityColumn;
import org.w3c.dom.Element;

/**
 * The external form of an attribute override, which is a child of an embedded mapping.
 *
 * @see ElementCollectionMapping
 * @see EmbeddedIDMapping
 * @see EmbeddedMapping
 * @see Entity
 *
 * @version 2.5
 * @author Les Davis
 * @author Pascal Filion
 */
final class AttributeOverride extends AbstractExternalForm
                              implements ExternalAttributeOverride {

	/**
	 * The position of the element within the list of children with the same type owned by the parent.
	 */
	private int index;

	/**
	 * Creates a new <code>AttributeOverride</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the external form of the mapping in the list of children
	 */
	AttributeOverride(ElementCollectionMapping parent, int index) {
		super(parent);
		this.index = index;
	}

	/**
	 * Creates a new <code>AttributeOverride</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	AttributeOverride(EmbeddedIDMapping parent, int index) {
		super(parent);
		this.index = index;
	}

	/**
	 * Creates a new <code>AttributeOverride</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the external form of the mapping in the list of children
	 */
	AttributeOverride(EmbeddedMapping parent, int index) {
		super(parent);
		this.index = index;
	}

	/**
	 * Creates a new <code>AttributeOverride</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the external form of the mapping in the list of children
	 */
	AttributeOverride(Entity parent, int index) {
		super(parent);
		this.index = index;
	}

	/**
	 * Creates a new <code>AttributeOverride</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the external form of the mapping in the list of children
	 */
	AttributeOverride(ObjectCollectionMapping parent, int index) {
		super(parent);
		this.index = index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected Element addSelf(String elementName, List<String> elementNamesOrder) {
		return addChild(getParent(), elementName, index, elementNamesOrder);
	}

	private EntityColumn buildColumn(int index) {
		return new EntityColumn(this, index);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(DESCRIPTION);
		return names;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntityColumn getColumn() {

		Element element = getChild(EntityColumn.COLUMN, index);

		if (element != null) {
			return buildColumn(index);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getDescription() {
		return getChildTextNode(DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Element getElement() {
		return getChild(getParent(), getElementName(), index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return ATTRIBUTE_OVERRIDE;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getIndex() {
		return index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getName() {
		return getAttribute(NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeColumn(int index) {
		EntityColumn column = buildColumn(index);
		column.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeSelf() {
		removeChild(getParent(), getElementName(), index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setColumn(String name) {

		EntityColumn column = buildColumn(index);

		if (name == null) {
			column.removeSelf();
		}
		else {
			column.addSelf();
			column.setName(name);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDescription(String description) {
		updateChildTextNode(DESCRIPTION, description);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setName(String name) {
		setAttribute(NAME, name);
	}
}