/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.annotations.OptimisticLockingType;
import org.eclipse.persistence.tools.mapping.orm.ExternalEntityColumn;
import org.eclipse.persistence.tools.mapping.orm.ExternalOptimisticLocking;
import org.eclipse.persistence.tools.utility.iterable.ListIterable;
import org.eclipse.persistence.tools.utility.iterable.ListListIterable;
import org.w3c.dom.Element;

/**
 * The external form of a optimistic locking, which is a child of an entity.
 *
 * @see MappedSuperClassEntity
 *
 * @version 2.5
 * @author Les Davis
 */
@SuppressWarnings("nls")
final class OptimisticLocking extends AbstractExternalForm
                              implements ExternalOptimisticLocking {

	/**
	 * The attribute name used to store and retrieve the cascade property.
	 */
	static final String CASCADE = "cascade";

	/**
	 * The attribute name used to store and retrieve the optimistic-locking property.
	 */
	static final String OPTIMISTIC_LOCKING = "optimistic-locking";

	/**
	 * The element name used to store and retrieve the base child node.
	 */
	static final String SELECTED_COLUMN = "selected-column";

	/**
	 * The attribute name used to store and retrieve the type property.
	 */
	static final String TYPE = "type";

	/**
	 * Creates a new <code>OptimisticLocking</code>.
	 *
	 * @param parent The parent of this external form
	 */
	OptimisticLocking(MappedSuperClassEntity parent) {
		super(parent);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addColumn(int index, String columnName) {
		EntityColumn column = buildColumn(index);
		column.addSelf();
		column.setName(columnName);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(TYPE);
		names.add(CASCADE);
		return names;
	}

	private EntityColumn buildColumn(int index) {
		return new EntityColumn(this, SELECTED_COLUMN, index);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(SELECTED_COLUMN);
		return names;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalEntityColumn> columns() {

		int count = columnsSize();
		List<ExternalEntityColumn> columns = new ArrayList<ExternalEntityColumn>(count);

		for (int index = count; --index >= 0;) {
			ExternalEntityColumn column = buildColumn(index);
			columns.add(0, column);
		}

		return new ListListIterable<ExternalEntityColumn>(columns);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int columnsSize() {
		return getChildrenSize(SELECTED_COLUMN);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntityColumn getColumn(int index) {

		Element element = getChild(SELECTED_COLUMN, index);

		if (element == null) {
			return null;
		}

		return buildColumn(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return OPTIMISTIC_LOCKING;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public OptimisticLockingType getOptimisticLockingType() {
		return getEnumAttribute(TYPE, OptimisticLockingType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeAllColumns() {
		removeChildren(SELECTED_COLUMN);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeColumn(int index) {
		EntityColumn column = buildColumn(index);
		column.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setOptimisticLockingType(OptimisticLockingType type) {

		setAttribute(TYPE, type);

		if (type == null) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setShouldCascade(Boolean value) {
		setAttribute(CASCADE, value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Boolean shouldCascade() {
		return getBooleanAttribute(CASCADE);
	}
}