/*******************************************************************************
 * Copyright (c) 2012, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.gen;

import org.eclipse.persistence.tools.utility.iterable.IterableTools;

/**
 * This enumeration defines the database type supported by dynamic entity generation.
 * <p>
 * Provisional API: This interface is part of an interim API that is still under development and
 * expected to change significantly before reaching stability. It is available at this early stage
 * to solicit feedback from pioneering adopters on the understanding that any code that uses this
 * API will almost certainly be broken (repeatedly) as the API evolves.<p>
 *
 * @see RelationalEntityGenerator
 *
 * @version 2.6
 * @author John Bracken
 */
@SuppressWarnings("nls")
public enum DatabaseType {

	/**
	 * The constant for MySQL database driver.
	 */
	MySql("MySQL", "com.mysql.jdbc.Driver"),

	/**
	 * The constant for Oracle database driver.
	 */
	Oracle("Oracle", "oracle.jdbc.driver.OracleDriver");

	/**
	 * The fully qualified class name of the driver.
	 */
	private String driver;

	/**
	 * Database type identifier.
	 */
	private String typeName;

	/**
	 * Creates a new <code>DatabaseType</code>.
	 */
	private DatabaseType(String typeName, String driver) {
		this.typeName = typeName;
		this.driver = driver;
	}

	/**
	 * Returns the list of all relational databases.
	 *
	 * @return The list of unique constants
	 */
	public static Iterable<DatabaseType> types() {
		return IterableTools.iterable(values());
	}

	/**
	 * Retrieves the enumeration constant for the given value. If the value is not known, then
	 * <code>null</code> will be returned.
	 *
	 * @param value The value to retrieve its constant version
	 * @return The constant version of the given value
	 */
	public static DatabaseType value(String value) {
		for (DatabaseType type : values()) {
			if (type.typeName.equals(value)) {
				return type;
			}
		}
		return null;
	}

	/**
	 * Returns the driver class specific to this database type.
	 *
	 * @return The driver class for this database type
	 */
	public String getDatabaseDriver() {
		return driver;
	}

	/**
	 * Returns the type name of the database.
	 *
	 * @return The name of the database
	 */
	public String getName() {
		return this.typeName;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString() {
		return this.typeName;
	}
}