/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.tests.dom.orm;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Random;
import org.eclipse.persistence.tools.mapping.ExternalForm;
import org.eclipse.persistence.tools.mapping.orm.ExternalEntity;
import org.eclipse.persistence.tools.mapping.orm.ExternalMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalORMConfiguration;
import org.eclipse.persistence.tools.mapping.tests.AbstractExternalFormTests;
import org.eclipse.persistence.tools.utility.ObjectTools;
import org.junit.After;
import org.w3c.dom.Node;

/**
 * This root hierarchy of the unit-test testing the behavior of {@link ExternalMapping}.
 *
 * @version 2.6
 */
@SuppressWarnings("nls")
public abstract class MappingTests<T extends ExternalMapping> extends AbstractExternalFormTests<T> {

	/**
	 * The cached name used when creating or accessing the mapping from its parent entity.
	 */
	private String mappingName;

	/**
	 * Adds the {@link AttributeNodeTester} for each attribute composing the mapping.
	 *
	 * @param tester The {@link RootNodeTester} is used to adds the appropriate testers to test the
	 * property of a node; i.e. its attributes and child nodes
	 */
	abstract void addAttributeTesters(RootNodeTester tester);

	/**
	 * Adds the {@link NodeTester} for each child elements composing the mapping.
	 *
	 * @param tester The {@link RootNodeTester} is used to adds the appropriate testers to test the
	 * property of a node; i.e. its attributes and child nodes
	 */
	abstract void addChildTesters(RootNodeTester tester);

	/**
	 * Adds to the given {@link ExternalEntity} the mapping being tested.
	 *
	 * @param entity The {@link ExternalEntity} representing the managed type
	 */
	abstract T addMapping(ExternalEntity entity);

	final AttributeNodeTester<T, String> buildAttributeTypeTester() {
		return new AttributeNodeTester<T, String>() {
			@Override
			public boolean doesNodeAlreadyExist() {
				return false;
			}
			@Override
			public String getDefaultValue() {
				return mappingName() + "_type_default";
			}
			@Override
			public String getNodeName() {
				return ExternalMapping.ATTRIBUTE_TYPE;
			}
			@Override
			public String getValue(T form) {
				return form.getAttributeType();
			}
			@Override
			public String getValue1() {
				return mappingName() + "_type_one";
			}
			@Override
			public String getValue2() {
				return mappingName() + "_type_two";
			}
			@Override
			public boolean isNodeDeletedWithNullValue() {
				return true;
			}
			@Override
			public boolean isNullAllowed() {
				return true;
			}
			@Override
			public void setValue(T form, String value) {
				form.setAttributeType(value);
			}
			@Override
			public String toString(String value) {
				return value;
			}
		};
	}

	private ExternalFormBuilder<T> buildMappingBuilder() {
		return new ExternalFormBuilder<T>() {
			@Override
			public T buildExternalForm() throws IOException {
				ExternalORMConfiguration orm = ORMConfigurationTests.buildExternalForm();
				ExternalEntity entity = orm.addEntity(entityClassName());
				return addMapping(entity);
			}
			@Override
			public T buildExternalForm(ExternalForm parentForm) {
				return addMapping((ExternalEntity) parentForm);
			}
			@Override
			public int getDefaultAttributeCount() {
				return 1;
			}
			@Override
			public Node getNode(T form) {
				return ObjectTools.execute(form, "getElement");
			}
			@Override
			public String getNodeName() {
				return elementName();
			}
			@Override
			public List<String> getTreeNodeNames() {
				List<String> names = new ArrayList<String>();
				names.add(ExternalORMConfiguration.ENTITY_MAPPINGS);
				names.add(ExternalEntity.ENTITY);
				names.add(ExternalMapping.ATTRIBUTES);
				names.add(elementName());
				return names;
			}
		};
	}

	final AttributeNodeTester<T, String> buildNameTester() {
		return new AttributeNodeTester<T, String>() {
			@Override
			public boolean doesNodeAlreadyExist() {
				return true;
			}
			@Override
			public String getDefaultValue() {
				return mappingName();
			}
			@Override
			public String getNodeName() {
				return ExternalMapping.NAME;
			}
			@Override
			public String getValue(T form) {
				return form.getName();
			}
			@Override
			public String getValue1() {
				return mappingName() + "_one";
			}
			@Override
			public String getValue2() {
				return mappingName() + "_two";
			}
			@Override
			public boolean isNodeDeletedWithNullValue() {
				return false;
			}
			@Override
			public boolean isNullAllowed() {
				return false;
			}
			@Override
			public void setValue(T form, String value) {
				form.setName(value);
			}
			@Override
			public String toString(String value) {
				return value;
			}
		};
	}

	/**
	 * The name of the element representing the mapping being tested.
	 *
	 * @return The mapping name
	 */
	abstract String elementName();

	private String entityClassName() {
		return "org.eclipse.persistence.tool.mappings.tests.Entity";
	}

	/**
	 * Returns a uniquely generated name to be used as the mapping name.
	 *
	 * @return A unique string that is the same throughout the execution of this unit-test
	 */
	final String mappingName() {
		if (mappingName == null) {
			return "mapping_name_" + new Random().nextInt();
		}
		return mappingName;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected final void populate(RootNodeTester tester) {

		tester.setBuilder(buildMappingBuilder());

		addAttributeTesters(tester);
		addChildTesters(tester);
	}

	@After
	public void tearDown() {
		mappingName = null;
	}
}