/*******************************************************************************
 * Copyright (c) 2012, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.gen.nosql.mongo.meta;

import java.util.HashMap;
import java.util.Map;

/**
 * Metadata class that describes a NoSql table which may also be referred
 * to as a Collection or Map depending on the database type.
 * <p>
 * Provisional API: This interface is part of an interim API that is still under development and
 * expected to change significantly before reaching stability. It is available at this early stage
 * to solicit feedback from pioneering adopters on the understanding that any code that uses this
 * API will almost certainly be broken (repeatedly) as the API evolves.<p>
 *
 * @author John Bracken
 * @version 2.6
 */
public final class CollectionDescriptor {

	/** Column descriptors associated with this collection */
	private Map<String, ColumnDescriptor> columnDescriptors;

	/** Name of this collection. */
	private String name;

	/**
	 * Constructor.
	 *
	 * @param name name of the Mondgo collection
	 */
	public CollectionDescriptor(String name) {
		super();
		this.name = name;
		this.columnDescriptors = new HashMap<String, ColumnDescriptor>();
	}

	/**
	 * Adds a new {@link LeafColumnDescriptor} with the provided name.
	 *
	 * @param name name of the column to add.
	 * @return the new {@link LeafColumnDescriptor}.
	 */
	public LeafColumnDescriptor addLeafColumn(String name) {
		LeafColumnDescriptor column = new LeafColumnDescriptor(name);
		this.columnDescriptors.put(name, column);

		return column;
	}

	/**
	 * Adds a new {@link NestedColumnDescriptor} with the provided name.
	 *
	 * @param name name of the column to add.
	 * @return the new {@link NestedColumnDescriptor}.
	 */
	public NestedColumnDescriptor addNestedColumn(String name) {
		NestedColumnDescriptor column = new NestedColumnDescriptor(name);
		this.columnDescriptors.put(name, column);

		return column;
	}

	/**
	 * Returns the {@link ColumnDescriptor}s associated with this
	 * collection.
	 *
	 * @return the {@link CollectionDescriptor}s associated with
	 * this collection.
	 */
	public Iterable<? extends ColumnDescriptor> columns() {
		return this.columnDescriptors.values();
	}


	/**
	 * Returns the {@link ColumnDescriptor} with the given name.
	 *
	 * @param name the name of the {@link ColumnDescriptor}.
	 * @return the {@link ColumnDescriptor}.
	 */
	public ColumnDescriptor getColumn(String name) {
		return this.columnDescriptors.get(name);
	}

	/**
	 * Returns the name of the Mongo collection represented by
	 * this descriptor.
	 *
	 * @return the name of this collection descriptor.
	 */
	public String getName() {
		return this.name;
	}

	/**
	 * Removes the provided {@link ColumnDescriptor} from this
	 * collection.
	 *
	 * @param columnDescriptor the column descriptor associated with
	 * this collection.
	 */
	public void removeColumn(ColumnDescriptor columnDescriptor) {
		this.columnDescriptors.remove(columnDescriptor.getColumnName());
	}
}