/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.utility;

/**
 * This class is used for assertion.
 *
 * @version 2.6
 */
@SuppressWarnings("nls")
public final class Assert {

	/**
	 * This flag is used to determine if the unit-tests are running.
	 */
	public static boolean UNIT_TESTS = Boolean.getBoolean("unit-tests");

	/**
	 * This class is not intended to be instantiated.
	 */
	private Assert() {
		super();
		throw new UnsupportedOperationException("Assert cannot be instantiated");
	}

	/**
	 * Shorthand for causing a check exception if the code reaches an unexpected location.
	 */
	public static void fail() {
		fail("fail");
	}

	/**
	 * Shorthand for causing a check exception if the code reaches an unexpected location.
	 *
	 * @param description The description of the failure
	 */
	public static void fail(String description) {
		throw new AssertionFailedException(description);
	}

	/**
	 * Asserts that the given boolean is <code>false</code>. If this is not the case, some kind of
	 * unchecked exception is thrown.
	 *
	 * @param expression The out code of the check
	 * @return <code>true</code> if the check passes (does not return if the check fails)
	 * @exception AssertionFailedException if the test failed
	 */
	public static boolean isFalse(boolean expression) {
		return isFalse(expression, StringTools.EMPTY_STRING);
	}

	/**
	 * Asserts that the given boolean is <code>false</code>. If this is not the case, some kind of
	 * unchecked exception is thrown. The given message is included in that exception, to aid debugging.
	 *
	 * @param expression The out code of the check
	 * @param message The message to include in the exception
	 * @return <code>true</code> if the check passes (does not return if the check fails)
	 * @exception AssertionFailedException if the test failed
	 */
	public static boolean isFalse(boolean expression, String message) {
		if (expression) {
			fail(message);
		}
		return expression;
	}

	/**
	 * Asserts that an argument is legal. If the given boolean is not <code>true</code>, an {@link
	 * AssertionFailedException} is thrown.
	 *
	 * @param expression The out code of the check
	 * @return <code>true</code> if the check passes (does not return if the check fails)
	 * @exception AssertionFailedException if the legality test failed
	 */
	public static boolean isLegal(boolean expression) {
		return isLegal(expression, StringTools.EMPTY_STRING);
	}

	/**
	 * Asserts that an argument is legal. If the given boolean is not <code>true</code>, an {@link
	 * AssertionFailedException} is thrown.The given message is included in that exception, to aid
	 * debugging.
	 *
	 * @param expression The out code of the check
	 * @param message The message to include in the exception
	 * @return <code>true</code> if the check passes (does not return if the check fails)
	 * @exception AssertionFailedException if the test failed
	 */
	public static boolean isLegal(boolean expression, String message) {
		if (!expression) {
			fail(message);
		}
		return expression;
	}

	/**
	 * Asserts that the given object is not <code>null</code>. If this is not the case, some kind of
	 * unchecked exception is thrown.
	 *
	 * @param object The value to test
	 * @exception AssertionFailedException if the test failed
	 */
	public static void isNotNull(Object object) {
		isNotNull(object, StringTools.EMPTY_STRING);
	}

	/**
	 * Asserts that the given object is not <code>null</code>. If this is not the case, some kind of
	 * unchecked exception is thrown. The given message is included in that exception, to aid debugging.
	 *
	 * @param object The value to test
	 * @param message The message to include in the exception
	 * @exception AssertionFailedException if the test failed
	 */
	public static void isNotNull(Object object, String message) {
		if (object == null) {
			fail(message);
		}
	}

	/**
	 * Asserts that the given boolean is <code>true</code>. If this is not the case, some kind of
	 * unchecked exception is thrown.
	 *
	 * @param expression The out code of the check
	 * @return <code>true</code> if the check passes (does not return if the check fails)
	 * @exception AssertionFailedException if the test failed
	 */
	public static boolean isTrue(boolean expression) {
		return isTrue(expression, StringTools.EMPTY_STRING);
	}

	/**
	 * Asserts that the given boolean is <code>true</code>. If this is not the case, some kind of
	 * unchecked exception is thrown. The given message is included in that exception, to aid debugging.
	 *
	 * @param expression The out code of the check
	 * @param message The message to include in the exception
	 * @return <code>true</code> if the check passes (does not return if the check fails)
	 * @exception AssertionFailedException if the test failed
	 */
	public static boolean isTrue(boolean expression, String message) {
		if (!expression) {
			fail(message);
		}
		return expression;
	}
}