/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.tools.mapping.orm.ExternalAssociationOverride;
import org.eclipse.persistence.tools.mapping.orm.ExternalAttributeOverride;
import org.eclipse.persistence.tools.mapping.orm.ExternalEmbeddedMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalMappingVisitor;
import org.eclipse.persistence.tools.mapping.orm.ExternalNoSqlField;

/**
 * The external form for a embedded mapping, which is a child of an entity.
 *
 * @see MappedSuperClassEntity
 *
 * @version 2.6
 */
final class EmbeddedMapping extends NonTransientMapping
                            implements ExternalEmbeddedMapping {

	/**
	 * Creates a new <code>EmbeddedMapping</code>.
	 *
	 * @param parent The parent of this external form
	 */
	EmbeddedMapping(Embeddable parent) {
		super(parent);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void accept(ExternalMappingVisitor visitor) {
		visitor.visit(this);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalAssociationOverride addAssociationOverride() {
		AssociationOverride associationOverride = buildAssociationOverride(-1);
		associationOverride.addSelf();
		return associationOverride;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalAttributeOverride addAttributeOverride() {
		AttributeOverride attributeOverride = buildAttributeOverride(-1);
		attributeOverride.addSelf();
		return attributeOverride;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ExternalAssociationOverride> associationOverrides() {

		int count = associationOverridesSize();
		List<ExternalAssociationOverride> associationOverrides = new ArrayList<ExternalAssociationOverride>(count);

		for (int index = 0; index < count; index++) {
			associationOverrides.add(buildAssociationOverride(index));
		}

		return associationOverrides;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int associationOverridesSize() {
		return getChildrenSize(AssociationOverride.ASSOCIATION_OVERRIDE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ExternalAttributeOverride> attributeOverrides() {

		int count = attributeOverridesSize();
		List<ExternalAttributeOverride> attributeOverrides = new ArrayList<ExternalAttributeOverride>(count);

		for (int index = 0; index < count; index++) {
			attributeOverrides.add(buildAttributeOverride(index));
		}

		return attributeOverrides;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int attributeOverridesSize() {
		return getChildrenSize(AttributeOverride.ATTRIBUTE_OVERRIDE);
	}

	private AssociationOverride buildAssociationOverride(int index) {
		return new AssociationOverride(this, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(NAME);
		names.add(ACCESS);
		names.add(ATTRIBUTE_TYPE);
		return names;
	}

	private AttributeOverride buildAttributeOverride(int index) {
		return new AttributeOverride(this, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(AttributeOverride.ATTRIBUTE_OVERRIDE);
		names.add(AssociationOverride.ASSOCIATION_OVERRIDE);
		names.add(Property.PROPERTY);
		names.add(AccessMethods.ACCESS_METHODS);
		return names;
	}

	private ExternalNoSqlField buildNoSqlField() {
		return new NoSqlField(this);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalAssociationOverride getAssociationOverride(int index) {

		if (hasChild(AssociationOverride.ASSOCIATION_OVERRIDE, index)) {
			return buildAssociationOverride(index);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalAttributeOverride getAttributeOverride(int index) {

		if (hasChild(AttributeOverride.ATTRIBUTE_OVERRIDE, index)) {
			return buildAttributeOverride(index);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return EMBEDDED;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getNoSqlField() {
		ExternalNoSqlField field = buildNoSqlField();
		return field.getName();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isEmbeddedMapping() {
		return true;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeAssociationOverride(int index) {
		AssociationOverride associationOverride = buildAssociationOverride(index);
		associationOverride.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeAttributeOverride(int index) {
		AttributeOverride attributeOverride = buildAttributeOverride(index);
		attributeOverride.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setNoSqlField(String field) {
		ExternalNoSqlField noSqlField = buildNoSqlField();
		noSqlField.setName(field);
	}
}