/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.tools.mapping.orm.ExternalJoinColumn;
import org.eclipse.persistence.tools.mapping.orm.ExternalManyToOneMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalMappingVisitor;
import org.eclipse.persistence.tools.mapping.orm.ExternalNoSqlJoinField;
import org.eclipse.persistence.tools.utility.TextRange;

/**
 * The external form for a M:1 mapping, which is a child of an entity.
 *
 * @see MappedSuperClassEntity
 *
 * @version 2.6
 */
final class ManyToOneMapping extends RelationshipMapping
                             implements ExternalManyToOneMapping {

	/**
	 * Creates a new <code>ManyToOneMapping</code>.
	 *
	 * @param parent The parent of this external form
	 */
	ManyToOneMapping(Embeddable parent) {
		super(parent);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void accept(ExternalMappingVisitor visitor) {
		visitor.visit(this);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalJoinColumn addJoinColumn(String name) {
		JoinColumn joinColumn = buildJoinColumn(-1);
		joinColumn.addSelf();
		joinColumn.setName(name);
		return joinColumn;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalNoSqlJoinField addJoinField(String name) {
		NoSqlJoinField joinField = buildJoinField(-1);
		joinField.addSelf();
		joinField.setName(name);
		return joinField;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(NAME);
		names.add(TARGET_ENTITY);
		names.add(FETCH);
		names.add(OPTIONAL);
		names.add(ACCESS);
		names.add(MAPPED_BY);
		names.add(MAPS_ID);
		names.add(ID);
		return names;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(JoinColumn.JOIN_COLUMN);
		names.add(JoinTable.JOIN_TABLE);
		names.add(JOIN_FETCH);
		names.add(BatchFetch.BATCH_FETCH);
		names.add(Property.PROPERTY);
		names.add(AccessMethods.ACCESS_METHODS);
		return names;
	}

	private JoinColumn buildJoinColumn(int index) {
		return new JoinColumn(this, JoinColumn.JOIN_COLUMN, index);
	}

	private NoSqlJoinField buildJoinField(int index) {
		return new NoSqlJoinField(this, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return MANY_TO_ONE;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Boolean getId() {
		return getBooleanAttribute(ID);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getIdTextRange() {
		return getAttributeTextRange(ID);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalJoinColumn getJoinColumn(int index) {

		if (hasChild(JoinColumn.JOIN_COLUMN, index)) {
			return buildJoinColumn(index);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getMappedByMappingName() {
		return getAttribute(MAPPED_BY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getMappedByMappingNameTextRange() {
		return getAttributeTextRange(MAPPED_BY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getMapsId() {
		return getAttribute(MAPS_ID);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getMapsIdTextRange() {
		return getAttributeTextRange(MAPS_ID);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ExternalJoinColumn> joinColumns() {

		int count = joinColumnsSize();
		List<ExternalJoinColumn> joinColumns = new ArrayList<ExternalJoinColumn>(count);

		for (int index = 0; index < count; index++) {
			joinColumns.add(buildJoinColumn(index));
		}

		return joinColumns;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int joinColumnsSize() {
		return getChildrenSize(JoinColumn.JOIN_COLUMN);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ExternalNoSqlJoinField> joinFields() {

		int count = joinFieldSize();
		List<ExternalNoSqlJoinField> joinFields = new ArrayList<ExternalNoSqlJoinField>(count);

		for (int index = 0; index < count; index++) {
			joinFields.add(buildJoinField(index));
		}

		return joinFields;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int joinFieldSize() {
		return getChildrenSize(NoSqlJoinField.JOIN_FIELD);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeJoinColumn(int index) {
		JoinColumn joinColumn = buildJoinColumn(index);
		joinColumn.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeJoinField(int index) {
		removeChild(NoSqlJoinField.JOIN_FIELD, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setId(Boolean id) {
		setAttribute(ID, id);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMappedByMappingName(String name) {
		setAttribute(MAPPED_BY, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMapsId(String mapsId) {
		setAttribute(MAPS_ID, mapsId);
	}
}