/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.annotations.CacheKeyType;
import org.eclipse.persistence.annotations.IdValidation;
import org.eclipse.persistence.tools.mapping.AbstractExternalForm;
import org.eclipse.persistence.tools.mapping.orm.ExternalEntityColumn;
import org.eclipse.persistence.tools.mapping.orm.ExternalPrimaryKey;
import org.eclipse.persistence.tools.utility.TextRange;

/**
 * The external form of a primary key, which is a child of an entity.
 *
 * @see MappedSuperClassEntity
 *
 * @version 2.6
 */
final class PrimaryKey extends AbstractExternalForm
                       implements ExternalPrimaryKey {

	/**
	 * Creates a new <code>PrimaryKey</code>.
	 *
	 * @param parent The parent of this external form
	 */
	PrimaryKey(MappedSuperclass parent) {
		super(parent);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntityColumn addColumn(String columnName) {
		EntityColumn column = buildColumn(-1);
		column.addSelf();
		column.setName(columnName);
		return column;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(ID_VALIDATION);
		return names;
	}

	private EntityColumn buildColumn(int index) {
		return new EntityColumn(this, EntityColumn.COLUMN, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(EntityColumn.COLUMN);
		return names;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ExternalEntityColumn> columns() {

		int count = columnsSize();
		List<ExternalEntityColumn> columns = new ArrayList<ExternalEntityColumn>(count);

		for (int index = 0; index < count; index++) {
			columns.add(buildColumn(index));
		}

		return columns;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int columnsSize() {
		return getChildrenSize(EntityColumn.COLUMN);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public CacheKeyType getCacheKey() {
		return getEnumAttribute(CACHE_KEY_TYPE, CacheKeyType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getCacheKeyTextRange() {
		return getAttributeTextRange(CACHE_KEY_TYPE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntityColumn getColumn(int index) {

		if (hasChild(EntityColumn.COLUMN, index)) {
			return buildColumn(index);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return PRIMARY_KEY;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public IdValidation getValidation() {
		return getEnumAttribute(ID_VALIDATION, IdValidation.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeAllColumns() {
		removeChildren(EntityColumn.COLUMN);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeColumn(int index) {
		EntityColumn column = buildColumn(index);
		column.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setCacheKey(CacheKeyType cacheKeyType) {
		setAttribute(CACHE_KEY_TYPE, cacheKeyType);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setValidation(IdValidation validation) {
		setAttribute(ID_VALIDATION, validation);
	}
}