/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.tools.mapping.AbstractExternalForm;
import org.eclipse.persistence.tools.mapping.orm.ExternalPrimaryKeyGenerator;
import org.eclipse.persistence.tools.utility.TextRange;
import org.w3c.dom.Element;

/**
 * The external form of a generator, which can either be a sequence generator or a table generator.
 *
 * @see SequenceGenerator
 * @see TableGenerator
 *
 * @version 2.6
 */
abstract class PrimaryKeyGenerator extends AbstractExternalForm
                                   implements ExternalPrimaryKeyGenerator {

	/**
	 * The position of the element within the list of children with the same type owned by the parent.
	 */
	private int index;

	/**
	 * Creates a new <code>PrimaryKeyGenerator</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	PrimaryKeyGenerator(AbstractExternalForm parent, int index) {
		super(parent);
		this.index = index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(DESCRIPTION);
		return names;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void calculateInsertionIndex(Element parent, Element child, String elementName) {
		if (elementName == getElementName()) {
			index = index(parent, child, elementName);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final Integer getAllocationSize() {
		return getIntegerAttribute(ALLOCATION_SIZE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getAllocationSizeTextRange() {
		return getAttributeTextRange(ALLOCATION_SIZE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getCatalogName() {
		return getAttribute(CATALOG);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getCatalogNameTextRange() {
		return getAttributeTextRange(CATALOG);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getDescription() {
		return getChildTextNode(DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final Element getElement() {

		if (index == -1) {
			return super.getElement();
		}

		return getChild(getParent(), getElementName(), index);
	}

	@Override
	@Deprecated
	public int getIndex() {
		return index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final Integer getInitialValue() {
		return getIntegerAttribute(INITIAL_VALUE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getInitialValueTextRange() {
		return getAttributeTextRange(INITIAL_VALUE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getName() {
		return getAttribute(NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getNameTextRange() {
		return getAttributeTextRange(NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getSchemaName() {
		return getAttribute(SCHEMA);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getSchemaNameTextRange() {
		return getAttributeTextRange(SCHEMA);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeSelf() {
		if (index == -1) {
			super.removeSelf();
		}
		else {
			removeChild(getParent(), getElementName(), index);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setAllocationSize(Integer size) {

		setAttribute(ALLOCATION_SIZE, size);

		if (shouldBeRemoved()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setCatalogName(String name) {

		setAttribute(CATALOG, name);

		if (shouldBeRemoved()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setDescription(String description) {
		updateChildTextNode(DESCRIPTION, description);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setInitialValue(Integer size) {

		setAttribute(INITIAL_VALUE, size);

		if (shouldBeRemoved()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setName(String name) {

		setAttribute(NAME, name);

		if (shouldBeRemoved()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setSchemaName(String name) {

		setAttribute(SCHEMA, name);

		if (shouldBeRemoved()) {
			removeSelf();
		}
	}

	final boolean shouldBeRemoved() {
		return (index == -1) && !hasAnything();
	}
}