/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.tools.mapping.orm.ExternalPrimaryKeyJoinColumn;
import org.eclipse.persistence.tools.mapping.orm.ExternalSecondaryTable;
import org.w3c.dom.Element;

/**
 * The external form of a secondary table, which is a child of an entity.
 *
 * @see Entity
 *
 * @version 2.6
 */
final class SecondaryTable extends Table
                           implements ExternalSecondaryTable {

	/**
	 * The position of the element within the list of children with the same type owned by the parent.
	 */
	private int index;

	/**
	 * Creates a new <code>EntitySecondaryTable</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned by the parent
	 */
	SecondaryTable(Entity parent, int index) {
		super(parent);
		this.index = index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalPrimaryKeyJoinColumn addPrimaryKeyJoinColumn(String name) {
		PrimaryKeyJoinColumn primaryKeyJoinColumn = buildPrimaryKeyJoinColumn(-1);
		primaryKeyJoinColumn.addSelf();
		primaryKeyJoinColumn.setName(name);
		return primaryKeyJoinColumn;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(PrimaryKeyJoinColumn.PRIMARY_KEY_JOIN_COLUMN);
		names.add(UniqueConstraint.UNIQUE_CONSTRAINT);
		return names;
	}

	private PrimaryKeyJoinColumn buildPrimaryKeyJoinColumn(int index) {
		return new PrimaryKeyJoinColumn(this, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void calculateInsertionIndex(Element parent, Element child, String elementName) {
		if (elementName == getElementName()) {
			index = index(parent, child, elementName);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Element getElement() {

		if (index == -1) {
			return super.getElement();
		}

		return getChild(getParent(), getElementName(), index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return SECONDARY_TABLE;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalPrimaryKeyJoinColumn getPrimaryKeyJoinColumn(int index) {

		if (hasChild(PrimaryKeyJoinColumn.PRIMARY_KEY_JOIN_COLUMN, index)) {
			return buildPrimaryKeyJoinColumn(index);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ExternalPrimaryKeyJoinColumn> primaryKeyJoinColumns() {

		int count = primaryKeyJoinColumnsSize();
		List<ExternalPrimaryKeyJoinColumn> primaryKeyJoinColumns = new ArrayList<ExternalPrimaryKeyJoinColumn>(count);

		for (int index = 0; index < count; index++) {
			primaryKeyJoinColumns.add(buildPrimaryKeyJoinColumn(index));
		}

		return primaryKeyJoinColumns;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int primaryKeyJoinColumnsSize() {
		return getChildrenSize(PrimaryKeyJoinColumn.PRIMARY_KEY_JOIN_COLUMN);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeAllPrimaryKeyJoinColumns() {
		removeChildren(PrimaryKeyJoinColumn.PRIMARY_KEY_JOIN_COLUMN);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removePrimaryKeyJoinColumn(int index) {
		PrimaryKeyJoinColumn pkJoinColumn = buildPrimaryKeyJoinColumn(index);
		pkJoinColumn.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeSelf() {
		if (index == -1) {
			super.removeSelf();
		}
		else {
			removeChild(getParent(), getElementName(), index);
		}
	}
}