"use strict";
/********************************************************************************
 * Copyright (C) 2017 TypeFox and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 ********************************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
exports.waitForEvent = exports.wait = exports.delay = exports.retry = exports.timeout = exports.Deferred = void 0;
const cancellation_1 = require("./cancellation");
/**
 * Simple implementation of the deferred pattern.
 * An object that exposes a promise and functions to resolve and reject it.
 */
class Deferred {
    constructor() {
        this.state = 'unresolved';
        this.promise = new Promise((resolve, reject) => {
            this.resolve = result => {
                resolve(result);
                if (this.state === 'unresolved') {
                    this.state = 'resolved';
                }
            };
            this.reject = err => {
                reject(err);
                if (this.state === 'unresolved') {
                    this.state = 'rejected';
                }
            };
        });
    }
}
exports.Deferred = Deferred;
/**
 * @returns resolves after a specified number of milliseconds
 * @throws cancelled if a given token is cancelled before a specified number of milliseconds
 */
function timeout(ms, token = cancellation_1.CancellationToken.None) {
    const deferred = new Deferred();
    const handle = setTimeout(() => deferred.resolve(), ms);
    token.onCancellationRequested(() => {
        clearTimeout(handle);
        deferred.reject(cancellation_1.cancelled());
    });
    return deferred.promise;
}
exports.timeout = timeout;
async function retry(task, retryDelay, retries) {
    let lastError;
    for (let i = 0; i < retries; i++) {
        try {
            return await task();
        }
        catch (error) {
            lastError = error;
            await timeout(retryDelay);
        }
    }
    throw lastError;
}
exports.retry = retry;
/**
 * A function to allow a promise resolution to be delayed by a number of milliseconds. Usage is as follows:
 *
 * `const stringValue = await myPromise.then(delay(600)).then(value => value.toString());`
 *
 * @param ms the number of millisecond to delay
 * @returns a function that returns a promise that returns the given value, but delayed
 */
function delay(ms) {
    return value => new Promise((resolve, reject) => { setTimeout(() => resolve(value), ms); });
}
exports.delay = delay;
/**
 * Constructs a promise that will resolve after a given delay.
 * @param ms the number of milliseconds to wait
 */
async function wait(ms) {
    await delay(ms)(undefined);
}
exports.wait = wait;
// eslint-disable-next-line @typescript-eslint/no-explicit-any
function waitForEvent(event, ms, thisArg, disposables) {
    return new Promise((resolve, reject) => {
        const registration = setTimeout(() => {
            listener.dispose();
            reject(new cancellation_1.CancellationError());
        }, ms);
        const listener = event((evt) => {
            clearTimeout(registration);
            listener.dispose();
            resolve(evt);
        }, thisArg, disposables);
    });
}
exports.waitForEvent = waitForEvent;
//# sourceMappingURL=promise-util.js.map