/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
var __values = (this && this.__values) || function(o) {
    var s = typeof Symbol === "function" && Symbol.iterator, m = s && o[s], i = 0;
    if (m) return m.call(o);
    if (o && typeof o.length === "number") return {
        next: function () {
            if (o && i >= o.length) o = void 0;
            return { value: o && o[i++], done: !o };
        }
    };
    throw new TypeError(s ? "Object is not iterable." : "Symbol.iterator is not defined.");
};
import { onUnexpectedError } from '../../../base/common/errors';
import * as strings from '../../../base/common/strings';
import { Position } from '../core/position';
import { Range } from '../core/range';
import { Selection } from '../core/selection';
import { TextModel } from '../model/textModel';
import { LanguageConfigurationRegistry } from '../modes/languageConfigurationRegistry';
var autoCloseAlways = function () { return true; };
var autoCloseNever = function () { return false; };
var autoCloseBeforeWhitespace = function (chr) { return (chr === ' ' || chr === '\t'); };
var CursorConfiguration = /** @class */ (function () {
    function CursorConfiguration(languageIdentifier, modelOptions, configuration) {
        var e_1, _a;
        this._languageIdentifier = languageIdentifier;
        var options = configuration.options;
        var layoutInfo = options.get(124 /* layoutInfo */);
        this.readOnly = options.get(75 /* readOnly */);
        this.tabSize = modelOptions.tabSize;
        this.indentSize = modelOptions.indentSize;
        this.insertSpaces = modelOptions.insertSpaces;
        this.stickyTabStops = options.get(99 /* stickyTabStops */);
        this.lineHeight = options.get(53 /* lineHeight */);
        this.pageSize = Math.max(1, Math.floor(layoutInfo.height / this.lineHeight) - 2);
        this.useTabStops = options.get(109 /* useTabStops */);
        this.wordSeparators = options.get(110 /* wordSeparators */);
        this.emptySelectionClipboard = options.get(28 /* emptySelectionClipboard */);
        this.copyWithSyntaxHighlighting = options.get(18 /* copyWithSyntaxHighlighting */);
        this.multiCursorMergeOverlapping = options.get(63 /* multiCursorMergeOverlapping */);
        this.multiCursorPaste = options.get(65 /* multiCursorPaste */);
        this.autoClosingBrackets = options.get(5 /* autoClosingBrackets */);
        this.autoClosingQuotes = options.get(7 /* autoClosingQuotes */);
        this.autoClosingOvertype = options.get(6 /* autoClosingOvertype */);
        this.autoSurround = options.get(10 /* autoSurround */);
        this.autoIndent = options.get(8 /* autoIndent */);
        this.surroundingPairs = {};
        this._electricChars = null;
        this.shouldAutoCloseBefore = {
            quote: CursorConfiguration._getShouldAutoClose(languageIdentifier, this.autoClosingQuotes),
            bracket: CursorConfiguration._getShouldAutoClose(languageIdentifier, this.autoClosingBrackets)
        };
        this.autoClosingPairs = LanguageConfigurationRegistry.getAutoClosingPairs(languageIdentifier.id);
        var surroundingPairs = CursorConfiguration._getSurroundingPairs(languageIdentifier);
        if (surroundingPairs) {
            try {
                for (var surroundingPairs_1 = __values(surroundingPairs), surroundingPairs_1_1 = surroundingPairs_1.next(); !surroundingPairs_1_1.done; surroundingPairs_1_1 = surroundingPairs_1.next()) {
                    var pair = surroundingPairs_1_1.value;
                    this.surroundingPairs[pair.open] = pair.close;
                }
            }
            catch (e_1_1) { e_1 = { error: e_1_1 }; }
            finally {
                try {
                    if (surroundingPairs_1_1 && !surroundingPairs_1_1.done && (_a = surroundingPairs_1.return)) _a.call(surroundingPairs_1);
                }
                finally { if (e_1) throw e_1.error; }
            }
        }
    }
    CursorConfiguration.shouldRecreate = function (e) {
        return (e.hasChanged(124 /* layoutInfo */)
            || e.hasChanged(110 /* wordSeparators */)
            || e.hasChanged(28 /* emptySelectionClipboard */)
            || e.hasChanged(63 /* multiCursorMergeOverlapping */)
            || e.hasChanged(65 /* multiCursorPaste */)
            || e.hasChanged(5 /* autoClosingBrackets */)
            || e.hasChanged(7 /* autoClosingQuotes */)
            || e.hasChanged(6 /* autoClosingOvertype */)
            || e.hasChanged(10 /* autoSurround */)
            || e.hasChanged(109 /* useTabStops */)
            || e.hasChanged(53 /* lineHeight */)
            || e.hasChanged(75 /* readOnly */));
    };
    Object.defineProperty(CursorConfiguration.prototype, "electricChars", {
        get: function () {
            var e_2, _a;
            if (!this._electricChars) {
                this._electricChars = {};
                var electricChars = CursorConfiguration._getElectricCharacters(this._languageIdentifier);
                if (electricChars) {
                    try {
                        for (var electricChars_1 = __values(electricChars), electricChars_1_1 = electricChars_1.next(); !electricChars_1_1.done; electricChars_1_1 = electricChars_1.next()) {
                            var char = electricChars_1_1.value;
                            this._electricChars[char] = true;
                        }
                    }
                    catch (e_2_1) { e_2 = { error: e_2_1 }; }
                    finally {
                        try {
                            if (electricChars_1_1 && !electricChars_1_1.done && (_a = electricChars_1.return)) _a.call(electricChars_1);
                        }
                        finally { if (e_2) throw e_2.error; }
                    }
                }
            }
            return this._electricChars;
        },
        enumerable: false,
        configurable: true
    });
    CursorConfiguration.prototype.normalizeIndentation = function (str) {
        return TextModel.normalizeIndentation(str, this.indentSize, this.insertSpaces);
    };
    CursorConfiguration._getElectricCharacters = function (languageIdentifier) {
        try {
            return LanguageConfigurationRegistry.getElectricCharacters(languageIdentifier.id);
        }
        catch (e) {
            onUnexpectedError(e);
            return null;
        }
    };
    CursorConfiguration._getShouldAutoClose = function (languageIdentifier, autoCloseConfig) {
        switch (autoCloseConfig) {
            case 'beforeWhitespace':
                return autoCloseBeforeWhitespace;
            case 'languageDefined':
                return CursorConfiguration._getLanguageDefinedShouldAutoClose(languageIdentifier);
            case 'always':
                return autoCloseAlways;
            case 'never':
                return autoCloseNever;
        }
    };
    CursorConfiguration._getLanguageDefinedShouldAutoClose = function (languageIdentifier) {
        try {
            var autoCloseBeforeSet_1 = LanguageConfigurationRegistry.getAutoCloseBeforeSet(languageIdentifier.id);
            return function (c) { return autoCloseBeforeSet_1.indexOf(c) !== -1; };
        }
        catch (e) {
            onUnexpectedError(e);
            return autoCloseNever;
        }
    };
    CursorConfiguration._getSurroundingPairs = function (languageIdentifier) {
        try {
            return LanguageConfigurationRegistry.getSurroundingPairs(languageIdentifier.id);
        }
        catch (e) {
            onUnexpectedError(e);
            return null;
        }
    };
    return CursorConfiguration;
}());
export { CursorConfiguration };
/**
 * Represents the cursor state on either the model or on the view model.
 */
var SingleCursorState = /** @class */ (function () {
    function SingleCursorState(selectionStart, selectionStartLeftoverVisibleColumns, position, leftoverVisibleColumns) {
        this.selectionStart = selectionStart;
        this.selectionStartLeftoverVisibleColumns = selectionStartLeftoverVisibleColumns;
        this.position = position;
        this.leftoverVisibleColumns = leftoverVisibleColumns;
        this.selection = SingleCursorState._computeSelection(this.selectionStart, this.position);
    }
    SingleCursorState.prototype.equals = function (other) {
        return (this.selectionStartLeftoverVisibleColumns === other.selectionStartLeftoverVisibleColumns
            && this.leftoverVisibleColumns === other.leftoverVisibleColumns
            && this.position.equals(other.position)
            && this.selectionStart.equalsRange(other.selectionStart));
    };
    SingleCursorState.prototype.hasSelection = function () {
        return (!this.selection.isEmpty() || !this.selectionStart.isEmpty());
    };
    SingleCursorState.prototype.move = function (inSelectionMode, lineNumber, column, leftoverVisibleColumns) {
        if (inSelectionMode) {
            // move just position
            return new SingleCursorState(this.selectionStart, this.selectionStartLeftoverVisibleColumns, new Position(lineNumber, column), leftoverVisibleColumns);
        }
        else {
            // move everything
            return new SingleCursorState(new Range(lineNumber, column, lineNumber, column), leftoverVisibleColumns, new Position(lineNumber, column), leftoverVisibleColumns);
        }
    };
    SingleCursorState._computeSelection = function (selectionStart, position) {
        var startLineNumber, startColumn, endLineNumber, endColumn;
        if (selectionStart.isEmpty()) {
            startLineNumber = selectionStart.startLineNumber;
            startColumn = selectionStart.startColumn;
            endLineNumber = position.lineNumber;
            endColumn = position.column;
        }
        else {
            if (position.isBeforeOrEqual(selectionStart.getStartPosition())) {
                startLineNumber = selectionStart.endLineNumber;
                startColumn = selectionStart.endColumn;
                endLineNumber = position.lineNumber;
                endColumn = position.column;
            }
            else {
                startLineNumber = selectionStart.startLineNumber;
                startColumn = selectionStart.startColumn;
                endLineNumber = position.lineNumber;
                endColumn = position.column;
            }
        }
        return new Selection(startLineNumber, startColumn, endLineNumber, endColumn);
    };
    return SingleCursorState;
}());
export { SingleCursorState };
var CursorContext = /** @class */ (function () {
    function CursorContext(model, coordinatesConverter, cursorConfig) {
        this.model = model;
        this.coordinatesConverter = coordinatesConverter;
        this.cursorConfig = cursorConfig;
    }
    return CursorContext;
}());
export { CursorContext };
var PartialModelCursorState = /** @class */ (function () {
    function PartialModelCursorState(modelState) {
        this.modelState = modelState;
        this.viewState = null;
    }
    return PartialModelCursorState;
}());
export { PartialModelCursorState };
var PartialViewCursorState = /** @class */ (function () {
    function PartialViewCursorState(viewState) {
        this.modelState = null;
        this.viewState = viewState;
    }
    return PartialViewCursorState;
}());
export { PartialViewCursorState };
var CursorState = /** @class */ (function () {
    function CursorState(modelState, viewState) {
        this.modelState = modelState;
        this.viewState = viewState;
    }
    CursorState.fromModelState = function (modelState) {
        return new PartialModelCursorState(modelState);
    };
    CursorState.fromViewState = function (viewState) {
        return new PartialViewCursorState(viewState);
    };
    CursorState.fromModelSelection = function (modelSelection) {
        var selectionStartLineNumber = modelSelection.selectionStartLineNumber;
        var selectionStartColumn = modelSelection.selectionStartColumn;
        var positionLineNumber = modelSelection.positionLineNumber;
        var positionColumn = modelSelection.positionColumn;
        var modelState = new SingleCursorState(new Range(selectionStartLineNumber, selectionStartColumn, selectionStartLineNumber, selectionStartColumn), 0, new Position(positionLineNumber, positionColumn), 0);
        return CursorState.fromModelState(modelState);
    };
    CursorState.fromModelSelections = function (modelSelections) {
        var states = [];
        for (var i = 0, len = modelSelections.length; i < len; i++) {
            states[i] = this.fromModelSelection(modelSelections[i]);
        }
        return states;
    };
    CursorState.prototype.equals = function (other) {
        return (this.viewState.equals(other.viewState) && this.modelState.equals(other.modelState));
    };
    return CursorState;
}());
export { CursorState };
var EditOperationResult = /** @class */ (function () {
    function EditOperationResult(type, commands, opts) {
        this.type = type;
        this.commands = commands;
        this.shouldPushStackElementBefore = opts.shouldPushStackElementBefore;
        this.shouldPushStackElementAfter = opts.shouldPushStackElementAfter;
    }
    return EditOperationResult;
}());
export { EditOperationResult };
/**
 * Common operations that work and make sense both on the model and on the view model.
 */
var CursorColumns = /** @class */ (function () {
    function CursorColumns() {
    }
    CursorColumns.visibleColumnFromColumn = function (lineContent, column, tabSize) {
        var lineContentLength = lineContent.length;
        var endOffset = column - 1 < lineContentLength ? column - 1 : lineContentLength;
        var result = 0;
        var i = 0;
        while (i < endOffset) {
            var codePoint = strings.getNextCodePoint(lineContent, endOffset, i);
            i += (codePoint >= 65536 /* UNICODE_SUPPLEMENTARY_PLANE_BEGIN */ ? 2 : 1);
            if (codePoint === 9 /* Tab */) {
                result = CursorColumns.nextRenderTabStop(result, tabSize);
            }
            else {
                var graphemeBreakType = strings.getGraphemeBreakType(codePoint);
                while (i < endOffset) {
                    var nextCodePoint = strings.getNextCodePoint(lineContent, endOffset, i);
                    var nextGraphemeBreakType = strings.getGraphemeBreakType(nextCodePoint);
                    if (strings.breakBetweenGraphemeBreakType(graphemeBreakType, nextGraphemeBreakType)) {
                        break;
                    }
                    i += (nextCodePoint >= 65536 /* UNICODE_SUPPLEMENTARY_PLANE_BEGIN */ ? 2 : 1);
                    graphemeBreakType = nextGraphemeBreakType;
                }
                if (strings.isFullWidthCharacter(codePoint) || strings.isEmojiImprecise(codePoint)) {
                    result = result + 2;
                }
                else {
                    result = result + 1;
                }
            }
        }
        return result;
    };
    CursorColumns.toStatusbarColumn = function (lineContent, column, tabSize) {
        var lineContentLength = lineContent.length;
        var endOffset = column - 1 < lineContentLength ? column - 1 : lineContentLength;
        var result = 0;
        var i = 0;
        while (i < endOffset) {
            var codePoint = strings.getNextCodePoint(lineContent, endOffset, i);
            i += (codePoint >= 65536 /* UNICODE_SUPPLEMENTARY_PLANE_BEGIN */ ? 2 : 1);
            if (codePoint === 9 /* Tab */) {
                result = CursorColumns.nextRenderTabStop(result, tabSize);
            }
            else {
                result = result + 1;
            }
        }
        return result + 1;
    };
    CursorColumns.visibleColumnFromColumn2 = function (config, model, position) {
        return this.visibleColumnFromColumn(model.getLineContent(position.lineNumber), position.column, config.tabSize);
    };
    CursorColumns.columnFromVisibleColumn = function (lineContent, visibleColumn, tabSize) {
        if (visibleColumn <= 0) {
            return 1;
        }
        var lineLength = lineContent.length;
        var beforeVisibleColumn = 0;
        var beforeColumn = 1;
        var i = 0;
        while (i < lineLength) {
            var codePoint = strings.getNextCodePoint(lineContent, lineLength, i);
            i += (codePoint >= 65536 /* UNICODE_SUPPLEMENTARY_PLANE_BEGIN */ ? 2 : 1);
            var afterVisibleColumn = void 0;
            if (codePoint === 9 /* Tab */) {
                afterVisibleColumn = CursorColumns.nextRenderTabStop(beforeVisibleColumn, tabSize);
            }
            else {
                var graphemeBreakType = strings.getGraphemeBreakType(codePoint);
                while (i < lineLength) {
                    var nextCodePoint = strings.getNextCodePoint(lineContent, lineLength, i);
                    var nextGraphemeBreakType = strings.getGraphemeBreakType(nextCodePoint);
                    if (strings.breakBetweenGraphemeBreakType(graphemeBreakType, nextGraphemeBreakType)) {
                        break;
                    }
                    i += (nextCodePoint >= 65536 /* UNICODE_SUPPLEMENTARY_PLANE_BEGIN */ ? 2 : 1);
                    graphemeBreakType = nextGraphemeBreakType;
                }
                if (strings.isFullWidthCharacter(codePoint) || strings.isEmojiImprecise(codePoint)) {
                    afterVisibleColumn = beforeVisibleColumn + 2;
                }
                else {
                    afterVisibleColumn = beforeVisibleColumn + 1;
                }
            }
            var afterColumn = i + 1;
            if (afterVisibleColumn >= visibleColumn) {
                var beforeDelta = visibleColumn - beforeVisibleColumn;
                var afterDelta = afterVisibleColumn - visibleColumn;
                if (afterDelta < beforeDelta) {
                    return afterColumn;
                }
                else {
                    return beforeColumn;
                }
            }
            beforeVisibleColumn = afterVisibleColumn;
            beforeColumn = afterColumn;
        }
        // walked the entire string
        return lineLength + 1;
    };
    CursorColumns.columnFromVisibleColumn2 = function (config, model, lineNumber, visibleColumn) {
        var result = this.columnFromVisibleColumn(model.getLineContent(lineNumber), visibleColumn, config.tabSize);
        var minColumn = model.getLineMinColumn(lineNumber);
        if (result < minColumn) {
            return minColumn;
        }
        var maxColumn = model.getLineMaxColumn(lineNumber);
        if (result > maxColumn) {
            return maxColumn;
        }
        return result;
    };
    /**
     * ATTENTION: This works with 0-based columns (as oposed to the regular 1-based columns)
     */
    CursorColumns.nextRenderTabStop = function (visibleColumn, tabSize) {
        return visibleColumn + tabSize - visibleColumn % tabSize;
    };
    /**
     * ATTENTION: This works with 0-based columns (as oposed to the regular 1-based columns)
     */
    CursorColumns.nextIndentTabStop = function (visibleColumn, indentSize) {
        return visibleColumn + indentSize - visibleColumn % indentSize;
    };
    /**
     * ATTENTION: This works with 0-based columns (as opposed to the regular 1-based columns)
     */
    CursorColumns.prevRenderTabStop = function (column, tabSize) {
        return column - 1 - (column - 1) % tabSize;
    };
    /**
     * ATTENTION: This works with 0-based columns (as opposed to the regular 1-based columns)
     */
    CursorColumns.prevIndentTabStop = function (column, indentSize) {
        return column - 1 - (column - 1) % indentSize;
    };
    return CursorColumns;
}());
export { CursorColumns };
export function isQuote(ch) {
    return (ch === '\'' || ch === '"' || ch === '`');
}
