/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __param = (this && this.__param) || function (paramIndex, decorator) {
    return function (target, key) { decorator(target, key, paramIndex); }
};
import './messageController.css';
import * as nls from '../../../nls';
import { TimeoutTimer } from '../../../base/common/async';
import { DisposableStore, MutableDisposable } from '../../../base/common/lifecycle';
import { alert } from '../../../base/browser/ui/aria/aria';
import { Range } from '../../common/core/range';
import { registerEditorContribution, EditorCommand, registerEditorCommand } from '../../browser/editorExtensions';
import { IContextKeyService, RawContextKey } from '../../../platform/contextkey/common/contextkey';
import { registerThemingParticipant } from '../../../platform/theme/common/themeService';
import { inputValidationInfoBorder, inputValidationInfoBackground, inputValidationInfoForeground } from '../../../platform/theme/common/colorRegistry';
import { ColorScheme } from '../../../platform/theme/common/theme';
var MessageController = /** @class */ (function () {
    function MessageController(editor, contextKeyService) {
        var _this = this;
        this._messageWidget = new MutableDisposable();
        this._messageListeners = new DisposableStore();
        this._editor = editor;
        this._visible = MessageController.MESSAGE_VISIBLE.bindTo(contextKeyService);
        this._editorListener = this._editor.onDidAttemptReadOnlyEdit(function () { return _this._onDidAttemptReadOnlyEdit(); });
    }
    MessageController.get = function (editor) {
        return editor.getContribution(MessageController.ID);
    };
    MessageController.prototype.dispose = function () {
        this._editorListener.dispose();
        this._messageListeners.dispose();
        this._messageWidget.dispose();
        this._visible.reset();
    };
    MessageController.prototype.isVisible = function () {
        return this._visible.get();
    };
    MessageController.prototype.showMessage = function (message, position) {
        var _this = this;
        alert(message);
        this._visible.set(true);
        this._messageWidget.clear();
        this._messageListeners.clear();
        this._messageWidget.value = new MessageWidget(this._editor, position, message);
        // close on blur, cursor, model change, dispose
        this._messageListeners.add(this._editor.onDidBlurEditorText(function () { return _this.closeMessage(); }));
        this._messageListeners.add(this._editor.onDidChangeCursorPosition(function () { return _this.closeMessage(); }));
        this._messageListeners.add(this._editor.onDidDispose(function () { return _this.closeMessage(); }));
        this._messageListeners.add(this._editor.onDidChangeModel(function () { return _this.closeMessage(); }));
        // 3sec
        this._messageListeners.add(new TimeoutTimer(function () { return _this.closeMessage(); }, 3000));
        // close on mouse move
        var bounds;
        this._messageListeners.add(this._editor.onMouseMove(function (e) {
            // outside the text area
            if (!e.target.position) {
                return;
            }
            if (!bounds) {
                // define bounding box around position and first mouse occurance
                bounds = new Range(position.lineNumber - 3, 1, e.target.position.lineNumber + 3, 1);
            }
            else if (!bounds.containsPosition(e.target.position)) {
                // check if position is still in bounds
                _this.closeMessage();
            }
        }));
    };
    MessageController.prototype.closeMessage = function () {
        this._visible.reset();
        this._messageListeners.clear();
        if (this._messageWidget.value) {
            this._messageListeners.add(MessageWidget.fadeOut(this._messageWidget.value));
        }
    };
    MessageController.prototype._onDidAttemptReadOnlyEdit = function () {
        if (this._editor.hasModel()) {
            this.showMessage(nls.localize('editor.readonly', "Cannot edit in read-only editor"), this._editor.getPosition());
        }
    };
    MessageController.ID = 'editor.contrib.messageController';
    MessageController.MESSAGE_VISIBLE = new RawContextKey('messageVisible', false, nls.localize('messageVisible', 'Whether the editor is currently showing an inline message'));
    MessageController = __decorate([
        __param(1, IContextKeyService)
    ], MessageController);
    return MessageController;
}());
export { MessageController };
var MessageCommand = EditorCommand.bindToContribution(MessageController.get);
registerEditorCommand(new MessageCommand({
    id: 'leaveEditorMessage',
    precondition: MessageController.MESSAGE_VISIBLE,
    handler: function (c) { return c.closeMessage(); },
    kbOpts: {
        weight: 100 /* EditorContrib */ + 30,
        primary: 9 /* Escape */
    }
}));
var MessageWidget = /** @class */ (function () {
    function MessageWidget(editor, _a, text) {
        var lineNumber = _a.lineNumber, column = _a.column;
        // Editor.IContentWidget.allowEditorOverflow
        this.allowEditorOverflow = true;
        this.suppressMouseDown = false;
        this._editor = editor;
        this._editor.revealLinesInCenterIfOutsideViewport(lineNumber, lineNumber, 0 /* Smooth */);
        this._position = { lineNumber: lineNumber, column: column - 1 };
        this._domNode = document.createElement('div');
        this._domNode.classList.add('monaco-editor-overlaymessage');
        var anchorTop = document.createElement('div');
        anchorTop.classList.add('anchor', 'top');
        this._domNode.appendChild(anchorTop);
        var message = document.createElement('div');
        message.classList.add('message');
        message.textContent = text;
        this._domNode.appendChild(message);
        var anchorBottom = document.createElement('div');
        anchorBottom.classList.add('anchor', 'below');
        this._domNode.appendChild(anchorBottom);
        this._editor.addContentWidget(this);
        this._domNode.classList.add('fadeIn');
    }
    MessageWidget.fadeOut = function (messageWidget) {
        var handle;
        var dispose = function () {
            messageWidget.dispose();
            clearTimeout(handle);
            messageWidget.getDomNode().removeEventListener('animationend', dispose);
        };
        handle = setTimeout(dispose, 110);
        messageWidget.getDomNode().addEventListener('animationend', dispose);
        messageWidget.getDomNode().classList.add('fadeOut');
        return { dispose: dispose };
    };
    MessageWidget.prototype.dispose = function () {
        this._editor.removeContentWidget(this);
    };
    MessageWidget.prototype.getId = function () {
        return 'messageoverlay';
    };
    MessageWidget.prototype.getDomNode = function () {
        return this._domNode;
    };
    MessageWidget.prototype.getPosition = function () {
        return { position: this._position, preference: [1 /* ABOVE */, 2 /* BELOW */] };
    };
    MessageWidget.prototype.afterRender = function (position) {
        this._domNode.classList.toggle('below', position === 2 /* BELOW */);
    };
    return MessageWidget;
}());
registerEditorContribution(MessageController.ID, MessageController);
registerThemingParticipant(function (theme, collector) {
    var border = theme.getColor(inputValidationInfoBorder);
    if (border) {
        var borderWidth = theme.type === ColorScheme.HIGH_CONTRAST ? 2 : 1;
        collector.addRule(".monaco-editor .monaco-editor-overlaymessage .anchor.below { border-top-color: " + border + "; }");
        collector.addRule(".monaco-editor .monaco-editor-overlaymessage .anchor.top { border-bottom-color: " + border + "; }");
        collector.addRule(".monaco-editor .monaco-editor-overlaymessage .message { border: " + borderWidth + "px solid " + border + "; }");
    }
    var background = theme.getColor(inputValidationInfoBackground);
    if (background) {
        collector.addRule(".monaco-editor .monaco-editor-overlaymessage .message { background-color: " + background + "; }");
    }
    var foreground = theme.getColor(inputValidationInfoForeground);
    if (foreground) {
        collector.addRule(".monaco-editor .monaco-editor-overlaymessage .message { color: " + foreground + "; }");
    }
});
