"use strict";
/********************************************************************************
 * Copyright (C) 2019 Red Hat, Inc. and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 ********************************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
/* eslint-disable @typescript-eslint/no-explicit-any */
const chai = require("chai");
const configuration_1 = require("./configuration");
const preference_scope_1 = require("@theia/core/lib/common/preferences/preference-scope");
const vscode_uri_1 = require("@theia/core/shared/vscode-uri");
const expect = chai.expect;
let inspect;
const projects = ['/projects/workspace/project1', '/projects/workspace/project2'];
const propertyName = 'tabSize';
const preferences = {
    [preference_scope_1.PreferenceScope.Default]: {
        [propertyName]: 6,
    },
    [preference_scope_1.PreferenceScope.User]: {
        [propertyName]: 5
    },
    [preference_scope_1.PreferenceScope.Workspace]: {
        [propertyName]: 4
    },
    [preference_scope_1.PreferenceScope.Folder]: {
        [projects[0]]: {
            [propertyName]: 3
        },
        [projects[1]]: {
            [propertyName]: 2
        }
    }
};
const workspace = {};
let configuration;
let defaultConfiguration;
let userConfiguration;
let workspaceConfiguration;
let folderConfigurations;
before(() => {
    workspace.getWorkspaceFolder = (uri => {
        const name = uri.toString().replace(/[^\/]+$/, '$1');
        const index = projects.indexOf(uri.toString());
        return { uri, name, index };
    });
    defaultConfiguration = new configuration_1.ConfigurationModel(preferences[preference_scope_1.PreferenceScope.Default], Object.keys(preferences[preference_scope_1.PreferenceScope.Default]));
    userConfiguration = new configuration_1.ConfigurationModel(preferences[preference_scope_1.PreferenceScope.User], Object.keys(preferences[preference_scope_1.PreferenceScope.User]));
    workspaceConfiguration = new configuration_1.ConfigurationModel(preferences[preference_scope_1.PreferenceScope.Workspace], Object.keys(preferences[preference_scope_1.PreferenceScope.Workspace]));
    folderConfigurations = projects.reduce((configurations, project) => {
        const folderPrefs = preferences[preference_scope_1.PreferenceScope.Folder][project];
        configurations[project] = new configuration_1.ConfigurationModel(folderPrefs, Object.keys(folderPrefs));
        return configurations;
    }, {});
});
describe('Configuration:', () => {
    describe('Default scope preferences:', () => {
        beforeEach(() => {
            configuration = new configuration_1.Configuration(defaultConfiguration, new configuration_1.ConfigurationModel({}, []), undefined, undefined);
            inspect = configuration.inspect(propertyName, workspace, undefined);
        });
        it('should have correct value of \'default\' property', () => {
            expect(inspect).to.have.property('default', preferences[preference_scope_1.PreferenceScope.Default][propertyName]);
            expect(inspect.default).to.equal(preferences[preference_scope_1.PreferenceScope.Default][propertyName]);
        });
        it('should have correct value of \'value\' property', () => {
            expect(inspect).to.have.property('value', preferences[preference_scope_1.PreferenceScope.Default][propertyName]);
            expect(inspect.value).to.equal(preferences[preference_scope_1.PreferenceScope.Default][propertyName]);
        });
    });
    describe('User scope preferences:', () => {
        beforeEach(() => {
            configuration = new configuration_1.Configuration(defaultConfiguration, userConfiguration, undefined, undefined);
            inspect = configuration.inspect(propertyName, workspace, undefined);
        });
        it('should have correct value of \'default\' property', () => {
            expect(inspect).to.have.property('default', preferences[preference_scope_1.PreferenceScope.Default][propertyName]);
            expect(inspect.default).to.equal(preferences[preference_scope_1.PreferenceScope.Default][propertyName]);
        });
        it('should have correct value of \'user\' property', () => {
            expect(inspect).to.have.property('user', preferences[preference_scope_1.PreferenceScope.User][propertyName]);
            expect(inspect.user).to.equal(preferences[preference_scope_1.PreferenceScope.User][propertyName]);
        });
        it('should have correct value of \'value\' property', () => {
            expect(inspect).to.have.property('value', preferences[preference_scope_1.PreferenceScope.User][propertyName]);
            expect(inspect.value).to.equal(preferences[preference_scope_1.PreferenceScope.User][propertyName]);
        });
    });
    describe('Workspace scope preferences:', () => {
        beforeEach(() => {
            configuration = new configuration_1.Configuration(defaultConfiguration, userConfiguration, workspaceConfiguration, undefined);
            inspect = configuration.inspect(propertyName, workspace, undefined);
        });
        it('should have correct value of \'default\' property', () => {
            expect(inspect).to.have.property('default', preferences[preference_scope_1.PreferenceScope.Default][propertyName]);
            expect(inspect.default).to.equal(preferences[preference_scope_1.PreferenceScope.Default][propertyName]);
        });
        it('should have correct value of \'user\' property', () => {
            expect(inspect).to.have.property('user', preferences[preference_scope_1.PreferenceScope.User][propertyName]);
            expect(inspect.user).to.equal(preferences[preference_scope_1.PreferenceScope.User][propertyName]);
        });
        it('should have correct value of \'workspace\' property', () => {
            expect(inspect).to.have.property('workspace', preferences[preference_scope_1.PreferenceScope.Workspace][propertyName]);
            expect(inspect.workspace).to.equal(preferences[preference_scope_1.PreferenceScope.Workspace][propertyName]);
        });
        it('should have correct value of \'value\' property', () => {
            expect(inspect).to.have.property('value', preferences[preference_scope_1.PreferenceScope.Workspace][propertyName]);
            expect(inspect.value).to.equal(preferences[preference_scope_1.PreferenceScope.Workspace][propertyName]);
        });
    });
    describe('Folder scope preferences:', () => {
        const project = projects[0];
        beforeEach(() => {
            configuration = new configuration_1.Configuration(defaultConfiguration, userConfiguration, workspaceConfiguration, folderConfigurations);
            const resource = vscode_uri_1.URI.revive({ path: project, scheme: '', authority: '', query: '', fragment: '' });
            inspect = configuration.inspect(propertyName, workspace, resource);
        });
        it('should have correct value of \'default\' property', () => {
            expect(inspect).to.have.property('default', preferences[preference_scope_1.PreferenceScope.Default][propertyName]);
            expect(inspect.default).to.equal(preferences[preference_scope_1.PreferenceScope.Default][propertyName]);
        });
        it('should have correct value of \'user\' property', () => {
            expect(inspect).to.have.property('user', preferences[preference_scope_1.PreferenceScope.User][propertyName]);
            expect(inspect.user).to.equal(preferences[preference_scope_1.PreferenceScope.User][propertyName]);
        });
        it('should have correct value of \'workspace\' property', () => {
            expect(inspect).to.have.property('workspace', preferences[preference_scope_1.PreferenceScope.Workspace][propertyName]);
            expect(inspect.workspace).to.equal(preferences[preference_scope_1.PreferenceScope.Workspace][propertyName]);
        });
        it('should have correct value of \'workspaceFolder\' property', () => {
            expect(inspect).to.have.property('workspaceFolder', preferences[preference_scope_1.PreferenceScope.Folder][project][propertyName]);
            expect(inspect.workspaceFolder).to.equal(preferences[preference_scope_1.PreferenceScope.Folder][project][propertyName]);
        });
        it('should have correct value of \'value\' property', () => {
            expect(inspect).to.have.property('value', preferences[preference_scope_1.PreferenceScope.Folder][project][propertyName]);
            expect(inspect.value).to.equal(preferences[preference_scope_1.PreferenceScope.Folder][project][propertyName]);
        });
    });
    describe('ConfigurationModel', () => {
        it('check merge', () => {
            defaultConfiguration = new configuration_1.ConfigurationModel(preferences[preference_scope_1.PreferenceScope.Default], Object.keys(preferences[preference_scope_1.PreferenceScope.Default]));
            userConfiguration = new configuration_1.ConfigurationModel(preferences[preference_scope_1.PreferenceScope.User], Object.keys(preferences[preference_scope_1.PreferenceScope.User]));
            workspaceConfiguration = new configuration_1.ConfigurationModel(preferences[preference_scope_1.PreferenceScope.Workspace], Object.keys(preferences[preference_scope_1.PreferenceScope.Workspace]));
            const mergedConfiguration = new configuration_1.ConfigurationModel().merge(defaultConfiguration, userConfiguration, workspaceConfiguration);
            expect(mergedConfiguration.getValue('tabSize')).to.equal(4);
        });
        it('Prototype pollution check', () => {
            const payload = JSON.parse('{"__proto__":{"injectedConfigurationPrototype": true}}');
            const configurationModel = new configuration_1.ConfigurationModel();
            configurationModel.merge(new configuration_1.ConfigurationModel(payload));
            const prototypeObject = Object.prototype;
            expect(prototypeObject.injectedConfigurationPrototype).to.be.an('undefined');
            const rawObject = {};
            expect(rawObject.injectedConfigurationPrototype).to.be.an('undefined');
        });
        it('Prototype constructor pollution check', () => {
            const payload = JSON.parse('{"constructor": {"prototype": {"injectedConfigurationConstructorPrototype": true}}}');
            const configurationModel = new configuration_1.ConfigurationModel();
            configurationModel.merge(new configuration_1.ConfigurationModel(payload));
            const prototypeObject = Object.prototype;
            expect(prototypeObject.injectedConfigurationConstructorPrototype).to.be.an('undefined');
            const rawObject = {};
            expect(rawObject.injectedConfigurationConstructorPrototype).to.be.an('undefined');
        });
    });
});
//# sourceMappingURL=configuration.spec.js.map