"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DebugExtImpl = void 0;
// *****************************************************************************
// Copyright (C) 2018 Red Hat, Inc. and others.
//
// This program and the accompanying materials are made available under the
// terms of the Eclipse Public License v. 2.0 which is available at
// http://www.eclipse.org/legal/epl-2.0.
//
// This Source Code may also be made available under the following Secondary
// Licenses when the conditions for such availability set forth in the Eclipse
// Public License v. 2.0 are satisfied: GNU General Public License, version 2
// with the GNU Classpath Exception which is available at
// https://www.gnu.org/software/classpath/license.html.
//
// SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
// *****************************************************************************
const event_1 = require("@theia/core/lib/common/event");
const path_1 = require("@theia/core/lib/common/path");
const vscode_uri_1 = require("@theia/core/shared/vscode-uri");
const plugin_api_rpc_1 = require("../../../common/plugin-api-rpc");
const types_impl_1 = require("../../types-impl");
const plugin_debug_adapter_executable_resolver_1 = require("./plugin-debug-adapter-executable-resolver");
const plugin_debug_adapter_session_1 = require("./plugin-debug-adapter-session");
const plugin_debug_adapter_starter_1 = require("./plugin-debug-adapter-starter");
const plugin_debug_adapter_tracker_1 = require("./plugin-debug-adapter-tracker");
const uuid = require("uuid");
/* eslint-disable @typescript-eslint/no-explicit-any */
// TODO: rename file to `debug-ext.ts`
/**
 * It is supposed to work at node only.
 */
class DebugExtImpl {
    constructor(rpc) {
        // debug sessions by sessionId
        this.sessions = new Map();
        // providers by type (initial)
        this.configurationProviders = new Map();
        // providers by type (dynamic)
        this.dynamicConfigurationProviders = new Map();
        /**
         * Only use internally, don't send it to the frontend. It's expensive!
         * It's already there as a part of the plugin metadata.
         */
        this.debuggersContributions = new Map();
        this.descriptorFactories = new Map();
        this.trackerFactories = [];
        this.contributionPaths = new Map();
        this.onDidChangeBreakpointsEmitter = new event_1.Emitter();
        this.onDidChangeActiveDebugSessionEmitter = new event_1.Emitter();
        this.onDidTerminateDebugSessionEmitter = new event_1.Emitter();
        this.onDidStartDebugSessionEmitter = new event_1.Emitter();
        this.onDidReceiveDebugSessionCustomEmitter = new event_1.Emitter();
        this._breakpoints = new Map();
        this.proxy = rpc.getProxy(plugin_api_rpc_1.PLUGIN_RPC_CONTEXT.DEBUG_MAIN);
        this.activeDebugConsole = {
            append: (value) => this.proxy.$appendToDebugConsole(value),
            appendLine: (value) => this.proxy.$appendLineToDebugConsole(value)
        };
    }
    get breakpoints() {
        return [...this._breakpoints.values()];
    }
    /**
     * Sets dependencies.
     */
    assistedInject(connectionExt, commandRegistryExt) {
        this.connectionExt = connectionExt;
        this.commandRegistryExt = commandRegistryExt;
    }
    /**
     * Registers contributions.
     * @param pluginFolder plugin folder path
     * @param contributions available debuggers contributions
     */
    registerDebuggersContributions(pluginFolder, contributions) {
        contributions.forEach(contribution => {
            this.contributionPaths.set(contribution.type, pluginFolder);
            this.debuggersContributions.set(contribution.type, contribution);
            this.proxy.$registerDebuggerContribution({
                type: contribution.type,
                label: contribution.label || contribution.type
            });
            console.log(`Debugger contribution has been registered: ${contribution.type}`);
        });
    }
    get onDidReceiveDebugSessionCustomEvent() {
        return this.onDidReceiveDebugSessionCustomEmitter.event;
    }
    get onDidChangeActiveDebugSession() {
        return this.onDidChangeActiveDebugSessionEmitter.event;
    }
    get onDidTerminateDebugSession() {
        return this.onDidTerminateDebugSessionEmitter.event;
    }
    get onDidStartDebugSession() {
        return this.onDidStartDebugSessionEmitter.event;
    }
    get onDidChangeBreakpoints() {
        return this.onDidChangeBreakpointsEmitter.event;
    }
    addBreakpoints(breakpoints) {
        const added = [];
        for (const b of breakpoints) {
            if (this._breakpoints.has(b.id)) {
                continue;
            }
            this._breakpoints.set(b.id, b);
            added.push(b);
        }
        if (added.length) {
            this.onDidChangeBreakpointsEmitter.fire({ added, removed: [], changed: [] });
            this.proxy.$addBreakpoints(added);
        }
    }
    removeBreakpoints(breakpoints) {
        const removed = [];
        const removedIds = [];
        for (const b of breakpoints) {
            if (!this._breakpoints.has(b.id)) {
                continue;
            }
            this._breakpoints.delete(b.id);
            removed.push(b);
            removedIds.push(b.id);
        }
        if (removed.length) {
            this.onDidChangeBreakpointsEmitter.fire({ added: [], removed, changed: [] });
            this.proxy.$removeBreakpoints(removedIds);
        }
    }
    startDebugging(folder, nameOrConfiguration, options) {
        return this.proxy.$startDebugging(folder, nameOrConfiguration, options);
    }
    stopDebugging(session) {
        return this.proxy.$stopDebugging(session === null || session === void 0 ? void 0 : session.id);
    }
    registerDebugAdapterDescriptorFactory(debugType, factory) {
        if (this.descriptorFactories.has(debugType)) {
            throw new Error(`Descriptor factory for ${debugType} has been already registered`);
        }
        this.descriptorFactories.set(debugType, factory);
        return types_impl_1.Disposable.create(() => this.descriptorFactories.delete(debugType));
    }
    registerDebugAdapterTrackerFactory(debugType, factory) {
        if (!factory) {
            return types_impl_1.Disposable.create(() => { });
        }
        this.trackerFactories.push([debugType, factory]);
        return types_impl_1.Disposable.create(() => {
            this.trackerFactories = this.trackerFactories.filter(tuple => tuple[1] !== factory);
        });
    }
    registerDebugConfigurationProvider(debugType, provider, trigger) {
        console.log(`Debug configuration provider has been registered: ${debugType}, trigger: ${trigger}`);
        const providersByTriggerKind = trigger === plugin_api_rpc_1.DebugConfigurationProviderTriggerKind.Initial ? this.configurationProviders : this.dynamicConfigurationProviders;
        let providers = providersByTriggerKind.get(debugType);
        if (!providers) {
            providersByTriggerKind.set(debugType, providers = new Set());
        }
        providers.add(provider);
        return types_impl_1.Disposable.create(() => {
            // eslint-disable-next-line @typescript-eslint/no-shadow
            const providers = providersByTriggerKind.get(debugType);
            if (providers) {
                providers.delete(provider);
                if (providers.size === 0) {
                    providersByTriggerKind.delete(debugType);
                }
            }
        });
    }
    async $onSessionCustomEvent(sessionId, event, body) {
        const session = this.sessions.get(sessionId);
        if (session) {
            this.onDidReceiveDebugSessionCustomEmitter.fire({ event, body, session });
        }
    }
    async $sessionDidCreate(sessionId) {
        const session = this.sessions.get(sessionId);
        if (session) {
            this.onDidStartDebugSessionEmitter.fire(session);
        }
    }
    async $sessionDidDestroy(sessionId) {
        const session = this.sessions.get(sessionId);
        if (session) {
            this.onDidTerminateDebugSessionEmitter.fire(session);
        }
    }
    async $sessionDidChange(sessionId) {
        this.activeDebugSession = sessionId ? this.sessions.get(sessionId) : undefined;
        this.onDidChangeActiveDebugSessionEmitter.fire(this.activeDebugSession);
    }
    async $breakpointsDidChange(added, removed, changed) {
        const a = [];
        const r = [];
        const c = [];
        for (const b of added) {
            if (this._breakpoints.has(b.id)) {
                continue;
            }
            const bExt = this.toBreakpointExt(b);
            if (bExt) {
                this._breakpoints.set(bExt.id, bExt);
                a.push(bExt);
            }
        }
        for (const id of removed) {
            const bExt = this._breakpoints.get(id);
            if (bExt) {
                this._breakpoints.delete(id);
                r.push(bExt);
            }
        }
        for (const b of changed) {
            const bExt = this._breakpoints.get(b.id);
            if (bExt) {
                const { functionName, location, enabled, condition, hitCondition, logMessage } = b;
                if (bExt instanceof types_impl_1.FunctionBreakpoint && functionName) {
                    Object.assign(bExt, { enabled, condition, hitCondition, logMessage, functionName });
                }
                else if (bExt instanceof types_impl_1.SourceBreakpoint && location) {
                    const range = new types_impl_1.Range(location.range.startLineNumber, location.range.startColumn, location.range.endLineNumber, location.range.endColumn);
                    Object.assign(bExt, { enabled, condition, hitCondition, logMessage, location: new types_impl_1.Location(vscode_uri_1.URI.revive(location.uri), range) });
                }
                c.push(bExt);
            }
        }
        this.onDidChangeBreakpointsEmitter.fire({ added: a, removed: r, changed: c });
    }
    toBreakpointExt({ functionName, location, enabled, condition, hitCondition, logMessage }) {
        if (location) {
            const range = new types_impl_1.Range(location.range.startLineNumber, location.range.startColumn, location.range.endLineNumber, location.range.endColumn);
            return new types_impl_1.SourceBreakpoint(new types_impl_1.Location(vscode_uri_1.URI.revive(location.uri), range), enabled, condition, hitCondition, logMessage);
        }
        if (functionName) {
            return new types_impl_1.FunctionBreakpoint(functionName, enabled, condition, hitCondition, logMessage);
        }
        return undefined;
    }
    async $createDebugSession(debugConfiguration) {
        const sessionId = uuid.v4();
        const theiaSession = {
            id: sessionId,
            type: debugConfiguration.type,
            name: debugConfiguration.name,
            configuration: debugConfiguration,
            customRequest: async (command, args) => {
                var _a;
                const response = await this.proxy.$customRequest(sessionId, command, args);
                if (response && response.success) {
                    return response.body;
                }
                return Promise.reject(new Error((_a = response.message) !== null && _a !== void 0 ? _a : 'custom request failed'));
            }
        };
        const tracker = await this.createDebugAdapterTracker(theiaSession);
        const communicationProvider = await this.createDebugAdapter(theiaSession, debugConfiguration);
        const debugAdapterSession = new plugin_debug_adapter_session_1.PluginDebugAdapterSession(communicationProvider, tracker, theiaSession);
        this.sessions.set(sessionId, debugAdapterSession);
        const connection = await this.connectionExt.ensureConnection(sessionId);
        debugAdapterSession.start(connection);
        return sessionId;
    }
    async $terminateDebugSession(sessionId) {
        const debugAdapterSession = this.sessions.get(sessionId);
        if (debugAdapterSession) {
            await debugAdapterSession.stop();
            this.sessions.delete(sessionId);
        }
    }
    async $getTerminalCreationOptions(debugType) {
        return this.doGetTerminalCreationOptions(debugType);
    }
    async doGetTerminalCreationOptions(debugType) {
        return undefined;
    }
    async $provideDebugConfigurations(debugType, workspaceFolderUri, dynamic = false) {
        let result = [];
        const providers = dynamic ? this.dynamicConfigurationProviders.get(debugType) : this.configurationProviders.get(debugType);
        if (providers) {
            for (const provider of providers) {
                if (provider.provideDebugConfigurations) {
                    result = result.concat(await provider.provideDebugConfigurations(this.toWorkspaceFolder(workspaceFolderUri)) || []);
                }
            }
        }
        return result;
    }
    async $resolveDebugConfigurations(debugConfiguration, workspaceFolderUri) {
        let current = debugConfiguration;
        for (const providers of [
            this.configurationProviders.get(debugConfiguration.type),
            this.dynamicConfigurationProviders.get(debugConfiguration.type),
            this.configurationProviders.get('*')
        ]) {
            if (providers) {
                for (const provider of providers) {
                    if (provider.resolveDebugConfiguration) {
                        try {
                            const next = await provider.resolveDebugConfiguration(this.toWorkspaceFolder(workspaceFolderUri), current);
                            if (next) {
                                current = next;
                            }
                            else {
                                return current;
                            }
                        }
                        catch (e) {
                            console.error(e);
                        }
                    }
                }
            }
        }
        return current;
    }
    async $resolveDebugConfigurationWithSubstitutedVariables(debugConfiguration, workspaceFolderUri) {
        let current = debugConfiguration;
        for (const providers of [
            this.configurationProviders.get(debugConfiguration.type),
            this.dynamicConfigurationProviders.get(debugConfiguration.type),
            this.configurationProviders.get('*')
        ]) {
            if (providers) {
                for (const provider of providers) {
                    if (provider.resolveDebugConfigurationWithSubstitutedVariables) {
                        try {
                            const next = await provider.resolveDebugConfigurationWithSubstitutedVariables(this.toWorkspaceFolder(workspaceFolderUri), current);
                            if (next) {
                                current = next;
                            }
                            else {
                                return current;
                            }
                        }
                        catch (e) {
                            console.error(e);
                        }
                    }
                }
            }
        }
        return current;
    }
    async createDebugAdapterTracker(session) {
        return plugin_debug_adapter_tracker_1.PluginDebugAdapterTracker.create(session, this.trackerFactories);
    }
    async createDebugAdapter(session, debugConfiguration) {
        const executable = await this.resolveDebugAdapterExecutable(debugConfiguration);
        const descriptorFactory = this.descriptorFactories.get(session.type);
        if (descriptorFactory) {
            // 'createDebugAdapterDescriptor' is called at the start of a debug session to provide details about the debug adapter to use.
            // These details must be returned as objects of type [DebugAdapterDescriptor](#DebugAdapterDescriptor).
            // Currently two types of debug adapters are supported:
            // - a debug adapter executable is specified as a command path and arguments (see [DebugAdapterExecutable](#DebugAdapterExecutable)),
            // - a debug adapter server reachable via a communication port (see [DebugAdapterServer](#DebugAdapterServer)).
            // If the method is not implemented the default behavior is this:
            //   createDebugAdapter(session: DebugSession, executable: DebugAdapterExecutable) {
            //      if (typeof session.configuration.debugServer === 'number') {
            //         return new DebugAdapterServer(session.configuration.debugServer);
            //      }
            //      return executable;
            //   }
            //  @param session The [debug session](#DebugSession) for which the debug adapter will be used.
            //  @param executable The debug adapter's executable information as specified in the package.json (or undefined if no such information exists).
            const descriptor = await descriptorFactory.createDebugAdapterDescriptor(session, executable);
            if (descriptor) {
                if (types_impl_1.DebugAdapterServer.is(descriptor)) {
                    return (0, plugin_debug_adapter_starter_1.connectSocketDebugAdapter)(descriptor);
                }
                else if (types_impl_1.DebugAdapterExecutable.is(descriptor)) {
                    return (0, plugin_debug_adapter_starter_1.startDebugAdapter)(descriptor);
                }
                else if (types_impl_1.DebugAdapterNamedPipeServer.is(descriptor)) {
                    return (0, plugin_debug_adapter_starter_1.connectPipeDebugAdapter)(descriptor);
                }
                else if (types_impl_1.DebugAdapterInlineImplementation.is(descriptor)) {
                    return (0, plugin_debug_adapter_starter_1.connectInlineDebugAdapter)(descriptor);
                }
            }
        }
        if ('debugServer' in debugConfiguration) {
            return (0, plugin_debug_adapter_starter_1.connectSocketDebugAdapter)({ port: debugConfiguration.debugServer });
        }
        else {
            if (!executable) {
                throw new Error('It is not possible to provide debug adapter executable.');
            }
            return (0, plugin_debug_adapter_starter_1.startDebugAdapter)(executable);
        }
    }
    async resolveDebugAdapterExecutable(debugConfiguration) {
        const { type } = debugConfiguration;
        const contribution = this.debuggersContributions.get(type);
        if (contribution) {
            if (contribution.adapterExecutableCommand) {
                const executable = await this.commandRegistryExt.executeCommand(contribution.adapterExecutableCommand);
                if (executable) {
                    return executable;
                }
            }
            else {
                const contributionPath = this.contributionPaths.get(type);
                if (contributionPath) {
                    return (0, plugin_debug_adapter_executable_resolver_1.resolveDebugAdapterExecutable)(contributionPath, contribution);
                }
            }
        }
        throw new Error(`It is not possible to provide debug adapter executable for '${debugConfiguration.type}'.`);
    }
    toWorkspaceFolder(folder) {
        if (!folder) {
            return undefined;
        }
        const uri = vscode_uri_1.URI.parse(folder);
        const path = new path_1.Path(uri.path);
        return {
            uri: uri,
            name: path.base,
            index: 0
        };
    }
}
exports.DebugExtImpl = DebugExtImpl;
//# sourceMappingURL=debug.js.map