"use strict";
// *****************************************************************************
// Copyright (C) 2018 Red Hat, Inc. and others.
//
// This program and the accompanying materials are made available under the
// terms of the Eclipse Public License v. 2.0 which is available at
// http://www.eclipse.org/legal/epl-2.0.
//
// This Source Code may also be made available under the following Secondary
// Licenses when the conditions for such availability set forth in the Eclipse
// Public License v. 2.0 are satisfied: GNU General Public License, version 2
// with the GNU Classpath Exception which is available at
// https://www.gnu.org/software/classpath/license.html.
//
// SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
// *****************************************************************************
Object.defineProperty(exports, "__esModule", { value: true });
exports.PreferenceRegistryExtImpl = exports.TheiaWorkspace = exports.PreferenceScope = void 0;
/* eslint-disable @typescript-eslint/no-explicit-any */
const event_1 = require("@theia/core/lib/common/event");
const os_1 = require("@theia/core/lib/common/os");
const vscode_uri_1 = require("@theia/core/shared/vscode-uri");
const map_1 = require("@theia/monaco-editor-core/esm/vs/base/common/map");
const configurationModels_1 = require("@theia/monaco-editor-core/esm/vs/platform/configuration/common/configurationModels");
const workspace_1 = require("@theia/monaco-editor-core/esm/vs/platform/workspace/common/workspace");
const uuid_1 = require("uuid");
const plugin_api_rpc_1 = require("../common/plugin-api-rpc");
const types_1 = require("../common/types");
const cloneDeep = require("lodash.clonedeep");
const injectionRe = /\b__proto__\b|\bconstructor\.prototype\b/;
var ConfigurationTarget;
(function (ConfigurationTarget) {
    ConfigurationTarget[ConfigurationTarget["Global"] = 1] = "Global";
    ConfigurationTarget[ConfigurationTarget["Workspace"] = 2] = "Workspace";
    ConfigurationTarget[ConfigurationTarget["WorkspaceFolder"] = 3] = "WorkspaceFolder"; // eslint-disable-line @typescript-eslint/no-shadow
})(ConfigurationTarget || (ConfigurationTarget = {}));
var PreferenceScope;
(function (PreferenceScope) {
    PreferenceScope[PreferenceScope["Default"] = 0] = "Default";
    PreferenceScope[PreferenceScope["User"] = 1] = "User";
    PreferenceScope[PreferenceScope["Workspace"] = 2] = "Workspace";
    PreferenceScope[PreferenceScope["Folder"] = 3] = "Folder";
})(PreferenceScope = exports.PreferenceScope || (exports.PreferenceScope = {}));
// eslint-disable-next-line @typescript-eslint/no-explicit-any
function lookUp(tree, key) {
    if (!key) {
        return;
    }
    const parts = key.split('.');
    let node = tree;
    for (let i = 0; node && i < parts.length; i++) {
        node = node[parts[i]];
    }
    return node;
}
class TheiaWorkspace extends workspace_1.Workspace {
    constructor(ext) {
        var _a, _b;
        const folders = ((_a = ext.workspaceFolders) !== null && _a !== void 0 ? _a : []).map(folder => new workspace_1.WorkspaceFolder(folder));
        super((0, uuid_1.v4)(), folders, false, (_b = ext.workspaceFile) !== null && _b !== void 0 ? _b : null, () => os_1.isOSX || os_1.isWindows);
    }
}
exports.TheiaWorkspace = TheiaWorkspace;
class PreferenceRegistryExtImpl {
    constructor(rpc, workspace) {
        this.workspace = workspace;
        this._onDidChangeConfiguration = new event_1.Emitter();
        this.onDidChangeConfiguration = this._onDidChangeConfiguration.event;
        this.OVERRIDE_PROPERTY = '^\\[(.*)\\]$';
        this.OVERRIDE_PROPERTY_PATTERN = new RegExp(this.OVERRIDE_PROPERTY);
        this.OVERRIDE_KEY_TEST = /^\[([^\]]+)\]\./;
        this.proxy = rpc.getProxy(plugin_api_rpc_1.PLUGIN_RPC_CONTEXT.PREFERENCE_REGISTRY_MAIN);
    }
    init(data) {
        this._preferences = this.parse(data);
    }
    $acceptConfigurationChanged(data, eventData) {
        this.init(data);
        this._onDidChangeConfiguration.fire(this.toConfigurationChangeEvent(eventData));
    }
    getConfiguration(rawSection, rawScope, extensionId) {
        const overrides = this.parseConfigurationAccessOptions(rawScope);
        const preferences = this.toReadonlyValue(this._preferences.getValue(rawSection, overrides, new TheiaWorkspace(this.workspace)));
        const configuration = {
            has(key) {
                return typeof lookUp(preferences, key) !== 'undefined';
            },
            get: (key, defaultValue) => {
                const result = lookUp(preferences, key);
                if (typeof result === 'undefined') {
                    return defaultValue;
                }
                else {
                    let clonedConfig = undefined;
                    const cloneOnWriteProxy = (target, accessor) => {
                        let clonedTarget = undefined;
                        const cloneTarget = () => {
                            clonedConfig = clonedConfig ? clonedConfig : cloneDeep(preferences);
                            clonedTarget = clonedTarget ? clonedTarget : lookUp(clonedConfig, accessor);
                        };
                        if (!(0, types_1.isObject)(target)) {
                            return target;
                        }
                        return new Proxy(target, {
                            get: (targ, prop) => {
                                const config = Object.getOwnPropertyDescriptor(targ, prop);
                                // This check ensures that https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Proxy/Proxy/get#invariants are satisfied
                                if ((config === null || config === void 0 ? void 0 : config.configurable) === false && (config === null || config === void 0 ? void 0 : config.writable) === false) {
                                    return targ[prop];
                                }
                                if (typeof prop === 'string' && prop.toLowerCase() === 'tojson') {
                                    cloneTarget();
                                    return () => clonedTarget;
                                }
                                if (clonedConfig) {
                                    clonedTarget = clonedTarget ? clonedTarget : lookUp(clonedConfig, accessor);
                                    return clonedTarget[prop];
                                }
                                const res = targ[prop];
                                if (typeof prop === 'string') {
                                    return cloneOnWriteProxy(res, `${accessor}.${prop}`);
                                }
                                return res;
                            },
                            set: (targ, prop, val) => {
                                cloneTarget();
                                clonedTarget[prop] = val;
                                return true;
                            },
                            deleteProperty: (targ, prop) => {
                                cloneTarget();
                                delete clonedTarget[prop];
                                return true;
                            },
                            defineProperty: (targ, prop, descr) => {
                                cloneTarget();
                                Object.defineProperty(clonedTarget, prop, descr);
                                return true;
                            }
                        });
                    };
                    return cloneOnWriteProxy(result, key);
                }
            },
            update: (key, value, targetScope, withLanguageOverride) => {
                var _a;
                const resourceStr = (_a = overrides.resource) === null || _a === void 0 ? void 0 : _a.toString();
                const fullPath = `${overrides.overrideIdentifier ? `[${overrides.overrideIdentifier}].` : ''}${rawSection}.${key}`;
                if (typeof value !== 'undefined') {
                    return this.proxy.$updateConfigurationOption(targetScope, fullPath, value, resourceStr, withLanguageOverride);
                }
                else {
                    return this.proxy.$removeConfigurationOption(targetScope, fullPath, resourceStr, withLanguageOverride);
                }
            },
            inspect: (key) => {
                var _a, _b, _c, _d;
                const path = `${rawSection}.${key}`;
                const result = this._preferences.inspect(path, overrides, new TheiaWorkspace(this.workspace));
                if (!result) {
                    return undefined;
                }
                const configInspect = { key };
                configInspect.defaultValue = (_a = result.default) === null || _a === void 0 ? void 0 : _a.value;
                configInspect.globalValue = (_b = result.user) === null || _b === void 0 ? void 0 : _b.value;
                configInspect.workspaceValue = (_c = result.workspace) === null || _c === void 0 ? void 0 : _c.value;
                configInspect.workspaceFolderValue = (_d = result.workspaceFolder) === null || _d === void 0 ? void 0 : _d.value;
                return configInspect;
            }
        };
        if (typeof preferences === 'object') {
            (0, types_1.mixin)(configuration, preferences, false);
        }
        return Object.freeze(configuration);
    }
    toReadonlyValue(data) {
        const readonlyProxy = (target) => (0, types_1.isObject)(target)
            ? new Proxy(target, {
                get: (targ, prop) => {
                    const config = Object.getOwnPropertyDescriptor(targ, prop);
                    // This check ensures that https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Proxy/Proxy/get#invariants are satisfied
                    if ((config === null || config === void 0 ? void 0 : config.configurable) === false && (config === null || config === void 0 ? void 0 : config.writable) === false) {
                        return targ[prop];
                    }
                    return readonlyProxy(targ[prop]);
                },
                set: (targ, prop, val) => {
                    throw new Error(`TypeError: Cannot assign to read only property '${prop}' of object`);
                },
                deleteProperty: (targ, prop) => {
                    throw new Error(`TypeError: Cannot delete read only property '${prop}' of object`);
                },
                defineProperty: (targ, prop) => {
                    throw new Error(`TypeError: Cannot define property '${prop}' of a readonly object`);
                },
                setPrototypeOf: (targ) => {
                    throw new Error('TypeError: Cannot set prototype for a readonly object');
                },
                isExtensible: () => false,
                preventExtensions: () => true
            })
            : target;
        return readonlyProxy(data);
    }
    parse(data) {
        const defaultConfiguration = this.getConfigurationModel(data[PreferenceScope.Default]);
        const userConfiguration = this.getConfigurationModel(data[PreferenceScope.User]);
        const workspaceConfiguration = this.getConfigurationModel(data[PreferenceScope.Workspace]);
        const folderConfigurations = new map_1.ResourceMap();
        Object.keys(data[PreferenceScope.Folder]).forEach(resource => {
            folderConfigurations.set(vscode_uri_1.URI.parse(resource), this.getConfigurationModel(data[PreferenceScope.Folder][resource]));
        });
        return new configurationModels_1.Configuration(defaultConfiguration, new configurationModels_1.ConfigurationModel(), /** policy configuration. */ new configurationModels_1.ConfigurationModel(), /** application configuration. */ userConfiguration, new configurationModels_1.ConfigurationModel(), /** remote configuration. */ workspaceConfiguration, folderConfigurations);
    }
    getConfigurationModel(data) {
        if (!data) {
            return new configurationModels_1.ConfigurationModel();
        }
        const configData = this.parseConfigurationData(data);
        return new configurationModels_1.ConfigurationModel(configData.contents, configData.keys, configData.overrides);
    }
    parseConfigurationData(data) {
        const keys = new Array();
        const overrides = Object.create(null);
        const contents = Object.keys(data).reduce((result, key) => {
            if (injectionRe.test(key)) {
                return result;
            }
            const parts = key.split('.');
            let branch = result;
            const isOverride = this.OVERRIDE_KEY_TEST.test(key);
            if (!isOverride) {
                keys.push(key);
            }
            for (let i = 0; i < parts.length; i++) {
                if (i === 0 && isOverride) {
                    const identifier = this.OVERRIDE_PROPERTY_PATTERN.exec(parts[i])[1];
                    if (!overrides[identifier]) {
                        overrides[identifier] = { keys: [], identifiers: [identifier], contents: Object.create(null) };
                    }
                    branch = overrides[identifier].contents;
                    overrides[identifier].keys.push(key.slice(parts[i].length + 1));
                }
                else if (i === parts.length - 1) {
                    branch[parts[i]] = data[key];
                }
                else {
                    if (!branch[parts[i]]) {
                        branch[parts[i]] = Object.create(null);
                    }
                    branch = branch[parts[i]];
                }
            }
            return result;
        }, Object.create(null));
        return { contents, keys, overrides: Object.values(overrides) };
    }
    toConfigurationChangeEvent(eventData) {
        return Object.freeze({
            affectsConfiguration: (section, scope) => {
                const { resource, overrideIdentifier } = this.parseConfigurationAccessOptions(scope);
                const sectionWithLanguage = overrideIdentifier ? `[${overrideIdentifier}].${section}` : section;
                return eventData.some(change => {
                    var _a, _b;
                    const matchesUri = !resource || !change.scope || (resource.toString() + '/').startsWith(change.scope.endsWith('/') ? change.scope : change.scope + '/');
                    const sliceIndex = overrideIdentifier ? 0 : ((_b = (_a = this.OVERRIDE_KEY_TEST.exec(change.preferenceName)) === null || _a === void 0 ? void 0 : _a[0].length) !== null && _b !== void 0 ? _b : 0);
                    const changedPreferenceName = sliceIndex ? change.preferenceName.slice(sliceIndex) : change.preferenceName;
                    return matchesUri && (sectionWithLanguage === changedPreferenceName
                        || sectionWithLanguage.startsWith(`${changedPreferenceName}.`)
                        || changedPreferenceName.startsWith(`${sectionWithLanguage}.`));
                });
            }
        });
    }
    parseConfigurationAccessOptions(scope) {
        if (!scope) {
            return {};
        }
        let overrideIdentifier = undefined;
        let resource;
        if ('uri' in scope || 'languageId' in scope) {
            resource = scope.uri;
        }
        else {
            resource = scope;
        }
        if ('languageId' in scope) {
            overrideIdentifier = scope.languageId;
        }
        return { resource, overrideIdentifier };
    }
}
exports.PreferenceRegistryExtImpl = PreferenceRegistryExtImpl;
//# sourceMappingURL=preference-registry.js.map